#!/usr/bin/python
# This file is part of the FireSync Project Generator.
# Copyright (C) 2008-2016 by LMI Technologies Inc.  All rights reserved.
# Distributed under the terms of the MIT License.
# Redistributed files must retain the above copyright notice.

import re
import os
import errno
from . import Project
from . import WriterBase
from . import Xml
from . import Utils

class Writer(WriterBase.WriterBase):

    #
    # Supported platforms and project types
    #
    _platforms = [
        'C64x',
    ]

    _projects = [
        'C/CPP',
    ]

    def __init__(self, platforms, projects):
        WriterBase.WriterBase.__init__(self, self._platforms, platforms, self._projects, projects)

    @property
    def Name(self):
        return 'Ccs6'

    def CfgFile(self, project):

        for src in self.SelectSources(project):

            if (os.path.splitext(src.Name)[-1]).lower() == '.cfg':
                return src.Name

        return None

    def HasSymbols(self, config):

        return True if 'Symbols' in config.Template.split('|') else False

    def IsDebug(self, config):

        # Using the configuration name instead of the template, which helps CCS
        # to match up configurations between projects and their dependencies.
        return True if config.Name == 'Debug' else False

    def DebugOrRelease(self, config):

        return 'Debug' if self.IsDebug(config) else 'Release'

    def ToolChainName(self, config):

        if config.Type == 'Executable':
            return 'exe.DebugToolchain' if self.IsDebug(config) else 'exe.ReleaseToolchain'
        else:
            return 'libraryDebugToolchain' if self.IsDebug(config) else 'library.ReleaseToolchain'

    def Id(self, name):

        hash = 0
        for char in name:
            hash = hash * 31 + ord(char)

        return str(hash).zfill(10)[0:10]

    def UniqueId(self, projectName, configName, id):

        return self.Id(projectName + configName + str(id))

    def CfgPath(self, project, config):
        herePath = os.path.dirname(os.path.realpath(__file__))
        return os.path.normpath(os.path.join(herePath, '..', '..', '..', 'kHardwareDm647', 'kHardwareDm647', 'Tcf'))

    def WriteLinkedResourcesFilters(self, project, sourceGroup, filterItem, filterName = ''):

        for childGroup in sourceGroup.Groups:
            linkItem = filterItem.AddElem('link')

            linkItem.AddElem('name', filterName + childGroup.Name)
            linkItem.AddElem('type', '2')
            linkItem.AddElem('locationURI', 'virtual:/virtual')

            self.WriteLinkedResourcesFilters(project, childGroup, filterItem, filterName + childGroup.Name + '/')

    def WriteLinkedResourcesFiles(self, project, sourceGroup, filterItem, filterName = ''):

        for childGroup in sourceGroup.Groups:
            self.WriteLinkedResourcesFiles(project, childGroup, filterItem, filterName + childGroup.Name + '/')

        for src in sourceGroup.Sources:
            linkItem = filterItem.AddElem('link')

            linkItem.AddElem('name', filterName + os.path.basename(src.Name))
            linkItem.AddElem('type', '1')
            linkItem.AddElem('location', os.path.join(os.path.dirname(project.FileName), src.Name))

    def WriteLinkedResources(self, project, sourceGroup, filterItem):
        self.WriteLinkedResourcesFilters(project, sourceGroup, filterItem)
        self.WriteLinkedResourcesFiles(project, sourceGroup, filterItem)

    def WriteDotProject(self, project, fileName):

        firstConfig = self.SelectConfigurations(project)[0]
        root = Xml.Element('projectDescription')
        root.AddElem('name', project.Name)
        root.AddElem('comment', '')

        dependencies = self.SelectDependencies(project)
        projectsItem = root.AddElem('projects')

        if firstConfig.Type != 'Executable' or len(dependencies) == 0:
            projectsItem.AddElem()
        else:
            for dependency in dependencies:
                projectsItem.AddElem('project', dependency.Name)

        buildSpecItem = root.AddElem('buildSpec')
        buildCommandItem = buildSpecItem.AddElem('buildCommand')
        buildCommandItem.AddElem('name', 'org.eclipse.cdt.managedbuilder.core.genmakebuilder')
        buildCommandItem.AddElem('triggers', 'clean,full,incremental,')
        argumentsItem = buildCommandItem.AddElem('arguments')
        argumentsItem.AddElem()
        buildCommandItem = buildSpecItem.AddElem('buildCommand')
        buildCommandItem.AddElem('name', 'org.eclipse.cdt.managedbuilder.core.ScannerConfigBuilder')
        buildCommandItem.AddElem('triggers', 'full,incremental,')
        argumentsItem = buildCommandItem.AddElem('arguments')
        argumentsItem.AddElem()

        naturesItem = root.AddElem('natures')
        naturesItem.AddElem('nature', 'org.eclipse.rtsc.xdctools.buildDefinitions.XDC.xdcNature')
        naturesItem.AddElem('nature', 'com.ti.ccstudio.core.ccsNature')
        naturesItem.AddElem('nature', 'org.eclipse.cdt.core.cnature')
        naturesItem.AddElem('nature', 'org.eclipse.cdt.managedbuilder.core.managedBuildNature')
        naturesItem.AddElem('nature', 'org.eclipse.cdt.core.ccnature')
        naturesItem.AddElem('nature', 'org.eclipse.cdt.managedbuilder.core.ScannerConfigNature')

        linkedResourcesItem = root.AddElem('linkedResources')
        self.WriteLinkedResources(project, project.SourceGroups, linkedResourcesItem)

        root.WriteDotProject(root, fileName)

    def WriteDotCProject(self, project, fileName):

        projectName = project.Name
        root = Xml.Element('cproject')
        root.AddAttr('storage_type_id', 'org.eclipse.cdt.core.XmlProjectDescriptionStorage')

        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('configRelations', '2')
        storageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.core.settings')

        for config in self.SelectConfigurations(project):
            debugOrRelease = self.DebugOrRelease(config)
            configName = config.Name
            cconfigurationItem = storageModuleItem.AddElem('cconfiguration')
            cconfigurationItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000.' + debugOrRelease + '.' + self.UniqueId(projectName, configName, 0))
            subStorageModule = cconfigurationItem.AddElem('storageModule')
            subStorageModule.AddAttr('buildSystemId', 'org.eclipse.cdt.managedbuilder.core.configurationDataProvider')
            subStorageModule.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000.%s.%s' % (debugOrRelease, self.UniqueId(projectName, configName, 0)))
            subStorageModule.AddAttr('moduleId', 'org.eclipse.cdt.core.settings')
            subStorageModule.AddAttr('name', configName)
            subStorageModule.AddElem('externalSettings')

            extensionsItem = subStorageModule.AddElem('extensions')

            extensionItem = extensionsItem.AddElem('extension')
            extensionItem.AddAttr('id', 'com.ti.ccstudio.errorparser.CoffErrorParser')
            extensionItem.AddAttr('point', 'org.eclipse.cdt.core.ErrorParser')

            extensionItem = extensionsItem.AddElem('extension')
            extensionItem.AddAttr('id', 'com.ti.ccstudio.errorparser.LinkErrorParser')
            extensionItem.AddAttr('point', 'org.eclipse.cdt.core.ErrorParser')

            extensionItem = extensionsItem.AddElem('extension')
            extensionItem.AddAttr('id', 'com.ti.ccstudio.errorparser.AsmErrorParser')
            extensionItem.AddAttr('point', 'org.eclipse.cdt.core.ErrorParser')

            extensionItem = extensionsItem.AddElem('extension')
            extensionItem.AddAttr('id', 'org.eclipse.rtsc.xdctools.parsers.ErrorParser')
            extensionItem.AddAttr('point', 'org.eclipse.cdt.core.ErrorParser')

            extensionItem = extensionsItem.AddElem('extension')
            extensionItem.AddAttr('id', 'com.ti.ccstudio.binaryparser.CoffParser')
            extensionItem.AddAttr('point', 'org.eclipse.cdt.core.BinaryParser')

            subStorageModule = cconfigurationItem.AddElem('storageModule')
            subStorageModule.AddAttr('moduleId', 'cdtBuildSystem')
            subStorageModule.AddAttr('version', '4.0.0')

            if config.Type == 'Executable':
                artifactName = os.path.normpath(os.path.join(os.path.dirname(project.FileName), config.BinDir, os.path.splitext(config.Target)[0]))
                artifactExtension = 'out'
            else:
                artifactName = os.path.normpath(os.path.join(os.path.dirname(project.FileName), config.LibDir, os.path.splitext(config.Target)[0]))
                artifactExtension = 'lib'

            configurationItem = subStorageModule.AddElem('configuration')
            configurationItem.AddAttr('artifactExtension', artifactExtension)
            configurationItem.AddAttr('artifactName',  artifactName)
            configurationItem.AddAttr('buildProperties', '')
            configurationItem.AddAttr('cleanCommand', '${CG_CLEAN_CMD}')
            configurationItem.AddAttr('description', '')
            configurationItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000.%s.%s' % (debugOrRelease, self.UniqueId(projectName, configName, 0)))
            configurationItem.AddAttr('name', configName)
            configurationItem.AddAttr('parent', 'com.ti.ccstudio.buildDefinitions.C6000.' + debugOrRelease)
            configurationItem.AddAttr('postbuildStep', '')
            configurationItem.AddAttr('prebuildStep', '')

            folderInfoElem = configurationItem.AddElem('folderInfo')
            folderInfoElem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000.%s.%s.' % (debugOrRelease, self.UniqueId(projectName, configName, 0)))
            folderInfoElem.AddAttr('name', '/')
            folderInfoElem.AddAttr('resourcePath', '')

            if config.Type == 'Executable':
                toolChainItem = folderInfoElem.AddElem('toolChain')
                toolChainItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.%s' % (self.ToolChainName(config), self.UniqueId(projectName, configName, 40)))
                toolChainItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s' % (self.ToolChainName(config)))
                toolChainItem.AddAttr('targetTool', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exe.linker%s.%s' % (debugOrRelease, self.UniqueId(projectName, configName, 50)))
            else:
                toolChainItem = folderInfoElem.AddElem('toolChain')
                toolChainItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.%s' % (self.ToolChainName(config), self.UniqueId(projectName, configName, 60)))
                toolChainItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s' % (self.ToolChainName(config)))
                toolChainItem.AddAttr('targetTool', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.library.librarian%s.%s' % (debugOrRelease, self.UniqueId(projectName, configName, 70)))

            optionItem = toolChainItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.core.OPT_TAGS.%s' % (self.UniqueId(projectName, configName, 80)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.core.OPT_TAGS')
            optionItem.AddAttr('valueType', 'stringList')

            appType = 'executable' if config.Type == 'Executable' else 'staticLibrary'

            if self.CfgFile(project) is not None:
                optionalOpts = [ 
                    'XDC_VERSION=3.31.0.24_core',
                    'RTSC_PRODUCTS=com.ti.rtsc.SYSBIOS:6.42.3.35;',
                    ]
            else:
                optionalOpts = []

            for listOptionValue in [ 
                    'DEVICE_CONFIGURATION_ID=TMS320C64XX.TMS320DM647',
                    'DEVICE_ENDIANNESS=little',
                    'OUTPUT_FORMAT=ELF',
                    'CCS_MBS_VERSION=5.5.0',
                    'RTSC_MBS_VERSION=2.2.0',] + optionalOpts + [
                    'EXPANDED_REPOS=',
                    'OUTPUT_TYPE=rtscApplication:' + appType,
                ]:
                    listOptionValueItem = optionItem.AddElem('listOptionValue')
                    listOptionValueItem.AddAttr('builtIn', 'false')
                    listOptionValueItem.AddAttr('value', listOptionValue)

            optionItem = toolChainItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.core.OPT_CODEGEN_VERSION.%s' % (self.UniqueId(projectName, configName, 90)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.core.OPT_CODEGEN_VERSION')
            optionItem.AddAttr('value', '7.4.13')
            optionItem.AddAttr('valueType', 'string')

            if config.Type == 'Executable':
                builderName = 'exe'
            else:
                builderName = 'library'

            targetPlatformItem = toolChainItem.AddElem('targetPlatform')
            targetPlatformItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.targetPlatform%s.%s' % (builderName, debugOrRelease, self.UniqueId(projectName, configName, 100)))
            targetPlatformItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.targetPlatform%s' % (builderName, debugOrRelease))

            builderItem = toolChainItem.AddElem('builder')
            builderItem.AddAttr('arguments', '-C %s -s -f %s-Ccs6.mk config=%s' % (os.path.dirname(project.FileName), project.Name, configName))
            builderItem.AddAttr('buildPath', '${BuildDirectory}')
            builderItem.AddAttr('command', '${CCS_UTILS_DIR}/bin/gmake')            
            builderItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.builder%s.%s' % (builderName, debugOrRelease, self.UniqueId(projectName, configName, 110)))
            builderItem.AddAttr('enableAutoBuild', 'false')
            builderItem.AddAttr('autoBuildTarget', 'all')
            builderItem.AddAttr('enableCleanBuild', 'true')
            builderItem.AddAttr('cleanBuildTarget', 'clean')
            builderItem.AddAttr('enabledIncrementalBuild', 'true')
            builderItem.AddAttr('incrementalBuildTarget', 'all')
            builderItem.AddAttr('keepEnvironmentInBuildfile', 'false')
            builderItem.AddAttr('managedBuildOn', 'false')
            builderItem.AddAttr('stopOnErr', 'true')
            builderItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.builder%s' % (builderName, debugOrRelease))

            toolItem = toolChainItem.AddElem('tool')
            toolItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.compiler%s.%s' % (builderName, debugOrRelease, self.UniqueId(projectName, configName, 120)))
            toolItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.%s.compiler%s' % (builderName, debugOrRelease))

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DEFINE.%s' % (self.UniqueId(projectName, configName, 130)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DEFINE')
            optionItem.AddAttr('valueType', 'definedSymbols')

            for listOptionValue in config.Defines:
                    listOptionValueItem = optionItem.AddElem('listOptionValue')
                    listOptionValueItem.AddAttr('builtIn', 'false')
                    listOptionValueItem.AddAttr('value', listOptionValue)

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.SILICON_VERSION.%s' % (self.UniqueId(projectName, configName, 140)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.SILICON_VERSION')
            optionItem.AddAttr('value', '64+')
            optionItem.AddAttr('valueType', 'string')

            if self.HasSymbols(config):
                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DEBUGGING_MODEL.%s' % (self.UniqueId(projectName, configName, 150)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DEBUGGING_MODEL')
                optionItem.AddAttr('value', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DEBUGGING_MODEL.SYMDEBUG__DWARF')
                optionItem.AddAttr('valueType', 'enumerated')

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DIAG_WARNING.%s' % (self.UniqueId(projectName, configName, 160)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DIAG_WARNING')
            optionItem.AddAttr('valueType', 'stringList')

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DISPLAY_ERROR_NUMBER.%s' % (self.UniqueId(projectName, configName, 170)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.DISPLAY_ERROR_NUMBER')
            optionItem.AddAttr('value', 'true')
            optionItem.AddAttr('valueType', 'boolean')

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.INCLUDE_PATH.%s' % (self.UniqueId(projectName, configName, 180)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.INCLUDE_PATH')
            optionItem.AddAttr('valueType', 'includePath')

            for listOptionValue in [
                '${CG_TOOL_ROOT}/include',
                ] + [ os.path.normpath(os.path.join(os.path.dirname(project.FileName), dir)) for dir in config.IncludeDirs ]:
                    listOptionValueItem = optionItem.AddElem('listOptionValue')
                    listOptionValueItem.AddAttr('builtIn', 'false')
                    listOptionValueItem.AddAttr('value', '&quot;' + listOptionValue + '&quot;')

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.ABI.%s' % (self.UniqueId(projectName, configName, 190)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.ABI')
            optionItem.AddAttr('value', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.ABI.coffabi')
            optionItem.AddAttr('valueType', 'enumerated')

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.MEM_MODEL__DATA.%s' % (self.UniqueId(projectName, configName, 200)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.MEM_MODEL__DATA')
            optionItem.AddAttr('value', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.MEM_MODEL__DATA.far')
            optionItem.AddAttr('valueType', 'enumerated')

            optionItem = toolItem.AddElem('option')
            optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.INTERRUPT_THRESHOLD.%s' % (self.UniqueId(projectName, configName, 210)))
            optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compilerID.INTERRUPT_THRESHOLD')
            optionItem.AddAttr('value', '--interrupt_threshold=1')
            optionItem.AddAttr('valueType', 'string')
            
            for inputType in [ 
                    [ 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__C_SRCS.%s' % (self.UniqueId(projectName, configName, 220)), 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__C_SRCS' ],
                    [ 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__CPP_SRCS.%s' % (self.UniqueId(projectName, configName, 230)), 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__CPP_SRCS' ],
                    [ 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__ASM_SRCS.%s' % (self.UniqueId(projectName, configName, 240)), 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__ASM_SRCS' ],
                    [ 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__ASM2_SRCS.%s' % (self.UniqueId(projectName, configName, 250)), 'com.ti.ccstudio.buildDefinitions.C6000_7.4.compiler.inputType__ASM2_SRCS' ],
                ]:
                optionItem = toolItem.AddElem('inputType')
                optionItem.AddAttr('id', inputType[0])
                optionItem.AddAttr('superClass', inputType[1])

            if config.Type == 'Library':
                toolItem = toolChainItem.AddElem('tool')
                toolItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.library.librarian%s.%s' % (debugOrRelease, self.UniqueId(projectName, configName, 70)))
                toolItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.library.librarian%s' % (debugOrRelease))

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.archiverID.OUTPUT_FILE.%s' % (self.UniqueId(projectName, configName, 270)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.archiverID.OUTPUT_FILE')
                optionItem.AddAttr('value', '&quot;%s&quot;' % (os.path.normpath(os.path.join(os.path.dirname(project.FileName), config.LibDir, config.Target))))
                optionItem.AddAttr('valueType', 'string')

            elif config.Type == 'Executable':
                toolItem = toolChainItem.AddElem('tool')
                toolItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exe.linker%s.%s' % (debugOrRelease, self.UniqueId(projectName, configName, 50)))
                toolItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exe.linker%s' % (debugOrRelease))

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.STACK_SIZE.%s' % (self.UniqueId(projectName, configName, 290)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.STACK_SIZE')
                optionItem.AddAttr('value', '0x800')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.HEAP_SIZE.%s' % (self.UniqueId(projectName, configName, 300)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.HEAP_SIZE')
                optionItem.AddAttr('value', '0x800')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.OUTPUT_FILE.%s' % (self.UniqueId(projectName, configName, 310)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.OUTPUT_FILE')
                optionItem.AddAttr('value', '&quot;%s.out&quot;' % (os.path.normpath(os.path.join(os.path.dirname(project.FileName), config.BinDir, os.path.splitext(config.Target)[0]))))
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.MAP_FILE.%s' % (self.UniqueId(projectName, configName, 320)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.MAP_FILE')
                optionItem.AddAttr('value', '&quot;${ProjName}.map&quot;')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.XML_LINK_INFO.%s' % (self.UniqueId(projectName, configName, 330)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.XML_LINK_INFO')
                optionItem.AddAttr('value', '&quot;${ProjName}_linkInfo.xml&quot;')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.DISPLAY_ERROR_NUMBER.%s' % (self.UniqueId(projectName, configName, 340)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.DISPLAY_ERROR_NUMBER')
                optionItem.AddAttr('value', 'true')
                optionItem.AddAttr('valueType', 'boolean')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.SEARCH_PATH.%s' % (self.UniqueId(projectName, configName, 350)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.SEARCH_PATH')
                optionItem.AddAttr('valueType', 'libPaths')

                for listOptionValue in [
                    '${CG_TOOL_ROOT}/lib',
                    '${CG_TOOL_ROOT}/include',
                    ] + [ os.path.normpath(os.path.join(os.path.dirname(project.FileName), dir)) for dir in config.LibraryDirs ] + \
                    [self.CfgPath(project, config)]:
                        listOptionValueItem = optionItem.AddElem('listOptionValue')
                        listOptionValueItem.AddAttr('builtIn', 'false')
                        listOptionValueItem.AddAttr('value', '&quot;' + listOptionValue + '&quot;')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.LIBRARY.%s' % (self.UniqueId(projectName, configName, 360)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.linkerID.LIBRARY')
                optionItem.AddAttr('valueType', 'libs')

                for listOptionValue in config.Libs:
                        listOptionValueItem = optionItem.AddElem('listOptionValue')
                        listOptionValueItem.AddAttr('builtIn', 'false')
                        listOptionValueItem.AddAttr('value', '&quot;' + listOptionValue + '&quot;')

                optionItem = toolItem.AddElem('inputType')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exeLinker.inputType__CMD_SRCS.%s' % (self.UniqueId(projectName, configName, 370)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exeLinker.inputType__CMD_SRCS')

                optionItem = toolItem.AddElem('inputType')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exeLinker.inputType__CMD2_SRCS.%s' % (self.UniqueId(projectName, configName, 380)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exeLinker.inputType__CMD2_SRCS')
                
                optionItem = toolItem.AddElem('inputType')
                optionItem.AddAttr('id', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exeLinker.inputType__GEN_CMDS.%s' % (self.UniqueId(projectName, configName, 390)))
                optionItem.AddAttr('superClass', 'com.ti.ccstudio.buildDefinitions.C6000_7.4.exeLinker.inputType__GEN_CMDS')

            if self.CfgFile(project) is not None:
                toolItem = toolChainItem.AddElem('tool')
                toolItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.%s' % (self.UniqueId(projectName, configName, 400)))
                toolItem.AddAttr('name', 'XDCtools')
                toolItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.CODEGEN_TOOL_DIR.%s' % (self.UniqueId(projectName, configName, 401)))
                optionItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.CODEGEN_TOOL_DIR')
                optionItem.AddAttr('value', '&quot;${CG_TOOL_ROOT}&quot;')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.TARGET.%s' % (self.UniqueId(projectName, configName, 402)))
                optionItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.TARGET')
                optionItem.AddAttr('value', 'ti.targets.C64P')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.PLATFORM.%s' % (self.UniqueId(projectName, configName, 403)))
                optionItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.PLATFORM')
                optionItem.AddAttr('value', 'kHwDm647')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.PLATFORM_RAW.%s' % (self.UniqueId(projectName, configName, 404)))
                optionItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.PLATFORM_RAW')
                optionItem.AddAttr('value', 'kHwDm647')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.BUILD_PROFILE.%s' % (self.UniqueId(projectName, configName, 405)))
                optionItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.BUILD_PROFILE')
                optionItem.AddAttr('value', 'release')
                optionItem.AddAttr('valueType', 'string')

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.XDC_PATH.%s' % (self.UniqueId(projectName, configName, 406)))
                optionItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.XDC_PATH')
                optionItem.AddAttr('valueType', 'stringList')

                for listOptionValue in [ 
                        '${BIOS_CG_ROOT}/packages',
                        '${TARGET_CONTENT_BASE}',
                        self.CfgPath(project, config),
                    ]:
                        listOptionValueItem = optionItem.AddElem('listOptionValue')
                        listOptionValueItem.AddAttr('builtIn', 'false')
                        listOptionValueItem.AddAttr('value', listOptionValue)

                optionItem = toolItem.AddElem('option')
                optionItem.AddAttr('id', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.COMPILE_OPTIONS.%s' % (self.UniqueId(projectName, configName, 407)))
                optionItem.AddAttr('superClass', 'com.ti.rtsc.buildDefinitions.XDC_3.16.tool.COMPILE_OPTIONS')
                optionItem.AddAttr('value', '&quot;-g --optimize_with_debug&quot;')
                optionItem.AddAttr('valueType', 'string')

            dependencies = self.SelectDependencies(project)
            if config.Type != 'Executable' or len(dependencies) == 0:
                subStorageModule = cconfigurationItem.AddElem('storageModule')
                subStorageModule.AddAttr('moduleId', 'org.eclipse.cdt.core.externalSettings')

            else:
                subStorageModule = cconfigurationItem.AddElem('storageModule')
                subStorageModule.AddAttr('moduleId', 'org.eclipse.cdt.core.externalSettings')

                for dependency in dependencies:
                    externalSettingsItem = subStorageModule.AddElem('externalSettings')
                    externalSettingsItem.AddAttr('containerId', dependency.Name + ';com.ti.ccstudio.buildDefinitions.C6000.' + debugOrRelease + '.' + self.UniqueId(dependency.Name, debugOrRelease, 0))
                    externalSettingsItem.AddAttr('factoryId', 'org.eclipse.cdt.core.cfg.export.settings.sipplier')

        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'org.eclipse.cdt.core.LanguageSettingsProviders')

        storageModuleItem = root.AddElem('storageModule')
        storageModuleItem.AddAttr('moduleId', 'cdtBuildSystem')
        storageModuleItem.AddAttr('version', '4.0.0')

        projectItem = storageModuleItem.AddElem('project')
        projectItem.AddAttr('id', '%s.com.ti.ccstudio.buildDefinitions.C6000.ProjectType.%s' % (project.Name, self.UniqueId(projectName, '', '440')))
        projectItem.AddAttr('name', 'C6000')
        projectItem.AddAttr('projectType', 'com.ti.ccstudio.buildDefinitions.C6000.ProjectType')

        scannerConfigurationItem = root.AddElem('storageModule')
        scannerConfigurationItem.AddAttr('moduleId', 'scannerConfiguration')

        langMappingItem = root.AddElem('storageModule')
        langMappingItem.AddAttr('moduleId', 'org.eclipse.cdt.core.language.mapping')
        projectMappingsItem = langMappingItem.AddElem('project-mappings')

        for itemContent in [
                ['', 'org.eclipse.cdt.core.asmSource', 'com.ti.ccstudio.core.TIASMLanguage', ], 
                ['', 'org.eclipse.cdt.core.cHeader', 'com.ti.ccstudio.core.TIGCCLanguage', ], 
                ['', 'org.eclipse.cdt.core.cSource', 'com.ti.ccstudio.core.TIGCCLanguage', ], 
                ['', 'org.eclipse.cdt.core.cxxHeader', 'com.ti.ccstudio.core.TIGPPLanguage', ], 
                ['', 'org.eclipse.cdt.core.cxxSource', 'com.ti.ccstudio.core.TIGPPLanguage', ], 
            ]:
                contentTypeItem = projectMappingsItem.AddElem('content-type-mapping')
                contentTypeItem.AddAttr('configuration', itemContent[0])
                contentTypeItem.AddAttr('content-type', itemContent[1])
                contentTypeItem.AddAttr('language', itemContent[2])

        root.WriteDotCProject(root, fileName, False)

    def WriteDotCcsProject(self, project, fileName):

        root = Xml.Element('projectOptions')

        deviceVariantItem = root.AddElem('deviceVariant')
        deviceVariantItem.AddAttr('value', 'TMS320C64XX.TMS320DM647')

        deviceFamilyItem = root.AddElem('deviceFamily')
        deviceFamilyItem.AddAttr('value', 'C6000')

        deviceEndiannessItem = root.AddElem('deviceEndianness')
        deviceEndiannessItem.AddAttr('value', 'little')

        codegenToolVersionItem = root.AddElem('codegenToolVersion')
        codegenToolVersionItem.AddAttr('value', '7.4.13')

        isElfFormatItem = root.AddElem('isElfFormat')
        isElfFormatItem.AddAttr('value', 'true')

        connectionItem = root.AddElem('connection')
        connectionItem.AddAttr('value', 'common/targetdb/connections/SD510USB_Connection.xml')

        createSlaveProjectsItem = root.AddElem('createSlaveProjects')
        createSlaveProjectsItem.AddAttr('value', '')

        templatePropertiesItem = root.AddElem('templateProperties')
        templatePropertiesItem.AddAttr('value', 'id=com.ti.rtsc.SYSBIOS.example_34,type=rtsc,products=com.ti.rtsc.SYSBIOS,buildProfile=release,isHybrid=true,configuroOptions=--compileOptions &quot;-g --optimize_with_debug&quot;,')

        isTargetManualItem = root.AddElem('isTargetManual')
        isTargetManualItem.AddAttr('value', 'false')

        root.WriteDotCcsProject(root, fileName)

    def WriteDotClean(self, project, dirName):

        for config in self.SelectConfigurations(project):
            try:
                os.makedirs(os.path.join(dirName, config.Name))
            except OSError as exc:
                if exc.errno != errno.EEXIST:
                    raise

            with Utils.Output(os.path.join(dirName, config.Name, '.clean')) as output:
                output.Format('# CCS6.1 requires a non-empty directory to exist in order for "Clean Project" to work.\n')

    def WriteProject(self, project):

        solution = project.Solution

        if solution is not None:
            dirName = os.path.join(os.path.dirname(solution.FileName), '.' + solution.Name + '.ccs6', project.Name)
        else:
            dirName = os.path.join(os.path.dirname(project.FileName), '.' + project.Name + '.ccs6')

        try:
            os.makedirs(dirName)
        except OSError as exc:
            if exc.errno != errno.EEXIST:
                raise

        self.WriteDotProject(project, os.path.join(dirName, '.project'))
        self.WriteDotCProject(project, os.path.join(dirName, '.cproject'))
        self.WriteDotCcsProject(project, os.path.join(dirName, '.ccsproject'))
        self.WriteDotClean(project, dirName)

    def WriteSolution(self, solution):
        # not implemented
        return
