#ifndef LMITECH_KVISION_VS_UTILITIES_H_INCLUDED
#define LMITECH_KVISION_VS_UTILITIES_H_INCLUDED

#include <kVision/Common/kVision.h>
#include <kVision/Vs/kVsModelDef.h>
#include <kFireSync/Utils/kSensorIdentity.h>
#include <kApi/Data/kString.h>
#include <kApi/Data/kArrayList.h>
#include <kVision/Common/kPolynomial.h>
#include <kVision/Common/kNullMath.h>
#include <math.h>

/**
* @brief Formats laser driver id into human readable form. As the format may vary in the future, the formatted
* id should _never_ be used for programmatic matching of physical driver id with respect to the calibration
* records. Use unformatted id for that purpose. 
*/
kVsFx(kStatus) kVsLaserDriverId_Format(kSize id, kChar* label, kSize capacity);

typedef kStatus(*PolyModelFit)(kPolynomial* poly, const k64f* x, const k64f* y, kSize count,  kAlloc alloc);
typedef k64f(*FunctionOfPoly)(kPolynomial poly);
typedef k64f(*FunctionOfPoly1)(kPolynomial poly, k64f param);

kVsFx(kStatus) kVsLine2d_FitPoint64f(const kPoint64f* points, k32u count, k64f* slope, k64f* intercept);
kVsFx(kStatus) kVsLine2d_Fit64f(const k64f* x, const k64f* y, k32u count, k64f* slope, k64f* intercept);

kVsFx(kStatus) kVsParse32u(const kChar* text, k32u* value);
kVsFx(kStatus) kVsParse64f(const kChar* text, k64f* value);

kVsFx(kStatus) kVsImage_Crop(kImage src, kRect32s roi, kImage* dst, kAlloc alloc);
kVsFx(kStatus) kVsArray2_Crop(kArray2 src, kRect32s roi, kArray2* dst, kAlloc alloc);

// drawing functions
kVsFx(kStatus) kVsImage_DrawCircle(kImage image, k64f xc, k64f yc, k64f radius, k8u value);
kVsFx(kStatus) kVsImage_DrawRectangle(kImage image, k32s x, k32s y, k32s width, k32s height, k8u value);
kVsFx(kStatus) kVsImage_FillCheckerPattern(kImage image, k32u rectSize, kPoint32s* inputReference, k32u* gridWidth, k32u* gridHeight, kPoint32s* outReference);
/**
* @brief Draws an ellipse on an image
* xc and yc are the center coordinates
* rx and ry are half the length of axes of the ellipse in each direction
*/
kVsFx(kStatus) kVsImage_DrawEllipse(kImage image, k64f xc, k64f yc, k64f rx, k64f ry, k8u value);

kVsFx(k64f) kVsMath_Percentile64f(const k64f* a, kSize count, k64f percentile);
kVsFx(k64f) kVsRand_Normal(k64f mean, k64f std);
kVsFx(k64f) kVsRand_Uniform();

/**
* @brief Constructs a linear spline of uniformly spaced knots determined by 'y' values over the range spanning
* 'xMin'to 'xMax'. 'Count' must be greater or equal to one
*/
kVsFx(kStatus) kVsSpline_ConstructUniform(kSpline* spline, k64f xMin, k64f xMax, const k64f* y, kSize count, kAlloc alloc);
kVsFx(kStatus) kVsConstraint_ConstructSteps(kArray1* constraint, kArrayList knots, k64f constraintMin, k64f constraintMax, kArray1 expectedValues, kAlloc alloc);
kVsFx(kStatus) kVsConstraint_TestSteps(kArray1 constraint, kArray1 error, kBool* passed);

//
//kVsFx(k64f) kVsRand_Normal(k64f mean, k64f std);

/**
 * @brief Returns kTRUE if string matches the pattern, where pattern may contain the two
 * special characters: '*' (match 0 or more arbitrary characters) and '?' (match exactly
 * one arbitrary character.
 *
 *  A ? sign matches any character, except an empty string.
 *  A * sign matches any string inclusive an empty string.
 * Based on code by Martin Richter: https://www.codeproject.com/Articles/188256/A-Simple-Wildcard-Matching-Function
 *
 * @param string The input string. Must be null-terminated
 * @param pattern The (possibly) wildcarded pattern
 * @return Returns kTRUE on a match, kFALSE otherwise. An empty string matches an empty
 * pattern. The pattern "*" matches anything, including an empty string.
 * 
 * Examples:
 *         kExpectTrue(kVsStrWildcardMatch("foo","foo"));
 *         kExpectTrue(kVsStrWildcardMatch("foo", "*o"));
 *         kExpectTrue(kVsStrWildcardMatch("foo", "f*"));
 *         kExpectTrue(kVsStrWildcardMatch("foo", "*"));
 *         kExpectTrue(kVsStrWildcardMatch("", ""));
 *         kExpectTrue(kVsStrWildcardMatch("1234567890", "1*0"));
 *         kExpectTrue(kVsStrWildcardMatch("M3_CFG623_6.2.4.0.kpl", "M3_CFG623_*.*.*.*.kpl"));
 *         kExpectTrue(kVsStrWildcardMatch("M3_CFG623_6.12.4.39.kpl", "M3_CFG623_*.*.*.*.kpl"));
 *         kExpectTrue(kVsStrWildcardMatch("1.2.3.4.5.6.7.8.9.0", "*.*.*.*.*.*.*.*.*.*"));
 *         kExpectTrue(kVsStrWildcardMatch("foo", "f?o"));
 *         kExpectTrue(kVsStrWildcardMatch("foo", "???"));
 *         kExpectTrue(kVsStrWildcardMatch("foo", "?*"));
 *         kExpectTrue(kVsStrWildcardMatch("foo", "*?"));
 *         kExpectTrue(kVsStrWildcardMatch("M3_CFG623_6.2.14.0.kpl", "M?_CFG???_*.*.*.*.kpl"));
 *         
 *         kExpectFalse(kVsStrWildcardMatch("", "*a"));
 *         kExpectFalse(kVsStrWildcardMatch("bbbbbbbbb", "*a"));
 *         kExpectFalse(kVsStrWildcardMatch("1.2.3.4.5.6.7.8.9", "*.*.*.*.*.*.*.*.*.*"));  //one too few '.'s
 *         kExpectFalse(kVsStrWildcardMatch("", "?"));
 *         kExpectFalse(kVsStrWildcardMatch("quux", "???"));
 * 
 **/
kVsFx(kBool) kVsStrWildcardMatch(const kChar* string, const kChar* pattern);


/**
 * @brief Filters a kArrayList<kString> with a wildcard filter.
 * This method creates a new kArrayList<kString> resultStrings containing duplicates of any entries in
 * srcStrings that match the pattern. Pattern matching is performed using kVsStrWildcardMatch.
 * @param srcStrings A kArrayList<kString> of strings to be matched
 * @param pattern A potentially wildcarded pattern for filtering the strings. See kVsStrWildcardMatch for pattern
 * specification
 * @param resultStrings A pointer to a kArrayList. This will be filled with a newly allocated kArrayList<kString>
 * containing duplicates of any strings in srcStrings that match the pattern.
 * @alloc a kAlloc allocator for creating resultStrings.
**/
kVsFx(kStatus) kVsStrWildcardFilter(kArrayList srcStrings, const kChar* pattern, kArrayList* resultStrings, kAlloc alloc);

/**
 * Log to a file
 * On PC call kVsLog_FileStart to redirect kLogf() to a file.
 *
 * @return                   Operation status.
 */

//#define kVS_LOG_FILE_ENABLED // define this to log to a file on PC

kInlineFx(kStatus) kVsLog_FileStart(); // starts logging int log.txt file

/**
 * PROFILER simple
 * Reports times spent between kVsProfile_Start_ and kVsProfile_End_
 * Employs kLogf() to report the times
 * Note: Will leak mem if you bail between START and END, use kTry kFinally
 *
 * Example use:
 *    kVsProfile_Start_(MyNewAlg);
 *    ... some code
 *    kVsProfile_End_(MyNewAlg);
 *
 *
 * @param   name             Any name you want to give to a profiled block, times will be reported with this name
 * @return                   Operation status.
 */

//#define kVS_PROFILE_ENABLED // define this to start profiling into kLogf()

#define kVsProfile_Start_( name ) kxVsProfile_Start_( name )  

#define kVsProfile_End_( name ) kxVsProfile_End_( name ) 

 //////////////////////////////////////////////////////////////////////////
 //
 //////////////////////////////////////////////////////////////////////////

#include <kVision/Vs/kVsUtilities.x.h>

#endif /* #ifndef LMITECH_KVISION_VS_UTILITIES_H_INCLUDED */
