/**
* @file     kVsTrackingWindow.h
* @brief    Declares the kVsTrackingWindow class
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/

#ifndef K_VISION_TRACKING_WINDOW_H
#define K_VISION_TRACKING_WINDOW_H

#include <kVision/Common/kVision.h>

/**
* @class       kVsTrackingWindow
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       
* 
* This is the class that will execute the tracking window algorithm, the algorithm take all parameter and 
* generate the new tracking window top position within the class structure. if the new window top y position
* is -1, this means a full search is needed. Parameter for this algorithm needs to be set up before executing the
* algorithm:
*   x: X position of tracking window
*   yMax: Maximum y position of tracking window
*   yMin: Minimum y position of tracking window
*   yGranularity: Y granularity shift of tracking window
*   minSpotCount: Minimum spot count threshold
*   height: Height of the tracking window
*/

typedef kObject kVsTrackingWindow;



/**
* Constructs a kVsTrackingWindow object.
*
* @public                   @memberof kVsTrackingWindow
* @param    trcWindow       Destination for the constructed object handle.
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_Construct(kVsTrackingWindow* trackingWindow, kAlloc allocator);

/**
* Executing the algorithm of simulating tracking window behavior, this algorithm only generates the row value for the next tracking window.
* In cases where the row value or total spot number does not meet the threshold value, this algorithm produces the row value as -1
* as a flag that full search is needed. The algorithm input is the average centroid value of image statistics and total spot count,
* output is the next tracking window position (-1 as a signal for full search)
*
* @public                   @memberof kVsTrackginWindow
* @param    trcWindow       TrackingWindow object
* @param    avgCentre       Average centroid value
* @param    totalSpot       Total spot number
* @param    y               Pointer to output y value
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_Execute(kVsTrackingWindow trackingWindow, kSize avgCentre, kSSize* y, kSize spotCount);

/**
* Sets the x size of tracking Window
*
* @public                   @memeberof kVsTrackingWindow
* @param    trcWindow       TrackingWindow object
* @param    x               X position of tracking window
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_SetX(kVsTrackingWindow trackingWindow, kSize x);

/*
* Gets the x size of tracking Window
*
* @public                   @memberof kVsTrackingWindow
* @param    trcWindow       TrakcingWindow object
* @return                   X position of trackingWindow  
*/
kVsFx(kSize) kVsTrackingWindow_X(kVsTrackingWindow trackingWindow);

/**
* Sets the maximum y position of tracking Window
*
* @public                   @memeberof kVsTrackingWindow
* @param    trcWindow       TrackingWindow object
* @param    yMax            Maximum y position of tracking window
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_SetYMax(kVsTrackingWindow trackingWindow, kSSize yMax);

/*
* Gets the maximum y position of tracking Window
*
* @public                   @memberof kVsTrackingWindow
* @param    trcWindow       TrackingWindow object
* @return                   Maximum y position of trackingWindow  
*/
kVsFx(kSSize) kVsTrackingWindow_YMax(kVsTrackingWindow trackingWindow);

/**
* Sets the minimum y position of tracking Window
*
* @public                   @memeberof kVsTrackingWindow
* @param    trcWindow       TrackingWindow object
* @param    yMin            Minimum y position of tracking window
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_SetYMin(kVsTrackingWindow trackingWindow, kSSize yMin);

/*
* Gets the minimum y position of tracking Window
*
* @public                   @memberof kVsTrackingWindow
* @param    trcWindow       TrakcingWindow object
* @return                   Minimum y position of trackingWindow  
*/
kVsFx(kSSize) kVsTrackingWindow_YMin(kVsTrackingWindow trackingWindow);

/**
* Sets the y granularity of tracking Window, this is the y granularity shift value
* (divided by 2 to the power of this value)
*
* @public                   @memeberof kVsTrackingWindow
* @param    trcWindow       TrackingWindow object
* @param    yGran           Y granularity profile of tracking window
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_SetYGranularity(kVsTrackingWindow trackingWindow, kSize yGranularity);

/*
* Gets the y granularity profile of tracking Window
*
* @public                   @memberof kVsTrackingWindow
* @param    trcWindow       TrakcingWindow object
* @return                   Y granularity profile of trackingWindow  
*/
kVsFx(kSize) kVsTrackingWindow_YGranularity(kVsTrackingWindow trackingWindow);

/**
* Sets the minimum spot count threshold of tracking Window
*
* @public                   @memeberof kVsTrackingWindow
* @param    trcWindow       TrackingWindow object
* @param    minSpot         Minimum spot threshold of tracking window
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_SetMinSpotCount(kVsTrackingWindow trackingWindow, kSize minSpotCount);

/*
* Gets the minimum spot count threshold of tracking Window
*
* @public                   @memberof kVsTrackingWindow
* @param    trcWindow       TrakcingWindow object
* @return                   Minimum spot threshold of trackingWindow  
*/
kVsFx(kSize) kVsTrackingWindow_MinSpotCount(kVsTrackingWindow trackingWindow);

/**
* Sets the height of tracking Window
*
* @public                   @memeberof kVsTrackingWindow
* @param    trcWindow       TrackingWindow object
* @param    height          Height of tracking window
* @return                   Operation status
*/
kVsFx(kStatus) kVsTrackingWindow_SetHeight(kVsTrackingWindow trackingWindow, kSize height);

/*
* Gets the height of tracking Window
*
* @public                   @memberof kVsTrackingWindow
* @param    trcWindow       TrakcingWindow object
* @return                   Height of trackingWindow  
*/
kVsFx(kSize) kVsTrackingWindow_Height(kVsTrackingWindow trackingWindow);

;

#include <kVision/Vs/kVsTrackingWindow.x.h>

#endif /*K_VISION_TRACKING_WINDOW_H*/