/**
* @file kVsSensorAdjust.h
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
*/

#ifndef K_VISION_VS_SENSOR_ADJUST_H
#define K_VISION_VS_SENSOR_ADJUST_H

#include <kVision/Common/kVision.h>
#include <kFireSync/Client/kNode.h>
#include <kApi/Data/kXml.h>

#define kVS_SENSOR_ADJUST_FILE_NAME "Adjust.xml"

/**
* @class       kVsSensorAdjust
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       A class representing the Adjust.xml file.
*/
typedef kObject kVsSensorAdjust;

/**
 * Constructs a kVsSensorAdjust object.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  allocator           Memory allocator.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_Construct(kVsSensorAdjust* adjust, kAlloc allocator);

/**
 * Load adjust xml data.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  node                Node to load file from. Use kNULL to load from disk.
 * @param  fileName            Filename.
 * @param  allocator           Memory allocator.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_Load(kVsSensorAdjust* adjust, kNode node, const kChar* fileName, kAlloc allocator);

/**
 * Save adjust xml data.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  node                Node to save file to. Use kNULL to save to disk.
 * @param  fileName            Filename.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_Save(kVsSensorAdjust adjust, kNode node, const kChar* filename);

/**
 * Sets camera count.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  count               Camera count.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_SetCameraCount(kVsSensorAdjust adjust, kSize count);

/**
 * Gets camera count.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @return                     Camera count.
 */
kVsFx(kSize) kVsSensorAdjust_CameraCount(kVsSensorAdjust adjust);

/**
 * Specifies if camera black level is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Camera black level is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasBlackLevel(kVsSensorAdjust adjust, kSize index);

/**
 * Gets camera black level.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Camera black level.
 */
kVsFx(k32s) kVsSensorAdjust_CameraBlackLevel(kVsSensorAdjust adjust, kSize index);

/**
 * Sets camera black level.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  blackLevel          Camera black level.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetBlackLevel(kVsSensorAdjust adjust, kSize index, k32s blackLevel);

/**
 * Specifies if digital gain is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Digital gain is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasDigitalGain(kVsSensorAdjust adjust, kSize index);

/**
 * Gets digital gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Digital gain.
 */
kVsFx(k64f) kVsSensorAdjust_CameraDigitalGain(kVsSensorAdjust adjust, kSize index);

/**
 * Sets digital gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  gain                Digital gain.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetDigitalGain(kVsSensorAdjust adjust, kSize index, k64f gain);

/**
 * Specifies if analog gain is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Analog gain is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasAnalogGain(kVsSensorAdjust adjust, kSize index);

/**
 * Gets analog gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Analog gain.
 */
kVsFx(k64f) kVsSensorAdjust_CameraAnalogGain(kVsSensorAdjust adjust, kSize index);

/**
 * Sets analog gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  gain                Analog gain.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetAnalogGain(kVsSensorAdjust adjust, kSize index, k64f gain);

/**
 * Specifies if exposure scale is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Exposure scale is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasExposureScale(kVsSensorAdjust adjust, kSize index);

/**
 * Gets exposure scale.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Exposure scale.
 */
kVsFx(k64f) kVsSensorAdjust_CameraExposureScale(kVsSensorAdjust adjust, kSize index);

/**
 * Sets exposure scale.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  scale               Exposure scale.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetExposureScale(kVsSensorAdjust adjust, kSize index, k64f scale);

/**
 * Specifies if red gain is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Red gain is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasRedGain(kVsSensorAdjust adjust, kSize index);

/**
 * Gets red gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Red gain.
 */
kVsFx(k64f) kVsSensorAdjust_CameraRedGain(kVsSensorAdjust adjust, kSize index);

/**
 * Sets red gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  redGain             Red gain.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetRedGain(kVsSensorAdjust adjust, kSize index, k64f redGain);

/**
 * Specifies if green0 gain is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Green0 gain is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasGreen0Gain(kVsSensorAdjust adjust, kSize index);

/**
 * Gets green0 gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Green0 gain.
 */
kVsFx(k64f) kVsSensorAdjust_CameraGreen0Gain(kVsSensorAdjust adjust, kSize index);

/**
 * Sets green0 gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  green0Gain          Green0 gain.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetGreen0Gain(kVsSensorAdjust adjust, kSize index, k64f green0Gain);

/**
 * Specifies if green1 gain is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Green1 gain is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasGreen1Gain(kVsSensorAdjust adjust, kSize index);

/**
 * Gets green1 gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Green1 gain.
 */
kVsFx(k64f) kVsSensorAdjust_CameraGreen1Gain(kVsSensorAdjust adjust, kSize index);

/**
 * Sets green1 gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  green1Gain          Green1 gain.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetGreen1Gain(kVsSensorAdjust adjust, kSize index, k64f green1Gain);

/**
 * Specifies if blue gain is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Blue gain is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasBlueGain(kVsSensorAdjust adjust, kSize index);

/**
 * Gets blue gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Blue gain.
 */
kVsFx(k64f) kVsSensorAdjust_CameraBlueGain(kVsSensorAdjust adjust, kSize index);

/**
 * Sets blue gain.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  blueGain            Blue gain.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetBlueGain(kVsSensorAdjust adjust, kSize index, k64f blueGain);

/**
 * Specifies if tracheid threshold is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Tracheid threshold is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasTracheidThreshold(kVsSensorAdjust adjust, kSize index);

/**
 * Gets tracheid threshold.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Tracheid threshold.
 */
kVsFx(k32u) kVsSensorAdjust_CameraTracheidThreshold(kVsSensorAdjust adjust, kSize index);

/**
 * Sets tracheid threshold.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  tracheidThreshold   Tracheid threshold.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetTracheidThreshold(kVsSensorAdjust adjust, kSize index, k32u tracheidThreshold);

/**
 * Specifies if camera x is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Camera x is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasX(kVsSensorAdjust adjust, kSize index);

/**
 * Gets camera x.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     X.
 */
kVsFx(k32s) kVsSensorAdjust_CameraX(kVsSensorAdjust adjust, kSize index);

/**
 * Sets camera x.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  x                   X.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetX(kVsSensorAdjust adjust, kSize index, k32s x);

/**
 * Specifies if camera y is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Camera y is set.
 */
kVsFx(kBool) kVsSensorAdjust_CameraHasY(kVsSensorAdjust adjust, kSize index);

/**
 * Gets camera y.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @return                     Y.
 */
kVsFx(k32s) kVsSensorAdjust_CameraY(kVsSensorAdjust adjust, kSize index);

/**
 * Sets camera y.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Camera index.
 * @param  y                   Y.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_CameraSetY(kVsSensorAdjust adjust, kSize index, k32s y);

/**
 * Sets analog output count.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  count               Analog output count.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_SetAnalogOutputCount(kVsSensorAdjust adjust, kSize count);

/**
 * Gets analog output count.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @return                     Analog output count.
 */
kVsFx(kSize) kVsSensorAdjust_AnalogOutputCount(kVsSensorAdjust adjust);

/**
 * Specifies if analog output count is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Analog output index.
 * @return                     Analog output is set
 */
kVsFx(kBool) kVsSensorAdjust_HasAnalogOutput(kVsSensorAdjust adjust, kSize index);

/**
 * Gets min. input current.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Analog output index.
 * @return                     Min. input current.
 */
kVsFx(k64f) kVsSensorAdjust_AnalogOutputLogicalMin(kVsSensorAdjust adjust, kSize index);

/**
 * Gets max. input current.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Analog output index.
 * @return                     Max. input current.
 */
kVsFx(k64f) kVsSensorAdjust_AnalogOutputLogicalMax(kVsSensorAdjust adjust, kSize index);

/**
 * Gets min. output current.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Analog output index.
 * @return                     Min. output current.
 */
kVsFx(k64f) kVsSensorAdjust_AnalogOutputPhysicalMin(kVsSensorAdjust adjust, kSize index);

/**
 * Gets max. output current.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Analog output index.
 * @return                     Max. output current.
 */
kVsFx(k64f) kVsSensorAdjust_AnalogOutputPhysicalMax(kVsSensorAdjust adjust, kSize index);

/**
 * Sets analog output.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Analog output index.
 * @param  logicalMin          Min. input current.
 * @param  logicalMax          Max. input current.
 * @param  physicalMin         Min. output current.
 * @param  physicalMax         Max. output current.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_SetAnalogOutput(kVsSensorAdjust adjust, kSize index, k64f logicalMin, k64f logicalMax, k64f physicalMin, k64f physicalMax);

/**
 * Sets projector count.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  count               Projector count.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_SetProjectorCount(kVsSensorAdjust adjust, kSize count);

/**
 * Gets projector count.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @return                     Projector count.
 */
kVsFx(kSize) kVsSensorAdjust_ProjectorCount(kVsSensorAdjust adjust);

/**
 * Specifies if projector max intensity is set.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Projector index.
 * @return                     Projector max intensity is set.
 */
kVsFx(kBool) kVsSensorAdjust_ProjectorHasMaxIntensity(kVsSensorAdjust adjust, kSize index);

/**
 * Gets projector max intensity.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Projector index.
 * @return                     Projector max intensity.
 */
kVsFx(k64f) kVsSensorAdjust_ProjectorMaxIntensity(kVsSensorAdjust adjust, kSize index);

/**
 * Sets projector max intensity.
 * 
 * @public                     @memberof kVsSensorAdjust
 * @param  adjust              kVsSensorAdjust object.
 * @param  index               Projector index.
 * @param  maxIntensity        Projector max intensity.
 * @return                     Operation status
 */
kVsFx(kStatus) kVsSensorAdjust_ProjectorSetMaxIntensity(kVsSensorAdjust adjust, kSize index, k64f maxIntensity);

#include <kVision/Vs/kVsSensorAdjust.x.h>

#endif /* #ifndef K_VISION_VS_SENSOR_ADJUST_H */

