/** 
 * @file    kVsModelDef.h
 * @brief   Declares the kVsModelDef class. 
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_VISION_MODEL_DEF_H
#define K_VISION_MODEL_DEF_H

#include <kVision/Common/kVision.h>
#include <kFireSync/Utils/kSensorIdentity.h>
#include <kFireSync/kNodeDef.h>

/**
 * @class       kVsModelDef
 * @extends     kObject
 * @ingroup     kVision-Vs
 * @brief       
 */
typedef kObject kVsModelDef;

/** 
 * Constructs a kVsModelDef object.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDefs   Destination for the constructed object handle.
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsModelDef_Construct(kVsModelDef* modelDefs, kAlloc allocator);

/** 
 * Returns the name at the index.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   index       Index of the field.
 * @return              Name of the field.
 */
kVsFx(const kChar*) kVsModelDef_FieldNameAt(kVsModelDef modelDef, kSize index);

/** 
 * Returns the value at the index.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   index       Index of the field.
 * @return              Value of the field. 
 */
kVsFx(const kChar*) kVsModelDef_ValueAt(kVsModelDef modelDef, kSize index);

/** 
 * Returns the value at the name, or a blank string if it doesn't exist.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   name        Name of the field.
 * @return              Value of the field.
 */
kVsFx(const kChar*) kVsModelDef_ValueAtKey(kVsModelDef modelDef, const kChar* name);

/** 
 * Returns the number of fields in this ModelDef.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @return              Number of fields.
 */
kVsFx(kSize) kVsModelDef_FieldCount(kVsModelDef modelDef);

/** 
 * Returns kTRUE if a given field exists, kFALSE otherwise.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   name        Name of the field to search for.
 * @return              Whether a field exists.
 */
kVsFx(kBool) kVsModelDef_FieldExists(kVsModelDef modelDef, const kChar* name);

/** 
 * Sets an existing fields value, or adds a new field with the given name and value.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   name        Name of the to field to add or modify.
 * @param   value       Value of the field.
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsModelDef_SetField(kVsModelDef modelDef, const kChar* name, const kChar* value);

/** 
 * Serializes the Model Def to a kXml object.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   xml         Handle to the constructed xml object.
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsModelDef_ToXml(kVsModelDef modelDef, kXml* xml, kAlloc allocator);

/** 
 * Constructs a new Model Def from a SensorIdentity object. This simply takes all fields from the identity and stores them
 * in the new modelDef object.
 *
 * The prefix is a filter on the identity. The identity is representative of an xml object, so information can be stored in 
 * 'branches' using '\\' separators. This function will only load fields from within the sub-branch indicated by the prefix.
 *
 * @public              @memberof kVsModelDef
 * @param   identity    kSensorIdentity object.
 * @param   modelDef    Handle to constructed kVsModelDef object.
 * @param   prefix      Prefix string.
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsModelDef_LoadFromSensorIdentity(kSensorIdentity identity, kVsModelDef* modelDef, const kChar* prefix);

/** 
 * Constructs a kMap of key-values using the Model Def fields, and optionally the nodeInfo ControllerType and task.
 *
 * This is a helper function for filtering kVsModelDefstore and OptionLists. nodeInfo and task can be left blank or kNULL
 * to return a map of only the modeldef fields.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   nodeInfo    Pointer to a nodeInfo containing ControllerType (M1, M2, etc.)
 * @param   task        Task identifier
 * @param   filter      Handle to constructed kMap object.
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsModelDef_ToFilterMap(kVsModelDef modelDef, const kNodeInfo* nodeInfo, kMap* filter);



/** 
 * Compares all the fields in the modelDef to the fields in the other and returns true only if they are all identical, otherwise, false. 
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object.
 * @param   other       kVsModelDef object to compare to. 
 * @return              Comparison result.
 */
kVsFx(kBool) kVsModelDef_IsEqualTo(kVsModelDef modelDef, kVsModelDef other);


/** 
 * Merges the Model Def object into a SensorIdentity object.
 *
 * The prefix is a filter on the identity. The identity is representative of an xml object, so information can be stored in 
 * 'branches' using '\\' separators. This function will merge fields into the branch indicated by the prefix.
 *
 * @public              @memberof kVsModelDef
 * @param   modelDef    kVsModelDef object, or kNULL to remove that prefix branch.
 * @param   identity    kSensorIdentity object to merge into.
 * @param   prefix      Prefix string.
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsModelDef_MergeIntoSensorIdentity(kVsModelDef modelDef, kSensorIdentity identity, const kChar* prefix);

kVsFx(kStatus) kVsModelDef_UnionModels(kVsModelDef modelA, kVsModelDef modelB, kVsModelDef* result);

/** 
 * Returns the Hardware Part Number;
 *
 * @param   modelDef    kVsModelDef object.
 * @return              Part Number string. 
 */
kVsFx(const kChar*) kVsModelDef_HardwarePartNumber(kVsModelDef modelDef);

/** 
 * Returns the (Final) Part Number;
 *
 * @param   modelDef    kVsModelDef object.
 * @return              Part Number string. 
 */
kVsFx(const kChar*) kVsModelDef_FinalPartNumber(kVsModelDef modelDef);

/**
* Uses kVsLog to display current set of filters
*
* @param   modelDef    kVsModelDef object.
* @return              Operation status.
*/
kVsFx(kStatus) kVsModelDef_Log(kVsModelDef modelDef);

/** 
 * Loads a Factory ModelDef object from a Sensor Identity object
 *
 * @param   identity    kSensorIdentity object.
 * @param   modelDef    Destination for the constructed object handle.
 * @return              Operation status. 
 */


kVsFx(kStatus) kVsModelDef_LoadFactoryDef(kSensorIdentity identity, kVsModelDef* modelDef);

/** 
 * Loads a Final ModelDef object from a Sensor Identity object
 *
 * @param   identity    kSensorIdentity object.
 * @param   modelDef    Destination for the constructed object handle.
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsModelDef_LoadFinalDef(kSensorIdentity identity, kVsModelDef* modelDef);

/** 
 * Merges a Model Def object into the sub folder for Factory keys in the Sensor Identity.
 *
 * This will create the fields if they don't exist, or overwrite existing values.
 *
 * @param   identity    kSensorIdentity object.
 * @param   modelDef    kVsModelDef object.
 * @return              Operation status. 
 */
kVsFx(kStatus) kSensorIdentity_MergeFactoryModelDef(kSensorIdentity identity, kVsModelDef modelDef);

/** 
 * Merges a Model Def object into the sub folder for Final keys in the Sensor Identity.
 *
 * This will create the fields if they don't exist, or overwrite existing values.
 *
 * @param   identity    kSensorIdentity object.
 * @param   modelDef    kVsModelDef object.
 * @return              Operation status. 
 */
kVsFx(kStatus) kSensorIdentity_MergeFinalModelDef(kSensorIdentity identity, kVsModelDef modelDef);

#include <kVision/Vs/kVsModelDef.x.h>

#endif  /* K_VISION_MODEL_DEF_H */
