/**
* @file    kS3dVolumeCheckUtilities.h
* @brief   Some commonly used and math-related utilities.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKUTILITIES_H
#define KVISION_KS3D_VOLUMECHECKUTILITIES_H

//#define KS3DVOLUMECHECK_DEBUGLOG

#define KVISION_EXPORTS 1

#include <kApi/kApiDef.h>
#include <kVision/Common/kVision.h>
#include <kApi/Data/kMath.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArray3.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kBytes.h>
#include <kApi/Data/kString.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <math.h>
#include <time.h>

#define kS3DVOLUMCHECK_MEDIAN(A, B, C)   kClamp_((A), (kMin_((B),(C))), (kMax_((B),(C))))

typedef struct kS3dTransform6fd
{
    k64f X, Y, Z, A, B, C;
} kS3dTransform6fd;


typedef  struct kS3dEllipseParams
{
    k64f   x0, y0, xm, ym;
    k64f   theta;
    k64f   a, b, e2;
    k64f   dev;
    kBool  success;
} kS3dEllipseParams;


typedef enum kS3dArray2DataType
{
    kS3D_OMC_ARRAY2_K32F = 0,
    kS3D_OMC_ARRAY2_KPOINT32F_X,
    kS3D_OMC_ARRAY2_KPOINT32F_Y,
    kS3D_OMC_ARRAY2_KPOINT3D32F_X,
    kS3D_OMC_ARRAY2_KPOINT3D32F_Y,
    kS3D_OMC_ARRAY2_KPOINT3D32F_Z,
    kS3D_OMC_ARRAY2_K32S,
    kS3D_OMC_ARRAY2_KPOINT32S_X,
    kS3D_OMC_ARRAY2_KPOINT32S_Y,
    kS3D_OMC_ARRAY2_KPOINT3D32S_X,
    kS3D_OMC_ARRAY2_KPOINT3D32S_Y,
    kS3D_OMC_ARRAY2_KPOINT3D32S_Z,
    kS3D_OMC_ARRAY2_K16S,
    kS3D_OMC_ARRAY2_KPOINT16S_X,
    kS3D_OMC_ARRAY2_KPOINT16S_Y,
    kS3D_OMC_ARRAY2_KPOINT3D16S_X,
    kS3D_OMC_ARRAY2_KPOINT3D16S_Y,
    kS3D_OMC_ARRAY2_KPOINT3D16S_Z,
    kS3D_OMC_ARRAY2_K8S,
    kS3D_OMC_ARRAY2_K8U,
    kS3D_OMC_ARRAY2_KPOINT3D16S,
    kS3D_OMC_ARRAY2_KPOINT3D32S,
    kS3D_OMC_ARRAY2_KPOINT3D32F,
    kS3D_OMC_ARRAY2_KPOINT3D64F
} kS3dArray2DataType;


/*typedef enum kM3dPolynomialPictureType
{
    kM3dPolynomialGridX_XkYk = 0,
    kM3dPolynomialGridY_XkYk = 1,
    kM3dPolynomialGrooveZx_XkYk = 0,
    kM3dPolynomialGrooveDZx_XkYk = 1,
    kM3dPolynomialPlaneZx_XkYk = 0,
    kM3dPolynomialPlaneDZx_XkYk = 1,
    kM3dPolynomialXk_GridXY = 2,
    kM3dPolynomialYk_GridXY = 3,
    kM3dPolynomialXp_XkYk = 4,
    kM3dPolynomialYp_XkYk = 5,
    kM3dPolynomialXp_GridXY = 5,
    kM3dPolynomialXk_XpYp = 6,
    kM3dPolynomialYk_XpYp = 7,
    kM3dPolynomialCameraLensDevX = 6,
    kM3dPolynomialCameraLensDevY = 7,
    kM3dPolynomialTemporary = 8
} kM3dPolynomialPictureType;
*/

typedef struct kS3dPictureInfoItem
{
    k64f R;
    k64f K;
    k64f weight;
    k64f rate;
    k64f gridContrast;
    k64f contrast;
    k64f uniformity;
    k64f symmetry;
    k64f flatness;
    k64f nonlinearity;

    k64f gridDev;
    k64f gridDevMax;
    k64f gridDevMin;

    k64f phaseDev;
    k64f phaseDevMax;
    k64f phaseDevMin;
    k64f invalidRatio;

    k32s pictureMode;

    k64f cameraRotation;
    k64f cameraTiltAngle;
    k64f dsistorSymmetry;
    kBool calculated;
    kBool absoluteCoord;
    k16s DotsCount;
    k16s DotsRowCount;
    kBool zeroPosition;
    k64f temperature;
    k64f increase;
    k64f projectRotation;
    k64f depth;
    k64f targetPosZ;
    kL3dTransform3d cameraT;
    kL3dTransform3d projectorT;
    k32s gridIndex;
    k32s planeIndex;
    k32s grooveCount;
} kS3dPictureInfoItem;


#define kS3D_JAKOBEIGENVALUE_EPS   (0.00000000001)
#define kS3D_JAKOBEIGENVALUE_MAXIMUM_PEPEAT   (30000)
#define kS3D_JAKOBEIGENVALUE_ORDER             (4) 
#define kS3D_JAKOBEIGENVALUE_MATRIXSIZE        (16) 

/**
* Rewrites arctangent function with overflow checking.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   A           value in y direction.
* @param   B           value in x direction.
* @return              angle in radians of k64f.
*/
kVsFx(k64f) kS3dMath_Atan2(k64f A, k64f B);

/**
* Rewrites arctangent function with overflow checking.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   A           value in y direction.
* @param   B           value in x direction.
* @return              angle in degree of k64f.
*/
kVsFx(k64f) kS3dMath_Atan2InDeg(k64f A, k64f B);

/**
* Rewrites arccosine function with range checking.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   value       input value.
* @return              angle in radians of k64f.
*/
kVsFx(k64f) kS3dMath_Acos(k64f value);

/**
* Rewrites arccosine function with range checking.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   value       input value.
* @return              angle in degree of k64f.
*/
kVsFx(k64f) kS3dMath_AcosInDeg(k64f value);

/**
* Rewrites arcsine function with range checking.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   value       input value.
* @return              angle in radians of k64f.
*/
kVsFx(k64f) kS3dMath_Asin(k64f value);

/**
* Rewrites arcsine function with range checking.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   value       input value.
* @return              angle in degree of k64f.
*/
kVsFx(k64f) kS3dMath_AsinInDeg(k64f value);

/**
* Calculating tilt transformation matrix from a normal vactor.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   T           Handel of a structure kL3dTransform3d. Output transform matrix.
* @param   nx          vector x component
* @param   ny          vector y component
* @param   nz          vector z component
* @return              Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_TiltTransform3d(kL3dTransform3d *T, k64f nx, k64f ny, k64f nz);

/**
* Multiply operation of a matrix leftT and the transpose of a matrix rightT.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   T           Handel of a structure kL3dTransform3d. Output transform matrix.
* @param   leftT       kL3dTransform3d object.
* @param   rightT      kL3dTransform3d object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_LeftTMulRightTt(kL3dTransform3d *T, kL3dTransform3d leftT, kL3dTransform3d rightT);

/**
* Converts a kS3dTransform6fd object to transformation matrix.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   T           Handel of a structure kL3dTransform3d. Output transform matrix.
* @param   pos         Input coordinate of kS3dTransform6fd object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_Transform6fdToTransform3d(kL3dTransform3d *T, kS3dTransform6fd pos);

/**
* Get value from a kArray2 object at given coordinate
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   data        kArray2 object of 32f or 32s.
* @param   j           Integer coordinate Y.
* @param   i           Integer coordinate X.
* @param   type        Type of Array object and output element (defined in kS3dArray2DataType).
* @return              Element value.
*/
kVsFx(k32f) kS3dUtilities_Array2Value(kArray2 data, k32s j, k32s i, kS3dArray2DataType type);

/**
* Set value into a kArray2 object at given coordinate
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   data        kArray2 object of 32f or 32s.
* @param   z           Input value.
* @param   j           Integer coordinate Y.
* @param   i           Integer coordinate X.
* @param   type        Type of Array object and output element (defined in kS3dArray2DataType).
* @return              Element value.
*/
kVsFx(kStatus) kS3dUtilities_SetArray2Value(kArray2 data, k32f z, k32s j, k32s i, kS3dArray2DataType type);


/**
* Calculates the inverse matrix of a 3 x 3 symmetric matrix.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   srcMat      Handel of a k64f array. Input symmetric matrix.
* @param   invMat      Handel of a k64f array. Output inverse matrix.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dUtilities_Inverse3(k64f* srcMat, k64f* invMat);

/**
* Calculates the inverse matrix of a symmetric matrix.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   srcMat      Handel of a k64f array. Input symmetric matrix.
* @param   invMat      Handel of a k64f array. Output inverse matrix.
* @param   size        size of symmetric matrix
* @return              Operation status.
*/
kVsFx(kStatus) kS3dUtilities_Inverse(k64f* srcMat, k64f* invMat, k32s size);

/**
* Smoothing height map using weighted average method.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   map         Input / output height map kArray2 <k32f > to be smotthed.
* @param   temp        Temporary buffer with same type and size to the map.
* @param   weight      Weight = 2 corresponds the gauss filter, weight = 1 corresponds average filter.  
* @param   repeat      Repeating count. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dUtilities_SmoothMap32F(kArray2 map, kArray2 temp, k32s weight, k32s repeat);

/**
* Set the transformation matrix to an identity matrix.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   T           Handel of a structure kL3dTransform3d. Output transform matrix.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_SetTransformZero(kL3dTransform3d* T);

/**
* Calculates the eigenvalues and eigenvectors of a symmetric matrix.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   pMat        Handel of a k64f array. Input symmetric matrix.
* @param   pVec        Handel of a k64f array. Output eigenvectors are stored here.
* @param   eps         Error tolerance.
* @param   order       Size of symmetric matrix
* @param   repeat      Maximal iterative step.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dUtilities_JakobiEigenValue(k64f* pMat, k64f* pVec, k64f eps, k16s order, k32s repeat);

/**
* Calculates the transform matrix from a symmetric matrix.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   T           Handel of a structure kL3dTransform3d. Output transform matrix.
* @param   pMat        Handel of a k64f array. Input symmetric matrix 4 x 4.
* @param   center1     Weighting center position of first point group.
* @param   center2     Weighting center position of second point group.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dUtilities_QuotermTransformMatrix(kL3dTransform3d* T, k64f* pMat, kPoint3d64f center1, kPoint3d64f center2);

/**
* Calculates a normal vector from AB angle valus.
*
* @public               @memberof kS3dVolumeCheckUtilities
* @param   vector       Output vector. Handle of kPoint3d64f.
* @param   A            Input tilt angle.
* @param   B            Input rotation angle.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_VectorFromAB(kPoint3d64f* vector, k64f A, k64f B);

/**
* Calculating AB angle valus from a normal vector.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   A           Handle of k64f. Output elevation angle.
* @param   B           Handle of k64f. Output tilt angle.
* @param   nx          vector x component
* @param   ny          vector y component
* @param   nz          vector z component
* @return              Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_VectorToAB(k64f *A, k64f *B, k64f nx, k64f ny, k64f nz);

/**
* Normalizing a normal vector.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   vector      Vector. Handle of kPoint3d64f.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_NormalizeVector(kPoint3d64f* vector);

/**
* Calculating ABC angle valus from a transformation matrix.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   A           Handle of k64f. Output elevation angle.
* @param   B           Handle of k64f. Output tilt angle.
* @param   C           Handle of k64f. Output azimuth angle.
* @param   T           kL3dTransform3d object. Input transform matrix.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dTransform3d_AbcFromTransform3D(k64f* A, k64f* B, k64f* C, kL3dTransform3d T);

/**
* Write the content of the using a printf-like format string and arguments to the log file:  "c:/temp/VolumeChecker.log"
* Only used during the development.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   format      Print format string.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckDebug_WriteLog(kChar* format, ...);

/**
* Write the transformation matrix to the log file:  "c:/temp/VolumeChecker.log"
* Only used during the development.
*
* @public              @memberof kS3dVolumeCheckUtilities
* @param   T           kL3dTransform3d object. Input transformation matrix.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckDebug_WriteTransform3DLog(kL3dTransform3d  T);

#include <kVision/S3d/kS3dVolumeCheckUtilities.x.h>

#endif  /* #ifndef KVISION_KS3D_VOLUMECHECKUTILITIES_H */
