/**
* @file    kS3dVolumeCheckSubPixLut.h
* @brief   Declares the kS3dVolumeCheckSubPixLut class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKSUBPIXLUT_H
#define KVISION_KS3D_VOLUMECHECKSUBPIXLUT_H

#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

#define kS3DSUBPIXELTABLE_WIDTH_SHIFT      (5)
#define kS3DSUBPIXELTABLE_WIDTH            (1 << kS3DSUBPIXELTABLE_WIDTH_SHIFT)
#define kS3DSUBPIXELTABLE_WINDOW           (2)
#define kS3DSUBPIXELTABLE_WEIGHT_POWER     (4)
#define kS3DSUBPIXELTABLE_MAX              (1024)
#define kS3DSUBPIXELTABLE_SIGMA            (2)  
#define kS3DSUBPIXELTABLE_THRESHOLD        (5)  

/**
* @class       kS3dVolumeCheckSubPixLut
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Class for the Lookup table to speed up the resampling of height image with small 3D transformation.
*/
typedef kObject kS3dVolumeCheckSubPixLut;

/**
* Constructs a kS3dVolumeCheckSubPixLut object
* To save time, the lookup table is not generated automatically. 
* But before using this class it will be checked, and recreated if required.
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSubPixLut_Construct(kS3dVolumeCheckSubPixLut* lut, kAlloc allocator);

/**
* Calculating the lookup table after the settings are updated or the default values are used.
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSubPixLut_Setup(kS3dVolumeCheckSubPixLut lut);

/**
* Ensures the lookup table setup.
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckSubPixLut_CheckSetup(kS3dVolumeCheckSubPixLut lut);

/**
*  Get the Lut data of kArray1 < kTypeOf(k16s) >.
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              Lookup table of kArray1 < kTypeOf(k16s) >.
*/
kVsFx(kArray1) kS3dVolumeCheckSubPixLut_Data(kS3dVolumeCheckSubPixLut lut);

/**
*  Get standard deviation sigma value in pixel. default: 2
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              standard deviation.
*/
kVsFx(k32f)  kS3dVolumeCheckSubPixLut_Sigma(kS3dVolumeCheckSubPixLut lut);

/**
*  Get the window width of resampling in pixel. default: 2
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              window width.
*/
kVsFx(k16u) kS3dVolumeCheckSubPixLut_Thick(kS3dVolumeCheckSubPixLut lut);

/**
*  Get the subpixel resolutuon in 1/pixel. default: 50   resolution = 0.02 pixel.
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              1 / subpixel resolutuon.
*/
kVsFx(k16u) kS3dVolumeCheckSubPixLut_WidthShift(kS3dVolumeCheckSubPixLut lut);

/**
*  Get power factor to the low pass function. The larger the value, the more the curve will narrow. default: 4
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              power factor.
*/
kVsFx(k16u) kS3dVolumeCheckSubPixLut_WeightPow(kS3dVolumeCheckSubPixLut lut);

/**
*  Get maximum value of the filter function.  default: 1024
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              maximum value of the filter function.
*/
kVsFx(k16s) kS3dVolumeCheckSubPixLut_MaxValue(kS3dVolumeCheckSubPixLut lut);

/**
*  Set standard deviation sigma value in pixel. default: 2
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @param   sigma       standard deviation.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckSubPixLut_SetSigma(kS3dVolumeCheckSubPixLut lut, k32f sigma);

/**
*  Set the window width of resampling in pixel. default: 2
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @param   window      Window width.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSubPixLut_SetWindow(kS3dVolumeCheckSubPixLut lut, k16u window);

/**
*  Set the subpixel resolutuon in 1. / ( 1 << withShift) pixel. default: 5;   width = 32  resolution = 0.03125 pixel.
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @param   widthShift  width = (1<<widthShift) = 1 / subpixel resolutuon.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSubPixLut_SetWidthShift(kS3dVolumeCheckSubPixLut lut, k16u widthShift);

/**
*  Set power factor to the low pass function. The larger the value, the more the curve will narrow. default: 4
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @param   weightPow   power factor.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSubPixLut_SetWeightPow(kS3dVolumeCheckSubPixLut lut, k16u weightPow);

/**
*  Set maximum value of the filter function.  default: 1024
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @param   maxValue    maximum value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSubPixLut_SetMaxValue(kS3dVolumeCheckSubPixLut lut, k16s maxValue);


/**
*  Get threshold of the filter function.
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @return              Threshold.
*/
kVsFx(k16s) kS3dVolumeCheckSubPixLut_Threshold(kS3dVolumeCheckSubPixLut lut);

/**
*  Set threshold of the filter function.  default: 5
*
* @public              @memberof kS3dVolumeCheckSubPixLut
* @param   lut         kS3dVolumeCheckSubPixLut object.
* @param   threshold   Threshold.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckSubPixLut_SetThreshold(kS3dVolumeCheckSubPixLut lut, k16s threshold);

#include <kVision/S3d/kS3dVolumeCheckSubPixLut.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKSUBPIXLUT_H */ 
