/**
* @file    kS3dVolumeCheckAlg.h
* @brief   Declares the kS3dVolumeCheckAlg class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKALG_H
#define KVISION_KS3D_VOLUMECHECKALG_H

#include <kVision/S3d/kS3dVolumeCheckDef.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/S3d/kS3dVolumeCheckAlign.h>
#include <kVision/S3d/kS3dVolumeCheckValveAlg.h>
#include <kVision/S3d/kS3dVolumeCheckSparkAlg.h>
#include <kVision/S3d/kS3dVolumeCheckSubPixLut.h>
#include <kVision/S3d/kS3dVolumeCheckPlotter.h>
#include <kVision/S3d/kS3dVolumeCheckLowPass.h>
#include <kApi/Threads/kTimer.h>
#include <kApi/Data/kImage.h>

#define kS3D_VOLUMECHECK_OUTPUTSIZEX          (1000) 
#define kS3D_VOLUMECHECK_OUTPUTSIZEY          (1000) 
#define kS3D_VOLUMECHECK_OUTPUTPIXELPITCH     (0.1) 

/**
* @class       kS3dVolumeCheckAlg
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       The main algorithm class for the VolumeChecker application.
*/
typedef kObject kS3dVolumeCheckAlg;

/**
* Constructs a kS3dVolumeCheckAlg object
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         Destination for the constructed object handle.
* @param   subPixLut   Object of kS3dVolumeCheckSubPixLut. Lookup table to speed up the resampling of height image with small 3D transformation.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_Construct(kS3dVolumeCheckAlg* alg, kS3dVolumeCheckSubPixLut  subPixLut, kAlloc allocator);


/**
* Set input data for the VolumeCheck measurement.
* The accepted kArray2 type are:
*     kArray2 < kPoint3d16s >   xyScale == zScale = range resolution in mm
*     kArray2 < kPoint3d32s >   xyScale == zScale = range resolution in mm
*     kArray2 < kPoint3d32f >   xyScale == zScale = 1
*     kArray2 < kPoint3d64f >   xyScale == zScale = 1
*     kArray2 < k16s >          xyScale= pixel pitch in mm, zScale = z resolution in mm
*     kArray2 < k32s >          xyScale= pixel pitch in mm, zScale = z resolution in mm
*     kArray2 < k32f >          xyScale= pixel pitch in mm, zScale = 1
*     kArray2 < k64f >          xyScale= pixel pitch in mm, zScale = 1
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   input       input data of kArray2.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetInputData(kS3dVolumeCheckAlg alg, kArray2 input);

/**
* Executes the algorithm for calculating the volume value of a combustion chamber.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_Run(kS3dVolumeCheckAlg alg);

/**
*  Get center of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              center of valve seat ring.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlg_ValveCenter(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Set center of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   center      center of valve seat ring.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValveCenter(kS3dVolumeCheckAlg alg, kPoint3d64f center, k32u valveIndex);

/**
*  Get rotation angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of valve.
* @return              rotation angle.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_ValvePolarAngle(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Set rotation angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   polarAngle  rotation angle.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValvePolarAngle(kS3dVolumeCheckAlg alg, k64f polarAngle, k32u valveIndex);

/**
*  Get slope angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of valve.
* @return              slope angle.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_ValveSlopeAngle(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Set slope angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   slopeAngle  slope angle.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValveSlopeAngle(kS3dVolumeCheckAlg alg, k64f slopeAngle, k32u valveIndex);

/**
*  Get nominal diameter of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of valve.
* @return              nominal diameter.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_ValveDiameter(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Set nominal diameter of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   diameter    nominal diameter.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckAlg_SetValveDiameter(kS3dVolumeCheckAlg alg, k64f diameter, k32u valveIndex);

// 
/**
*  Get Z search range in mm, default 2.5 mm
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of valve.
* @return              Z search range.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_ValveZRange(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Set Z search range in mm, default 2.5 mm
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   zRange      Z search range.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValveZRange(kS3dVolumeCheckAlg alg, k64f zRange, k32u valveIndex);

// 
/**
*  Get phase thick of the seat ring in mm, default kS3D_VOLUMECHECK_DEFAUT_SEATRING_THICK mm
*
* @public               @memberof kS3dVolumeCheckAlg
* @param   alg          kS3dVolumeCheckAlg object.
* @param   valveIndex   number of valve.
* @return               phase thick of the seat ring.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_RingPhaseThick(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Set phase thick of the seat ring in mm, default kS3D_VOLUMECHECK_DEFAUT_SEATRING_THICK mm
*
* @public               @memberof kS3dVolumeCheckAlg
* @param   alg          kS3dVolumeCheckAlg object.
* @param   thick        phase thick of the seat ring.
* @param   valveIndex   number of the valve.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetRingPhaseThick(kS3dVolumeCheckAlg alg, k64f thick, k32u valveIndex);

/**
*  Queries whether software cover for the valve is provided.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of valve.
* @return              Software cover mode.
*/
kVsFx(kS3dVolumeCheckValveSoftwareCoverMode) kS3dVolumeCheckAlg_ValveSoftCoverMode(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Set software cover mode of the valve.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   softCover   Software cover mode.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValveSoftCoverMode(kS3dVolumeCheckAlg alg, kS3dVolumeCheckValveSoftwareCoverMode softCover, k32u valveIndex);

/**
*  Get center of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sparkIndex  number of the spark.
* @return              center of the spark.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlg_SparkCenter(kS3dVolumeCheckAlg alg, k32u sparkIndex);

/**
*  Set center of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   center      center of the spark.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSparkCenter(kS3dVolumeCheckAlg alg, kPoint3d64f center, k32u sparkIndex);

/**
*  Get rotation angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sparkIndex  number of spark.
* @return              rotation angle.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_SparkPolarAngle(kS3dVolumeCheckAlg alg, k32u sparkIndex);

/**
*  Set rotation angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   polarAngle  rotation angle.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSparkPolarAngle(kS3dVolumeCheckAlg alg, k64f polarAngle, k32u sparkIndex);

/**
*  Get slope angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sparkIndex  number of spark.
* @return              slope angle.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_SparkSlopeAngle(kS3dVolumeCheckAlg alg, k32u sparkIndex);

/**
*  Set slope angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   slopeAngle  slope angle.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSparkSlopeAngle(kS3dVolumeCheckAlg alg, k64f slopeAngle, k32u sparkIndex);

/**
*  Get nominal diameter of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sparkIndex  number of spark.
* @return              nominal diameter.
*/
kVsFx(k64f)  kS3dVolumeCheckAlg_SparkDiameter(kS3dVolumeCheckAlg alg, k32u sparkIndex);

/**
*  Set nominal diameter of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   diameter    nominal diameter.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckAlg_SetSparkDiameter(kS3dVolumeCheckAlg alg, k64f diameter, k32u sparkIndex);

/**
*  Queries whether software cover for the spark is provided.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sparkIndex  number of spark.
* @return              is provided?
*/
kVsFx(kBool) kS3dVolumeCheckAlg_SparkSoftCover(kS3dVolumeCheckAlg alg, k32u sparkIndex);

/**
*  Set a boolean value to determine of the software cover for the spark.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   softCover   Software cover mode.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSparkSoftCover(kS3dVolumeCheckAlg alg, kBool softCover, k32u sparkIndex);

/**
* Set maximum iterative steps for the surface alignment.
*
* @public                  @memberof kS3dVolumeCheckAlg
* @param   alg             kS3dVolumeCheckAlg object.
* @param   repeat  maximum iterative steps
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetMaxAlignRepeat(kS3dVolumeCheckAlg alg, k32u repeat);

/**
* Get maximum iterative steps for the surface alignment.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              maximum iterative steps.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_MaxAlignRepeat(kS3dVolumeCheckAlg alg);

/**
* Set maximum iterative steps for the surface 2D rotation.
*
* @public            @memberof kS3dVolumeCheckAlg
* @param   alg       kS3dVolumeCheckAlg object.
* @param   repeat    maximum iterative steps
* @return            Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetMaxRotationRepeat(kS3dVolumeCheckAlg alg, k32u repeat);

/**
* Get maximum iterative steps for the surface 2D rotation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              maximum iterative steps.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_MaxRotationRepeat(kS3dVolumeCheckAlg alg);

/**
* Set the tolerance limit for the surface 2D rotation to stop the iterative rotation operation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   tolerance   Tolerance limit for the surface 2D rotation.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetRotationTolerance(kS3dVolumeCheckAlg alg, k64f tolerance);

/**
* Get the tolerance limit for the surface 2D rotation to stop the iterative rotation operation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Tolerance limit for the surface 2D rotation.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_RotationTolerance(kS3dVolumeCheckAlg alg);

/**
* Set the tolerance limit for the surface 2D translation to stop the iterative operation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   tolerance   Tolerance limit for the surface 2D translation.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetMoveTolerance(kS3dVolumeCheckAlg alg, k64f tolerance);

/**
* Get the tolerance limit for the surface 2D translation to stop the iterative operation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Tolerance limit for the surface 2D translation.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_MoveTolerance(kS3dVolumeCheckAlg alg);

/**
* Set window size for the smoothing border contour.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   window      Window size for the smoothing border contour.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetContourAverageWindow(kS3dVolumeCheckAlg alg, k32s window);

/**
* Get window size for the smoothing border contour.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Window size for the smoothing border contour.
*/
kVsFx(k32s) kS3dVolumeCheckAlg_ContourAverageWindow(kS3dVolumeCheckAlg alg);

/**
*  Set a boolean value to determine of the outlier filtering operation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   used        Set value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetOutlierFilterUsed(kS3dVolumeCheckAlg alg, kBool used);

/**
*  Queries whether outlier filtering is used.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              is used?
*/
kVsFx(kBool) kS3dVolumeCheckAlg_OutlierFilterUsed(kS3dVolumeCheckAlg alg);

/**
* Set window size for Outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   window      Window size for Outlier filtering.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetOutlierFilterWindow(kS3dVolumeCheckAlg alg, k32u window);

/**
* Get window size for Outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Window size for Outlier filtering.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_OutlierFilterWindow(kS3dVolumeCheckAlg alg);

/**
* Set repeat count for the outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   repeat      Repeat count for the outlier filtering
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetOutlierFilterRepeat(kS3dVolumeCheckAlg alg, k32u repeat);

/**
* Get repeat count for the outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Repeat count for the outlier filtering.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_OutlierFilterRepeat(kS3dVolumeCheckAlg alg);

/**
* Set tolerance value to identify the outliers. Points within the tolerance range are not affected.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   tolerance   Tolerance value to identify the outliers
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetOutlierFilterTolerance(kS3dVolumeCheckAlg alg, k64f tolerance);

/**
* Get tolerance value to identify the outliers. Points within the tolerance range are not affected.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Tolerance value to identify the outliers.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_OutlierFilterTolerance(kS3dVolumeCheckAlg alg);

/**
*  Set a boolean value to determine of the smooth filtering operation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   used        Set value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSmoothFilterUsed(kS3dVolumeCheckAlg alg, kBool used);

/**
*  Queries whether smooth filtering is used.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              is used?
*/
kVsFx(kBool) kS3dVolumeCheckAlg_SmoothFilterUsed(kS3dVolumeCheckAlg alg);

/**
* Set window size for smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   window      Window size for smooth filtering.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSmoothFilterWindow(kS3dVolumeCheckAlg alg, k32u window);

/**
* Get window size for smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Window size for smooth filtering.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_SmoothFilterWindow(kS3dVolumeCheckAlg alg);

/**
* Set repeat count for the smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   repeat      Repeat count for the smooth filtering
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSmoothFilterRepeat(kS3dVolumeCheckAlg alg, k32u repeat);

/**
* Get repeat count for the smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Repeat count for the smooth filtering.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_SmoothFilterRepeat(kS3dVolumeCheckAlg alg);

/**
* Set a correction factor to correct the calculated volume value when the sensor is found a systematic deviation,
* for example, by measuring a master-body. Normally the factor should be 1.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   factor      Correction factor.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetVolumeCorrectFactor(kS3dVolumeCheckAlg alg, k64f factor);

/**
* Get the correction factor to correct the calculated volume value when the sensor is found a systematic deviation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Correction factor.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_VolumeCorrectFactor(kS3dVolumeCheckAlg alg);

/**
* Get the rotation mode of the input data.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Rotation mode of the input data.
*/
kVsFx(kS3dVolumeCheckSensorRotationMode) kS3dVolumeCheckAlg_RotationMode(kS3dVolumeCheckAlg alg);

/**
* Set the rotation mode of the input data.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   mode        Rotation mode of the input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetRotationMode(kS3dVolumeCheckAlg alg, kS3dVolumeCheckSensorRotationMode mode);


/**
*  Queries whether the specified valve hole is detected.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              is used?
*/
kVsFx(kBool) kS3dVolumeCheckAlg_ValveHoleExists(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Get the actual calculated center of the specified valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              center of valve seat ring.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlg_ActualValveCenter(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Get the nominal center of the specified valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              Nominal center of valve seat ring.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlg_NominalValveCenter(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Get the indicator gaps-area ratio in determining the valve seat ring in %.
*
* To reconstruct the profile of the valve seat ring safely, 
* the completeness of the data collected in the observed radius area are substantially. 
* Unfortunately, it is not always the case. The uncertainty of the position determination 
* is affected dramatically by larger gaps. The reconstruction of the ring profile is performed 
* by iterative calculation of the circular area center and the roundness of the section contour 
* on each level. Here, the center should not change at different levels. The gaps or obviously 
* unusable points are filled gradually according to such restrictions.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              Gaps-area ratio of the valve seat ring.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_ValveRingGaps(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
*  Get the Indicator uncertainty of the profile of the valve seat ring in mm.
*
* This indicator is used as Hints of the reliability of the ring profile by backward calculation 
* of the point distance to the ring profile with a fixed noise limit. Since the noise ratios in 
* this area are much higher than on the reference plane. The akteptable noise limit was increased 
* to 5 pixel pitch. Small difference indicates that the reconstructed ring profile is relatively close 
* to the actually measured data. Following points may cause the increased uncertainty:
*
* 1. The quality of the measurement data by reflection and unfavorable orientation to the sensor too bad;
* 2. A large gap causes systematic errors in iterative reconstruction of the ring profile;
* 3. The seat ring is mounted a little different from the specified orientation;
* 4. The seat ring is not very circular (this could be very likely excluded).
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              Uncertainty of the profile of the valve seat ring.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_ValveAveDifference(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
* Get the width of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Width of height map
*/
kVsFx(k32s) kS3dVolumeCheckAlg_OutputSizeX(kS3dVolumeCheckAlg alg);

/**
* Set the width of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sizeX       Width of height map
* @return              Operation status.
*/

kVsFx(kStatus) kS3dVolumeCheckAlg_SetOutputSizeX(kS3dVolumeCheckAlg alg, k32s sizeX);

/**
* Get the height of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Height of height map
*/
kVsFx(k32s) kS3dVolumeCheckAlg_OutputSizeY(kS3dVolumeCheckAlg alg);

/**
* Set the height of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sizeY       Height of height map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetOutputSizeY(kS3dVolumeCheckAlg alg, k32s sizeY);

/**
* Set scaling of the specified output height map kArray2 < k32s >. It is always scale.x == scale.y
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   scale       Scaling of the height map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetOutputScale(kS3dVolumeCheckAlg alg, kPoint3d64f scale);

/**
* Get scaling of the specified output height map kArray2 < k32s >. It is always scale.x == scale.y
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Scaling of the height map.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlg_OutputScale(kS3dVolumeCheckAlg alg);

/**
*  Set the valve number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveCount  valve number.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValveCount(kS3dVolumeCheckAlg alg, k32u valveCount);

/**
*  Get the valve number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              valve number.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_ValveCount(kS3dVolumeCheckAlg alg);

/**
* Get scaling of the specified input height map or point clouds of kArray2.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Scaling the pouint cloud or height map.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlg_InputScale(kS3dVolumeCheckAlg alg);

/**
* Set scaling of the specified input height map or point clouds of kArray2.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   scale       Scaling of the pouint cloud or height map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetInputScale(kS3dVolumeCheckAlg alg, kPoint3d64f scale);

/**
*  Get magnification factor relative to nominal diameter to determine the search window.
*  The value depends on the positioning accuracy of the combustion chamber measurement. default value: 1.2
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              magnification factor.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_ValveRadiusFactor(kS3dVolumeCheckAlg alg);

/**
*  Set magnification factor relative to nominal diameter to determine the search window.
*  The value depends on the positioning accuracy of the combustion chamber measurement. default value: 1.4
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   factor      magnification factor.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValveRadiusFactor(kS3dVolumeCheckAlg alg, k64f factor);

/**
*  Get the volume offset value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              volume offset value.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_VolumeOffset(kS3dVolumeCheckAlg alg);

/**
*  Set the volume offset value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   offset      volume offset value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetVolumeOffset(kS3dVolumeCheckAlg alg, k64f offset);


/**
*  Get the nominal volume value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Nominal volume value.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_NominalVolume(kS3dVolumeCheckAlg alg);

/**
*  Set the nominal volume value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   volume      Nominal volume value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetNominalVolume(kS3dVolumeCheckAlg alg, k64f volume);

/**
*  Get the volume tolerance of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Volume tolerance.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_VolumeTolerance(kS3dVolumeCheckAlg alg);

/**
*  Set the volume tolerance of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   tol         Volume tolerance.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetVolumeTolerance(kS3dVolumeCheckAlg alg, k64f tol);

/**
*  Get the increment of searching of valve center in mm. default value: 0.01 mm
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              increment.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_ValveSearchStep(kS3dVolumeCheckAlg alg);

/**
*  Set the increment of searching of valve center in mm. default value: 0.01 mm
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   step        increment.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetValveSearchStep(kS3dVolumeCheckAlg alg, k64f step);

/**
*  Set the spark number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   sparkCount  spark number.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetSparkCount(kS3dVolumeCheckAlg alg, k32u sparkCount);

/**
*  Get the spark number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              spark number.
*/
kVsFx(k32u) kS3dVolumeCheckAlg_SparkCount(kS3dVolumeCheckAlg alg);

/**
* Get the volume value of the measurement.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              volume value of the measurement.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_Volume(kS3dVolumeCheckAlg alg);

/**
* Set the current number of measurement. Only used for internal statistic calculation and for storing files. Only for verification
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   index       current number of measurement.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetMeasureIndex(kS3dVolumeCheckAlg alg, k32s index);

/**
* Get the current number of measurement. Only used for internal statistic calculation and for storing files.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              current number of measurement.
*/
kVsFx(k32s) kS3dVolumeCheckAlg_MeasureIndex(kS3dVolumeCheckAlg alg);

/**
* Create a height map of valve seat ring for drawing: kArray2 <k64f>. Only for verification
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   map         Object handle of height map array. 
* @param   valveIndex  number of the valve.
* @param   isInput     If idInput == kTRUE return the height map from original data else return the fitting height map. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_ValveSeatRingMap64f(kS3dVolumeCheckAlg alg, kImage* map, k32u valveIndex, kBool isInput, kRect3d64f* rect);

/**
* Return sectional profile of kArrayList <kPoint32f>. Only used for verification
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              Sectional profile.
*/
kVsFx(kArrayList) kS3dVolumeCheckAlg_ValveProfile(kS3dVolumeCheckAlg alg, k32u valveIndex, k32s axisNo);

/**
* Creates a sectional profile of the specified valve seat ring. Only used for verification
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetupValveProfile(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
* Get the nominal radius  of the specified valve seat ring.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              Nominal radius.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_ValveNominalRadius(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
* Get the actual z position of the specified valve seat ring center.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   valveIndex  number of the valve.
* @return              Z position.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_ValveActualZposition(kS3dVolumeCheckAlg alg, k32u valveIndex);

/**
* Return the output height map of kArray2 <k32s> for drawing. Only used for verification.
* It can also be used for example to store a new template.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Operation status.
*/
kVsFx(kArray2)  kS3dVolumeCheckAlg_OutputMap(kS3dVolumeCheckAlg alg);

/**
* Creates a height map from input point cloud of kArray2 <k32s> for drawing. Only used for verification
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_ResampleInputMap(kS3dVolumeCheckAlg alg);


/**
* Get the actual offset position. 
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              actual offset position.
*/
kVsFx(kPoint64f) kS3dVolumeCheckAlg_ActualCenterOffset(kS3dVolumeCheckAlg alg);


/**
*  Get the Indicator uncertainty of the alignment with the reference plane in mm.
*
* The alignment is performed by iterative determination of the balancing plane. 
* A small percentage of data are excreted with larger deviations from the mean plane 
* for the following calculation in each step, until the change of the mean plane reaches 
* a certain limit. This indicator is used as Hints of the reliability of the plane determining 
* by backward calculation of the plane with a fixed noise limit. The average pixel pitch of about 
* 0.1 mm was used in the calculation. Small difference indicates that the balancing plane is 
* relatively insensitive of the selection of the involved points in the calculation.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Uncertainty of the alignment with the reference plane.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_ActualPlaneDifference(kS3dVolumeCheckAlg alg);

/**
*  Get offset values between the defined coordinate origin and calculated roughly chamber center in mm.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Lateral offset value.
*/
kVsFx(kPoint64f) kS3dVolumeCheckAlg_ChamberCenterOffset(kS3dVolumeCheckAlg alg);

/**
*  Set offset values between the defined coordinate origin and calculated roughly chamber center in mm.
*  (center of valid measuring surface area).
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   center      Lateral offset value of kPoint64f.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetChamberCenterOffset(kS3dVolumeCheckAlg alg, kPoint64f center);

/**
* Get the slope angle of the plane surface normal vector corresponding to the upper reference plane
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Slope angle of the surface normal vector.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_SlopeAngle(kS3dVolumeCheckAlg alg);

/**
* Get the orientation angle of the plane surface normal vector corresponding to the upper reference plane
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Orientation angle of the surface normal vector.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_PolarAngle(kS3dVolumeCheckAlg alg);

/**
* Get the rotation angle of the combustion chamber which has been rotated
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Rotation angle.
*/
kVsFx(k64f) kS3dVolumeCheckAlg_RotationAngle(kS3dVolumeCheckAlg alg);

/**
* Get the part running time from start to the time point that the point clouds are aligned to the upper flat area.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Part running time.
*/
kVsFx(k64s) kS3dVolumeCheckAlg_AlignTime(kS3dVolumeCheckAlg alg);

/**
* Get the part running time from start to the time point that the point clouds are segmented roughly into different parts.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Part running time.
*/
kVsFx(k64s) kS3dVolumeCheckAlg_SegmentTime(kS3dVolumeCheckAlg alg);

/**
* Get the part running time from start to the time point that the valves are detected.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Part running time.
*/
kVsFx(k64s) kS3dVolumeCheckAlg_ValveTime(kS3dVolumeCheckAlg alg);

/**
* Get the part running time from start to the time point that the output height map is created.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Part running time.
*/
kVsFx(k64s) kS3dVolumeCheckAlg_MapTime(kS3dVolumeCheckAlg alg);

/**
* Get the part running time from start to the time point that holes are filled.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Part running time.
*/
kVsFx(k64s) kS3dVolumeCheckAlg_SparkTime(kS3dVolumeCheckAlg alg);

/**
* Get the part running time from start to the time point that the optional filtering is completed.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Part running time.
*/
kVsFx(k64s) kS3dVolumeCheckAlg_FilterTime(kS3dVolumeCheckAlg alg);

/**
* Get the total running time of the measurement. 
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Total running time.
*/
kVsFx(k64s) kS3dVolumeCheckAlg_TotalTime(kS3dVolumeCheckAlg alg);

/**
* Get origin of the sensor coordinate system to the origin of the coordinate system of combustion chamber.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Origin position.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckAlg_Center(kS3dVolumeCheckAlg alg);

/**
* Creats a kPlot object to plot the output surface.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   outPlot     Object handle of output kPlot. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_Plot(kS3dVolumeCheckAlg alg, kPlot* outPlot);

/**
* The input height map will be replaced by the calculated height map. The function can be used in GDK optionally.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   offset      An offset value in z direction to ensure the reference plane is set to zero in the diagram.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_ResampleInputData(kS3dVolumeCheckAlg alg, k64f offset);

/**
* Set the template kArray2 for the filling missing points.
* Note:  checking kS3dVolumeCheckAlg_OutputSizeX(kS3dVolumeCheckAlg alg);
*                 kS3dVolumeCheckAlg_OutputSizeX(kS3dVolumeCheckAlg alg);
*                 kS3dVolumeCheckAlg_OutputScale(kS3dVolumeCheckAlg alg);
*                 The parameters of output data should be identical to the template.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   templateMap template kArray2.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetTemplateArray(kS3dVolumeCheckAlg alg, const kArray2 templateMap);


/**
* Loads a template height map for the filling missing points.
* Note:  checking kS3dVolumeCheckAlg_OutputSizeX(kS3dVolumeCheckAlg alg);
*                 kS3dVolumeCheckAlg_OutputSizeX(kS3dVolumeCheckAlg alg);
*                 kS3dVolumeCheckAlg_OutputScale(kS3dVolumeCheckAlg alg);
*                 The parameters of output data should be identical to the template.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   fileName    File name of the template. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_LoadTemplateData(kS3dVolumeCheckAlg alg, const kChar* fileName);

/**
* Create a temporary template height map by using the current output height map. 
* This is only a test function used in the G3-kFactory activity.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_ReplaceTemplateData(kS3dVolumeCheckAlg alg);

/**
* Stores a template height map for the filling missing points by using the current output height map.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   fileName    File name of the template.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_ExportTemplateData(kS3dVolumeCheckAlg alg, const kChar* fileName);

/**
* Set a boolean value to switch whether the measuring mode with template should be used.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   used        If the value "used" is set kTRUE and the template exists, the measuring mode with template is turned on.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetTemplateUsed(kS3dVolumeCheckAlg alg, kBool used);

/**
* Get a boolean value for switching the measuring mode with or without template.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   used        If the value "used" is set kTRUE and the template exists, the option measuring with template is turned on.
* @return              The boolean value for switching the measuring mode.
*/
kVsFx(kBool) kS3dVolumeCheckAlg_TemplateUsed(kS3dVolumeCheckAlg alg);

/**
* Set a boolean value for displaying an alternative difference height map compared to the template.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @param   showDiff    If the value "showDiff" is set kTRUE and the template measuring mode is on, the displaying difference height map is turned on.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckAlg_SetShowDifferenceMap(kS3dVolumeCheckAlg alg, kBool showDiff);

/**
* Get a boolean value for displaying in template measuring mode.
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   alg         kS3dVolumeCheckAlg object.
* @return              The boolean value displaying in template measuring mode.
*/
kVsFx(kBool) kS3dVolumeCheckAlg_ShowDifferenceMap(kS3dVolumeCheckAlg alg);

#include <kVision/S3d/kS3dVolumeCheckAlg.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKALG_H */
