/**
 * @file    ${filename_header_public}}
 * @brief   Declares the ${classname} class.
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kS3D_SURFACE_GENERATOR_H
#define kS3D_SURFACE_GENERATOR_H


#include <kVision/Vs/kVsJobQueue.h>
#include <kVision/S3d/kS3dStereoProfiler.h>
#include <kVision/S3d/kS3dMonoMergeAlg.h>
#include <kVision/S3d/kS3dMonoMergeLocalAlg.h>
#include <kVision/S3d/kS3dMonoMergeRigidAlg.h>

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

/**
 * @class                    kS3dSurfaceGenerator
 * @extends                  kObject
 * @ingroup                  kVision - S3d
 * @brief                    Algorithm class for merging mono and stereo meshes
 */

typedef kObject kS3dSurfaceGenerator;

/**
 * Constructs a kS3dSurfaceGenerator object.
 *
 * @public                   @memberof kS3dSurfaceGenerator
 * @param   alg              Destination for the constructed object handle.
 * @param   profiler         kS3dStereoProfiler object.
 * @param   allocator        Memory allocator (or kNULL for default).
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dSurfaceGenerator_Construct(kS3dSurfaceGenerator* alg, kS3dStereoProfiler profiler, kAlloc allocator);

/**
 * Optional function to preallocate some of the internal buffers.
 * Requires kS3dSurfaceGenerator_SetOutputHeight() and kS3dSurfaceGenerator_SetOutputWidth() to be set.
 *
 * @public                   @memberof kS3dSurfaceGenerator
 * @param   alg              Alg object
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dSurfaceGenerator_Setup(kS3dSurfaceGenerator alg);

/**
 * Optional function to set the input output array dimensions.
 * Only required if kS3dSurfaceGenerator_Setup() is used for preallocation.
 *
 * @public                   @memberof kS3dSurfaceGenerator
 * @param   alg              Alg object
 * @param   height           Input/output arrays height.
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dSurfaceGenerator_SetOutputHeight(kS3dSurfaceGenerator alg, kSize height);

/**
 * Optional function to set the input output array dimensions.
 * Only required if kS3dSurfaceGenerator_Setup() is used for preallocation.
 *
 * @public                   @memberof kS3dSurfaceGenerator
 * @param   alg              Alg object
 * @param   width            Input/output arrays width.
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dSurfaceGenerator_SetOutputWidth(kS3dSurfaceGenerator alg, kSize width);

/**
 * Set the job queue for parallel processing.
 *
 * @public                   @memberof kS3dSurfaceGenerator
 * @param   alg              Alg object
 * @param   queue            Job queue for parallel processing.
 * @return                   Operation status.
 */

kVsFx(kStatus) kS3dSurfaceGenerator_SetJobQueue(kS3dSurfaceGenerator alg, kVsJobQueue queue);

/**
* Converts resampled phase maps to 3D points.
*
* @public                    @memberof kS3dSurfaceGenerator
* @param   alg               Alg object
* @param   yBegin            Sample start
* @param   yStep             Sample step
* @param   x0                Resampled phase map 0
* @param   x1                Resampled phase map 1
* @param   phaseTable        Phase table
* @param   output            An output array of 3D points
* @return                    Operation status
*/
kVsFx(kStatus) kS3dSurfaceGenerator_LookupMap(kS3dSurfaceGenerator alg, kSSize yBegin, kSize yStep, kArray2 x0, kArray2 x1, kArray1 phaseTable, kArray2 output);

/**
* Converts resampled phase maps to 3D points.
* Same as kS3dSurfaceGenerator_LookupMap() only executes in Cuda and assumes the data is allocated on device.
*
* @public                    @memberof kS3dSurfaceGenerator
* @param   alg               Alg object
* @param   yBegin            Sample start
* @param   yStep             Sample step
* @param   x0                Resampled phase map 0
* @param   x1                Resampled phase map 1
* @param   phaseTable        Phase table
* @param   output            An output array of 3D points
* @return                    Operation status
*/

#if defined (K_HAVE_CUDA)
kVsFx(kStatus) kS3dSurfaceGenerator_LookupMapCuda(kS3dSurfaceGenerator alg, kSSize yBegin, kSize yStep, kArray2 x0, kArray2 x1, kArray1 phaseTable, kArray2 output);
#else
kInlineFx(kStatus) kS3dSurfaceGenerator_LookupMapCuda(kS3dSurfaceGenerator alg, kSSize yBegin, kSize yStep, kArray2 x0, kArray2 x1, kArray1 phaseTable, kArray2 output) { return kERROR; }
#endif

/**
 * Set outlier tolerance for the merging algorithms.
 * Used for outlier removal as an average multiplier during the sampling stage.
 * Tolerance = Sqr(tol) * average
 *
 * @public              @memberof kS3dSurfaceGenerator
 * @param   alg         Alg object
 * @param   tolerance   New tolerance value
 * @return              Operation status
 */

kVsFx(kStatus) kS3dSurfaceGenerator_SetMergeOutlierTolerance(kS3dSurfaceGenerator alg, k64f tolerance);

/**
 * Get outlier tolerance for the merging algorithms.
 * Used for outlier removal as an average multiplier during the sampling stage.
 * Tolerance = Sqr(tol) * average
 *
 * @public              @memberof kS3dSurfaceGenerator
 * @param   alg         Alg object
 * @return              Outlier tolerance
 */

kVsFx(k64f) kS3dSurfaceGenerator_MergeOutlierTolerance(kS3dSurfaceGenerator alg);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dSurfaceGenerator.x.h>

#endif  /* kS3D_SURFACE_GENERATOR_H */
