#ifndef LMITECH_KVISION_S3D_STRIPEPROCESSOR_H_INCLUDED
#define LMITECH_KVISION_S3D_STRIPEPROCESSOR_H_INCLUDED

#include <kVision/S3d/kS3dCommon.h>
#include <kVision/S3d/kS3dStereoCal.h>
#include <kVision/S3d/kS3dStereoProfilerDep.h>
#include <kVision/S3d/kS3dScanInput.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kFireSync/Data/kPlot.h>

typedef kPointer kS3dStripeProcessor;

typedef struct kS3dStripeProcessorParams 
{
    kRect3d64f fov;
    kRect3d64f roi;
    
    k64f rangeSampleBegin;
    k64f rangeSampleSpan;
    k64f samplingResolution;
    k64f erosion;

    k32u sequenceCount;
    
    k32u referencePatternCount;
    k32u stripePatternCount;
    k32u phasePatternCount;
    k32u widePhasePatternCount;
    k32u widePhaseDepth;
    
    k32u phaseSampleCount;
    k32u intensitySequenceIndex;

    k32u phaseThreshold;
    k32u stripeThreshold;

    k32u roiStartStripe;
    k32u roiStripeCount;

    kBool retainImages;     
    kBool filterRanges;
    kBool clipRanges;

} kS3dStripeProcessorParams;

typedef struct kS3dStripeProcessorRoi 
{
    kRect3d64f roi;

    k32u imageWidth;
    k32u imageHeight;

    k32u roiStartStripe;
    k32u roiStripeCount;

    k64f rangeSampleBegin;
    k64f rangeSampleSpan;

} kS3dStripeProcessorRoi;

kVsFx(kStatus) kS3dStripeProcessor_Construct(kS3dStripeProcessor* processor, const kS3dStripeProcessorParams* params, kS3dStereoProfilerDep profiler);
kVsFx(kStatus) kS3dStripeProcessor_Destroy(kS3dStripeProcessor processor);

kVsFx(kStatus) kS3dStripeProcessor_UpdateRoi(kS3dStripeProcessor processor, const kS3dStripeProcessorRoi* roi);
kVsFx(kStatus) kS3dStripeProcessor_Reset(kS3dStripeProcessor processor);

kVsFx(kStatus) kS3dStripeProcessor_SetLookupType(kS3dStripeProcessor processor, kS3dStripeLookupType type);
kVsFx(kS3dStripeLookupType) kS3dStripeProcessor_LookupType(kS3dStripeProcessor processor);

kVsFx(kStatus) kS3dStripeProcessor_SetIntensityType(kS3dStripeProcessor processor, kS3dStripeIntensityType type);
kVsFx(kS3dStripeIntensityType) kS3dStripeProcessor_IntensityType(kS3dStripeProcessor processor);

kVsFx(kStatus) kS3dStripeProcessor_SetWideSmoothWindow(kS3dStripeProcessor processor, k32u window);
kVsFx(k32u) kS3dStripeProcessor_WideSmoothWindow(kS3dStripeProcessor processor);

kVsFx(kStatus) kS3dStripeProcessor_EnableWidePhase(kS3dStripeProcessor processor, kBool enable);
kVsFx(kBool) kS3dStripeProcessor_WidePhaseEnabled(kS3dStripeProcessor processor);

kVsFx(kStatus) kS3dStripeProcessor_Update(kS3dStripeProcessor processor, kSize viewIndex, kImage image);
kVsFx(kStatus) kS3dStripeProcessor_UpdateImageList(kS3dStripeProcessor processor, kImage* images0, kImage* images1, kSize count);
kVsFx(kStatus) kS3dStripeProcessor_UpdateScanInput(kS3dStripeProcessor processor, kS3dScanInput scanInput);

/**
 * @brief Computes range information based on current buffered image data.
 * @pre <kS3dStripeProcessor_Update> Must have been called N*K times, where N is the sequence length, and K is the number of sequences.
 * Additionally, if thermal expansion correction is enabled, the current controller temperature should be been provided prior to 
 * this call via <kS3dStripeProcessor_SetTemperature>.
 * @post Fills in the (k16s) ranges field. This field can be accessed via kS3dStripeProcessor_Ranges16s. The point cloud conversion
 * to k64f (mm) units is performed by <kS3dStripeProcessor_Ranges64f>
 **/
kVsFx(kStatus) kS3dStripeProcessor_UpdateRanges(kS3dStripeProcessor processor);
kVsFx(kStatus) kS3dStripeProcessor_UpdateIntensity(kS3dStripeProcessor processor);

kVsFx(kStatus) kS3dStripeProcessor_StripePlot(kS3dStripeProcessor processor, kSize viewIndex, kSize stripeFrameIndex, kPlot* plot);
kVsFx(kStatus) kS3dStripeProcessor_PhasePlot(kS3dStripeProcessor processor, kSize viewIndex, kSize phaseIndex, kPlot* plot);
kVsFx(kStatus) kS3dStripeProcessor_PhaseDeltaPlot(kS3dStripeProcessor processor, kSize viewIndex, kPlot* plot);
kVsFx(kStatus) kS3dStripeProcessor_ResampledSpotPlot(kS3dStripeProcessor processor, kSize viewIndex, kPlot* plot);
kVsFx(kStatus) kS3dStripeProcessor_ResampledProjPlot(kS3dStripeProcessor processor, kSize viewIndex, kPlot* plot);
kVsFx(kStatus) kS3dStripeProcessor_RowPlot(kS3dStripeProcessor processor, kSize viewIndex, kSize rowIndex, kPlot* plot);

kVsFx(kArray2) kS3dStripeProcessor_SampledSpots(kS3dStripeProcessor processor, kSize viewIndex);

kVsFx(kStatus) kS3dStripeProcessor_ExecutionReport(kS3dStripeProcessor processor, kString* report);

kVsFx(kArray2) kS3dStripeProcessor_Intensity(kS3dStripeProcessor processor);
kVsFx(kArray2) kS3dStripeProcessor_Ranges16s(kS3dStripeProcessor processor);
kVsFx(kStatus) kS3dStripeProcessor_Ranges64f(kS3dStripeProcessor processor, kArray2* ranges);

kVsFx(kStatus) kS3dStripeProcessor_HeightMap(kS3dStripeProcessor processor, kArray2* map);
kVsFx(kStatus) kS3dStripeProcesor_SampledSpotCoords(kS3dStripeProcessor processor, kSize viewIndex, kArray2* x, kArray2* y);

kVsFx(kStatus) kS3dStripeProcessor_ExportScan(kS3dStripeProcessor processor, const kChar* path);

kVsFx(k64f) kS3dStripeProcessor_Erosion(kS3dStripeProcessor processor);
kVsFx(kStatus) kS3dStripeProcessor_SetErosion(kS3dStripeProcessor processor, k64f erosion);

kVsFx(kSize) kS3dStripeProcessor_OutputWidth(kS3dStripeProcessor processor);
kVsFx(kSize) kS3dStripeProcessor_OutputHeight(kS3dStripeProcessor processor);

kVsFx(kSize) kS3dStripeProcessor_MemoryUsage(kS3dStripeProcessor processor);

///@brief exports the raw intensity image from the specified view. This is distinct from the resampled intensity image
kVsFx(kStatus) kS3dStripeProcessor_ViewIntensity(kS3dStripeProcessor processor, kSize viewIndex, kArray2* viewIntensity);

#endif /* #define LMITECH_KVISION_S3D_STRIPEPROCESSOR_H_INCLUDED */
