/**
* @file    kS3dStripeDecoder.h
* @brief   Declares the kS3dStripeDecoder class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_STRIPE_DECODER_H
#define kS3D_STRIPE_DECODER_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dStripeDecoder
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs Gray code decoding by thresholding a sequence of stripe images against a single reference image
*/
typedef kObject kS3dStripeDecoder;

/**
* Constructs a kS3dStripeDecoder object
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_Construct(kS3dStripeDecoder* decoder, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dStripeDecoder_Begin).However, the initialization time may be non-negligible, 
* which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dStripeDecoder_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_Setup(kS3dStripeDecoder decoder);

/**
* Initiates decoding sequence and initializes the output buffer for use.  Should be called prior to calling 
* kS3dStripeDecoder_UpdateReferenceImage, kS3dStripeDecoder_UpdateStripeImage. The output buffer handle is retained by the decoder
* object and is used to populate the stripe map. Upon completion of the sequence (kS3dStripeDecoder_IsComplete returns kTRUE)
* each byte of the output buffer is populated with a code (0... 2^stripe image count) or k8U_NULL for invalid
* 
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @param   output      Output buffer. Expected kArray2<k8u> equal in size to the image data
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_Begin(kS3dStripeDecoder decoder, kArray2 output);

/**
* Returns true when all stripe sequence images have been processed and the output has been finalized. 
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kBool) kS3dStripeDecoder_IsComplete(kS3dStripeDecoder decoder);

/**
* Update reference image used for stripe image thresholding
* 
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @param   image       Reference image  (kImage<k8u>)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_UpdateReferenceImage(kS3dStripeDecoder decoder, kImage image);

/**
* Update stripe code sequence image. The order of the images must be in descending order of bit significance. 
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @param   index       Index of the image within the sequence
* @param   image       Stripe image  (kImage<k8u>)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_UpdateStripeImage(kS3dStripeDecoder decoder, kSize index, kImage image);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder      Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_SetImageSize(kS3dStripeDecoder decoder, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder      Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_ImageSize(kS3dStripeDecoder decoder, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the number of images in the stipe sequence (excluding the reference image) as well as the bit depth of the stripe map
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @param   imageCount  Number of images in the stripe sequence (excluding the reference image)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_SetStripeImageCount(kS3dStripeDecoder decoder, kSize imageCount);

/**
* Gets the number of images in the stipe sequence (excluding the reference image) as well as the bit depth of the stripe map
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @return              Number of images in the stripe sequence (excluding the reference image)
*/
kVsFx(kSize) kS3dStripeDecoder_StripeImageCount(kS3dStripeDecoder decoder);

/**
* Sets contrast threshold for stripe decoding. The difference between stripe image pixel intensity and reference pixel intensity 
* must be higher or equal to this value in order for the decoding at that pixel to be valid
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @param   threshold   Contrast threshold for stripe decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dStripeDecoder_SetContrastThreshold(kS3dStripeDecoder decoder, kSize threshold);

/**
* Gets contrast threshold for stripe decoding
*
* @public              @memberof kS3dStripeDecoder
* @param   decoder     Decoder object
* @return              Contrast threshold for stripe decoding
*/
kVsFx(kSize) kS3dStripeDecoder_ContrastThreshold(kS3dStripeDecoder decoder);

#include <kVision/S3d/kS3dStripeDecoder.x.h>

#endif  /* kS3D_STRIPE_DECODER_H */
