/**
* @file    kS3dSpeckleProcessor.h
* @brief   Declares the kS3dSpeckleProcessor class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef KS3D_SPECKLE_PROCESSOR_H
#define KS3D_SPECKLE_PROCESSOR_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/S3d/kS3dStereoProfiler.h>

typedef enum kS3dSpeckleProcessorMatchAlgType
{
    kS3D_SPECKLE_PROCESSOR_MATCH_TYPE_SGBM,
    kS3D_SPECKLE_PROCESSOR_MATCH_TYPE_BM
} kS3dSpeckleProcessorMatchAlgType;

/**
* @class       kS3dSpeckleProcessor
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Provides end to end functionality for triangulation of pseudo-random speckle stereo. The class supports multi-resolution
*              matching using SGBM or BM. Image pre-filtering is currently limited to adaptive normalization.
*/
typedef kObject kS3dSpeckleProcessor;

/**
* Constructs a kS3dSpeckleProcessor object. The function accepts a calibration object and a profiler object (typically expensive to construct 
* and may be shared between several processors) and a number of resolution levels to be used in stereo matching.
*
* Processing of high resolution images with a wide disparity search range can be very slow (the time is proportional to image 
* resolution and disparity range). Using one or more coarse resolution levels allows the process to be sped up significantly. To 
* configure the scaling of each level use kS3dSpeckleProcessor_SetLevelScale function.
* To specify disparity range of each level kS3dSpeckleProcessor_SetDisparityRange function.
*
* Note that the processor retains the handles to the calibration object and the profiler for the duration of its existence. They cannot be 
* destroyed until after the processor is destroyed.
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Destination for the constructed object handle.
* @param   cal         Calibration object
* @param   profiler    Profiler object
* @param   levelCount  Number of resolution levels
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_Construct(kS3dSpeckleProcessor* processor, kS3dStereoCal cal, kS3dStereoProfiler profiler, k32u levelCount, kAlloc alloc);

/**
* Get the number of resolution levels used for stereo matching
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @return              Number of resolution levels used for stereo matching
*/
kVsFx(k32u) kS3dSpeckleProcessor_LevelCount(kS3dSpeckleProcessor processor);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dSpeckleProcessor_Run).
*
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dSpeckleProcessor_Setup
* after all of the parameters have been configured.
*
* Where possible partial re-initialization will be performed. This is particularly useful on DSP where updates
* to user configuration must take effect with minimal delay.
*
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @return              Operation status.
*/

kVsFx(kStatus) kS3dSpeckleProcessor_Setup(kS3dSpeckleProcessor processor);

/**
*  Executes milti-resolution texture matching algorithm on a pair of raw stereo images to produce a two-dimensional array of 3D points
*  The steps executed by the algorithm are as follows:
*
*  - Rectify full resolution images
*
*  - Perform resolution independent pre-filtering (e.g. intensity rescaling)
*
*  - Downsample rectified images for each resolution level
*
*  - For each resolution level:
*       - If not the coarsest level use the disparity output from the previous level to resample/re-rectify images to line up the disparities to zero
*       - Perform resolution specific pre-filtering (adaptive norm)
*       - Perform stereo matching using either SGBM or BM algorithms
*           If parallel processing is enabled, the images are split into strips, parts are processed independently and then the disparities are combined
*       - Perform disparity filtering (removal of outliers and low texture regions)
*       - If not the coarsest level, combine resulting disparity with the template from the previous level
*       - If not the final (most dense) layer, prepare a disparity template for the next level up (currently this is just a smooth operation)
*
*   - Triangulate finest disparity output into 3D data
* 
*  The output buffer must be allocated by the caller. Use kS3dSpeckleProcessor_OutputSize to retrieve the required dimensions. The 
*  type of the output may be kPoint3d16s (output is scaled to profiler resolution) or kPoint3d64f (output is in mm)
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   image0      Raw left input image
* @param   image1      Raw right input image
* @param   output      Output triangulated points of type kPoint3d16s or kPoint3d64f (use kS3dSpeckleProcessor_OutputSize to pre-allocate the correct size buffer)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_Run(kS3dSpeckleProcessor processor, kImage image0, kImage image1, kArray2 output);

/**
* Gets the size of the output 3D point array. The function should be called after all of the processor parameters have
* been configured and kS3dSpeckleProcessor_Setup has been issued. The dimensions returned by this function should be used to construct
* the output array.
*
* The output of this function is equivalent to the output of kS3dSpeckleProcessor_RectifiedSize for level 0
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   width       Width of the output array
* @param   height      Height of the output array
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_OutputSize(kS3dSpeckleProcessor processor, kSize* width, kSize* height);

/**
* Gets the size of the rectified images for a given resolution level. The values may be used to abstract some 
* of the parameters controlling the behaviour of each level as a fraction of the overall rectified image 
* size (e.g. kS3dSpeckleProcessor_SetFilterWindow, kS3dSpeckleProcessor_SetDisparityRegionMinSize)
*
* The output of this function for level 0 provides the dimensions of the required output buffer
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level
* @param   width       Width of the rectified images at the specified resolution level
* @param   height      Height of the rectified images at the specified resolution level
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_RectifiedSize(kS3dSpeckleProcessor processor, k32u level, kSize* width, kSize* height);

/**
* Sets the pixel resolution used in the rectification step. The value directly controls the density of the output mesh
* and should be set to roughly match the pixel resolution of the sensor at mid range. 
*
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   resolution  Rectification resolution (mm)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetRectificationResolution(kS3dSpeckleProcessor processor, k64f resolution);

/**
* Get the rectification resolution
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @return              Rectification resolution (mm)
*/
kVsFx(k64f) kS3dSpeckleProcessor_RectificationResolution(kS3dSpeckleProcessor processor);

/**
* Set the downsampling scale (X and Y) for a resolution level. The value is set relative to the full resolution and 
* in the current implementation must be a power of two. For example, setting level 1 resolution to 4 will result in
* second finest level to use 1/4 resolution of the top level. The user must ensure that the values increase for increasing
* level values. Resolution of the top level (level 0) must be fixed at 1. 
*
* Future versions of the algorithm may support first level resolution to be reduced relative to rectification resolution, but
* currently there is no use case for that.
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level index (zero corresponds to the full resolution level and must not be modified)
* @param   scale       Downsampling scale for a resolution level (e.g. 4 means 1/4 resolution). 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetLevelScale(kS3dSpeckleProcessor processor, k32u level, k32u scale);

/**
* Get the downsampling scale for a resolution level
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level
* @return              Downsampling scale for a resolution level
*/
kVsFx(k32u) kS3dSpeckleProcessor_LevelScale(kS3dSpeckleProcessor processor, k32u level);

/**
* Set maximum number of parallel operations to be performed during processing. Where possible,
* long processing steps will be split and executed by several threads simultaneously. ons that are parallelized
* include any pre-processing of camera images (two images are pre-processed mostly independently) and block matching (images are
* explicitly split into the specified number of strips). 
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   count       Maximum number of parallel operations 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetParallelCount(kS3dSpeckleProcessor processor, k32u count);

/**
* Get the maximum number of parallel operations 
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @return              Maximum number of parallel operations 
*/
kVsFx(k32u) kS3dSpeckleProcessor_ParallelCount(kS3dSpeckleProcessor processor);

/**
* Set the block matching algorithm type for processing at a given resolution level. For descriptions of the different methods 
* see documentation for kS3dSpeckleMatchAlg and deriving classes
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   type         Type of the algorithm to use
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetMatchType(kS3dSpeckleProcessor processor, k32u level, kS3dSpeckleProcessorMatchAlgType type);

/**
* Get the block matching algorithm type used for processing a given resolution level.
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Block matching algorithm type used for processing a given resolution level.
*/
kVsFx(kS3dSpeckleProcessorMatchAlgType) kS3dSpeckleProcessor_MatchType(kS3dSpeckleProcessor processor, k32u level);

/**
* Set block matching window size for a given resolution level. See kS3dSpeckleMatchAlg_SetMatchWindow for details
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   window       Block matching window (number of rectified pixels)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetMatchWindow(kS3dSpeckleProcessor processor, k32u level, k32u window);

/**
* Get the block matching window used for processing a given resolution level.
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Block matching window (number of rectified pixels)
*/
kVsFx(k32u) kS3dSpeckleProcessor_MatchWindow(kS3dSpeckleProcessor processor, k32u level);

/**
* Set block matching disparity range for a given resolution level. Note that depending on the choice of the algorithm
* the actual disparity range may be rounded up to the granularity allowed by the algorithm. See kS3dSpeckleMatchAlg_SetDisparityRange
* for further details
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   range        Block matching disparity range (number of rectified pixels)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetDisparityRange(kS3dSpeckleProcessor processor, k32u level, k32u range);

/**
* Get the block matching disparity range used for processing a given resolution level.
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Block matching disparity range (number of rectified pixels)
*/
kVsFx(k32u) kS3dSpeckleProcessor_DisparityRange(kS3dSpeckleProcessor processor, k32u level);

/**
* Set margin in percentage (0 to 1) by which the best (minimum) computed cost function value should "win" the second
* best value to consider the found match correct. Normally, a value within the 0.05-0.15 range is sufficient
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   ratio        Uniqueness ratio (0 to 1)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetUniquenessRatio(kS3dSpeckleProcessor processor, k32u level, k64f ratio);

/**
* Get the block matching uniqueness ratio
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Block matching uniqueness ratio (0 to 1)
*/
kVsFx(k64f) kS3dSpeckleProcessor_UniquenessRatio(kS3dSpeckleProcessor processor, k32u level);

/**
* Set maximum allowed difference (in integer pixel units) in the left-right disparity check. Set it to k32U_NULL
* to disable the check
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   threshold    Left-right disparity check threshold (number of rectified pixels)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetDisparityLrThreshold(kS3dSpeckleProcessor processor, k32u level, k32u threshold);

/**
* Get the left-right disparity check threshold
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Left-right disparity check threshold (number of rectified pixels)
*/
kVsFx(k32u) kS3dSpeckleProcessor_DisparityLrThreshold(kS3dSpeckleProcessor processor, k32u level);

/**
* Set the weight of the adjacency cost used in selecting a particular disparity path (SGBM). See 
* kS3dSpeckleMatchAlg_SetContinuityWeight for details.
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   weight       Weight of adjacency within a disparity path
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetContinuityWeight(kS3dSpeckleProcessor processor, k32u level, k64f weight);

/**
* Get the weight of adjacency within a disparity path (SGBM)
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Weight of adjacency within a disparity path
*/
kVsFx(k64f) kS3dSpeckleProcessor_ContinuityWeight(kS3dSpeckleProcessor processor, k32u level);

/**
* Set the weight of the penalty introduced in a correspondence path due to a change of more than one pixel (SGBM)
* kS3dSpeckleMatchAlg_SetOcclusionWeight for details.
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   weight       Weight of occlusion penalty in a disparity path
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetOcclusionWeight(kS3dSpeckleProcessor processor, k32u level, k64f weight);

/**
* Get the weight of occlusion penalty in a disparity path (SGBM)
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Weight of occlusion penalty in a disparity path
*/
kVsFx(k64f) kS3dSpeckleProcessor_OcclusionWeight(kS3dSpeckleProcessor processor, k32u level);

/**
* Set intensity truncation value. See kS3dSpeckleMatchAlg_SetFilterCap for details.
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   filterCap    Image processor truncation value
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetFilterCap(kS3dSpeckleProcessor processor, k32u level, k32u filterCap);

/**
*  Get the image processor truncation value. 
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Image processor truncation value
*/
kVsFx(k32u) kS3dSpeckleProcessor_FilterCap(kS3dSpeckleProcessor processor, k32u level);

/**
* Set the size of the window used to normalize the intensities of the rectified images prior to matching. 
* See kS3dSpecklePrefilter_SetWindow for details. 
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   window       Image filtering window size (rectified pixels)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetFilterWindow(kS3dSpeckleProcessor processor, k32u level, k32u window);

/**
* Get image filtering window size
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level.
* @return              Image filtering window size (number of rectified pixels)
*/

kVsFx(k32u) kS3dSpeckleProcessor_FilterWindow(kS3dSpeckleProcessor processor, k32u level);

/**
* Set texture threshold for the pre-filtering stage. The value defines minimum amount of contrast within the filtering 
* window to consider a patch valid. Intensity range lower than that results in the pixel centered at that window to 
* be invalidated.
*
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   threshold    Texture threshold
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetTextureThreshold(kS3dSpeckleProcessor processor, k32u level, k32u threshold);

/**
*  Get the texture threshold for the pre-filtering stage. 
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @return              Texture threshold value
*/
kVsFx(k32u) kS3dSpeckleProcessor_TextureThreshold(kS3dSpeckleProcessor processor, k32u level);

/**
* Configures minimum disparity region used in the disparity outlier removal step. 
* The value controls the minimum number of values within a continuous region in a disparity map 
* for that region to be considered valid. 
*
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   size         Minimium number of disparity values in a valid continuous region
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetDisparityRegionMinSize(kS3dSpeckleProcessor processor, k32u level, k32u size);

/**
*  Get the minimium number of disparity values in a valid continuous region
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @return              Minimium number of disparity values in a valid continuous region
*/
kVsFx(k32u) kS3dSpeckleProcessor_DisparityRegionMinSize(kS3dSpeckleProcessor processor, k32u level);

/**
* Configures the maximum step between adjacent disparity values to be still considered a part of a continuos region. 
*
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   threshold    Maximum difference in adjacent disparity values to be considered part of the same region (number of rectified pixels)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetDisparityRegionStepThreshold(kS3dSpeckleProcessor processor, k32u level, k64f threshold);

/**
*  Get the maximum step between adjacent disparity values to be still considered a part of a continuos region. 
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @return              Maximum difference in adjacent disparity values to be considered part of the same region (number of rectified pixels)
*/
kVsFx(k64f) kS3dSpeckleProcessor_DisparityRegionStepThreshold(kS3dSpeckleProcessor processor, k32u level);

/**
* Configures the size of the smoothing filter applied to the disparity at each resolution level prior to it being used as a template
* for higher resolution steps. As texture matching techniques are prone to outliers and pixel lock-in (bias toward specific pixel values)
* smoothing is required to ensure that these errors do not propagate to subsequent levels. 
*
* Note that smoothing can be applied to level zero (highest resolution) to smooth the final mesh. 
*
* @public               @memberof kS3dSpeckleProcessor
* @param   processor    Processor object
* @param   level        Resolution level. Zero corresponds to the full resolution level. Specify k32_NULL to apply to all resolutions
* @param   window       Disparity template smoothing window (number of rectified pixels)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleProcessor_SetDisparityTemplateSmoothWindow(kS3dSpeckleProcessor processor, k32u level, k32u window);

/**
*  Get the disparity template smoothing window
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @return              Disparity template smoothing window (number of rectified pixels)
*/
kVsFx(k32u) kS3dSpeckleProcessor_DisparityTemplateSmoothWindow(kS3dSpeckleProcessor processor, k32u level);

/**
*  Get disparity output for a given resolution level
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @return              Disparity output for a given resolution level
*/
kVsFx(kArray2) kS3dSpeckleProcessor_Disparity(kS3dSpeckleProcessor processor, k32u level);

/**
*  Get disparity template generated by a given resolution level to be used in the next resolution level. For full resolution
*  level (0) the output is null.
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @return              Disparity template generated by a given resolution level 
*/
kVsFx(kArray2) kS3dSpeckleProcessor_DisparityTemplate(kS3dSpeckleProcessor processor, k32u level);

/**
*  Get rectified image at a given resolution level
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @param   viewIndex   View index (0 or 1)
* @return              Rectified image at a given resolution level
*/
kVsFx(kImage) kS3dSpeckleProcessor_RectifiedImage(kS3dSpeckleProcessor processor, k32u level, k32u viewIndex);

/**
*  Get rectified image after remapping with the disparity template produced at previous resolution level
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @param   viewIndex   View index (0 or 1)
* @return              Rectified image after remapping with the disparity template produced at previous resolution level
*/
kVsFx(kImage) kS3dSpeckleProcessor_RemappedRectifiedImage(kS3dSpeckleProcessor processor, k32u level, k32u viewIndex);

/**
*  Get pre-filtered image at a given resolution level. The image serves as a direct input to the texture matching logic
*
* @public              @memberof kS3dSpeckleProcessor
* @param   processor   Processor object
* @param   level       Resolution level. Zero corresponds to the full resolution level
* @param   viewIndex   View index (0 or 1)
* @return              Pre-filtered image at a given resolution level
*/
kVsFx(kImage) kS3dSpeckleProcessor_FilteredImage(kS3dSpeckleProcessor processor, k32u level, k32u viewIndex);

#include <kVision/S3d/kS3dSpeckleProcessor.x.h>

#endif  /* KS3D_SPECKLE_PROCESSOR_H */
