/**
 * @file    kS3dPatternSequence.h
 * @brief   Declares the kS3dPatternSequence class.
 *
 * @internal
 * Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kS3D_PATTERN_SEQUENCE_H
#define kS3D_PATTERN_SEQUENCE_H

#include <kVision/S3d/kS3dEncodingMath.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArrayList.h>

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

/**
 * @class       kS3dPatternSequence
 * @extends     kObject
 * @ingroup     kVision-S3d
 * @brief       A class for generating various projection patterns for various decoding schemes
 */
typedef kObject kS3dPatternSequence;

/**
 * Constructs a kS3dPatternSequence object.
 *
 * @public              @memberof kS3dPatternSequence
 * @param   pattern     Destination for the constructed object handle.
 * @param   width       Pattern width to generate.
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status.
 */
kVsFx(kStatus) kS3dPatternSequence_Construct(kS3dPatternSequence* pattern, kSize width, kAlloc allocator);

//////////////////////////////////////////////////////////////////////////
// Setters & Getters
//////////////////////////////////////////////////////////////////////////

/**
* Gets the width of the pattern to be generated.
*
* @public              @memberof kS3dPatternSequence
* @param   pattern     Pattern object
* @return              Number of pixels (width) in the pattern
*/
kVsFx(kSize) kS3dPatternSequence_Width(kS3dPatternSequence pattern);

/**
* Gets the uniformity coefficient (defaults to 1).
* Used for projector fall off compensation.
* Intensity scale of the center of the pattern relative to the lateral extents. 
* Quadratically interpolated across the full pattern width.
*
* @public              @memberof kS3dPatternSequence
* @param   pattern     Pattern object
* @return              Uniformity value.
*/
kVsFx(k64f) kS3dPatternSequence_Uniformity(kS3dPatternSequence pattern);

/**
* Sets the uniformity coefficient.
*
* @public              @memberof kS3dPatternSequence
* @param   decoder     Decoder object
* @param   uniformity  Uniformity coefficient to use
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPatternSequence_SetUniformity(kS3dPatternSequence pattern, k64f uniformity);

/**
* Gets the max pattern intensity (default is 1)
*
* @public              @memberof kS3dPatternSequence
* @param   pattern     Pattern object
* @return              Max intensity value
*/
kVsFx(k64f) kS3dPatternSequence_Intensity(kS3dPatternSequence pattern);

/**
* Sets the max pattern intensity to be used.
*
* @public              @memberof kS3dPatternSequence
* @param   decoder     Decoder object
* @param   intensity   Intensity to be used
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPatternSequence_SetIntensity(kS3dPatternSequence pattern, k64f intensity);

/**
* Gets the number of images on the sequence.
*
* @public              @memberof kS3dPatternSequence
* @param   pattern     Pattern object
* @return              Total number of images
*/
kVsFx(kSize) kS3dPatternSequence_ImageCount(kS3dPatternSequence pattern);

//////////////////////////////////////////////////////////////////////////
// Operations 
//////////////////////////////////////////////////////////////////////////

/**
* Generates pattern sequence for stripe scanning.
* It is safe to call Setup several times with different parameters as well 
* as calling different type of Setup (kS3dPatternSequence_SetupEmbeddedPhase)
*
* Sample code: https://docs.google.com/document/d/1rFWbk50PfrkcAe5IHGnkNDYgrNdNGYKUPKwElGUDGzE/edit#bookmark=id.5ydh055h23j2
*
* @public                 @memberof kS3dPatternSequence
* @param   pattern        Pattern object
* @param   referenceCount Number of reference images to use
* @param   stripeCount    Number of stripe (binary) images to use
* @param   phaseCount     Number of phase images to use
* @param   period         Phase period (finest stripe width)
* @return                 Operation status.
*/
kVsFx(kStatus) kS3dPatternSequence_SetupStripePhase(kS3dPatternSequence pattern, kSize referenceCount, kSize stripeCount, kSize phaseCount, k64f period);

/**
* Generates pattern sequence for embedded phase scanning. 
*
* Epic:        https://jira.lmi-tech.com:8443/browse/SSW-463
* Sample code: https://docs.google.com/document/d/1rFWbk50PfrkcAe5IHGnkNDYgrNdNGYKUPKwElGUDGzE/edit#bookmark=id.udj2ggrh8l3e
*
* @public                 @memberof kS3dPatternSequence
* @param   pattern        Pattern object
* @param   periods        Array of pattern periods 
* @param   stepCounts     Array specifying the number of images per period to use (same size as "periods" and == periodCount).
* @param   periodCount    Size of the "periods" and "stepCounts" arrays.
* @return                 Operation status.
*/
kVsFx(kStatus) kS3dPatternSequence_SetupEmbeddedPhase(kS3dPatternSequence pattern, const k64f* periods, const kSize* stepCounts, kSize periodCount);

/**
* Renders prepared pattern sequence into an output array.
*
* @public                 @memberof kS3dPatternSequence
* @param   pattern        Pattern object
* @param   output         Array2<k8u>(imageCount, width) one row for each image.
* @return                 Operation status.
*/
kVsFx(kStatus) kS3dPatternSequence_Render(kS3dPatternSequence pattern, kArray2* output);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dPatternSequence.x.h>

#endif  /* kS3D_PATTERN_SEQUENCE_H */
