/**
 * @file    kS3dEmbeddedPhaseDecoderInt.h
 * @brief   Declares the ImageFilter class.
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kS3D_EMBEDDED_PHASE_DECODER_INT_H
#define kS3D_EMBEDDED_PHASE_DECODER_INT_H

#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kImage.h>
#include <kVision/S3d/kS3dEncodingMath.h>
#include <kVision/S3d/kS3dEmbeddedPhaseDecoder.h>
#include <kVision/Vs/kVsJobQueue.h>

kBeginHeader()

//////////////////////////////////////////////////////////////////////////
// Embedded Phase Shifting
// Epic:       https://jira.lmi-tech.com:8443/browse/SSW-463
// Doc:        https://docs.google.com/document/d/1rFWbk50PfrkcAe5IHGnkNDYgrNdNGYKUPKwElGUDGzE/edit#
// Paper:      http://mesh.brown.edu/eps/
// Paper SVN:  https://svn1.lmi-tech.com/sensors/Gocator/Documents/3xxx/Papers/Moreno_Embedded_Phase_Shifting_2015_CVPR_paper.pdf
//////////////////////////////////////////////////////////////////////////

/**
*
* @class       kS3dEmbeddedPhaseDecoderInt
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs embedded phase shift sequence decoding
*/
typedef kObject kS3dEmbeddedPhaseDecoderInt;

/**
* Constructs a kS3dEmbeddedPhaseDecoderInt object
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/

kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_Construct(kS3dEmbeddedPhaseDecoderInt* decoder, kAlloc alloc);

//////////////////////////////////////////////////////////////////////////
// Setters & Getters
//////////////////////////////////////////////////////////////////////////

/**
* Sets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                     @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder            Decoder object
* @param   coefficients       Array of coefficients describing embedded periods (k64f).
* @param   stepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   coefficientCount   Size of the "coefficients" and "stepCounts" arrays.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_SetSequenceInfo(kS3dEmbeddedPhaseDecoderInt decoder, const k64f* coefficients, const kSize* stepCounts, kSize coefficientCount);

/**
* Gets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                        @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder               Decoder object
* @param   outCoefficients       Array of coefficients describing embedded periods (k64f).
* @param   outStepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   outCoefficientCount   Number of values in the "coefficients" and "stepCounts" arrays returned
* @param   capacity              Size of "coefficients" and "stepCounts" arrays passed in (capacity should be >= coefficientCount).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_SequenceInfo(kS3dEmbeddedPhaseDecoderInt decoder, k64f* outCoefficients, kSize* outStepCounts, kSize* outCoefficientCount, kSize capacity);

/**
* Gets the number of images in the whole sequence aka total number of images used.
* Sum of "stepCounts" values from the kS3dEmbeddedPhaseDecoderInt_SetSequenceInfo()
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @return              Number of images in the sequence
*/
kVsFx(kSize) kS3dEmbeddedPhaseDecoderInt_SequenceLength(kS3dEmbeddedPhaseDecoderInt decoder);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_SetImageSize(kS3dEmbeddedPhaseDecoderInt decoder, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of the images used.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_ImageSize(kS3dEmbeddedPhaseDecoderInt decoder, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the scale of the phase. The resulting phase output is scaled such that [0, 1] phase values are mapped to [0, phaseScale]
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   phaseScale  Phase scale
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_SetPhasePeriod(kS3dEmbeddedPhaseDecoderInt decoder, kSize phaseScale);

/**
* Sets contrast threshold for phase decoding. The range of intensities for a given pixel location among all phase
* images must exceed this threshold value in order for the phase decoding to produce a valid result.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @param   threshold   Contrast threshold for phase decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_SetContrastThreshold(kS3dEmbeddedPhaseDecoderInt decoder, kSize threshold);

/**
* Gets contrast threshold for phase decoding
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @return              Contrast threshold for phase decoding
*/
kVsFx(kSize) kS3dEmbeddedPhaseDecoderInt_ContrastThreshold(kS3dEmbeddedPhaseDecoderInt decoder);

/**
* Sets the image orientation.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   processor   Processor object
* @param   orientation Orientation
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_SetOrientation(kS3dEmbeddedPhaseDecoderInt decoder, kS3dEmbeddedPhaseDecoderOrientation orientation);

/**
* Gets the image orientation.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @return              Orientation.
*/
kVsFx(kS3dEmbeddedPhaseDecoderOrientation) kS3dEmbeddedPhaseDecoderInt_Orientation(kS3dEmbeddedPhaseDecoderInt decoder);

//////////////////////////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////////////////////////

/**
* Updates working buffers to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution of the algorithm (kS3dEmbeddedPhaseDecoderInt_Begin).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dEmbeddedPhaseDecoderInt_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_Setup(kS3dEmbeddedPhaseDecoderInt decoder);

/**
* Initiates decoding sequence and initializes the output buffer for use.
* Should be called prior to calling kS3dEmbeddedPhaseDecoderInt_Update.
* The output buffer handle is retained by the decoder object and is used to populate the phase map and the base map.
* Upon completion of the sequence (kS3dEmbeddedPhaseDecoderInt_IsComplete returns kTRUE)
*
* Each entry in the output phase map is populated with the decoded phase value contrast and intensity of the phase map.
* Each byte of the output base map is populated with a repetition code or k8U_NULL for invalid.
* Pixels which were not decoded are identified by having the phase value of the entry set to k16S_NULL
* Both maps are synchronized in terms of NULL values (NULLS appear in the same places)
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @param   output      Output buffers: Expected phase map kArray2<kPhasePixel> and base map kArray2<k8u> equal in size to the image data
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_Begin(kS3dEmbeddedPhaseDecoderInt decoder, kArray2 outputBaseMap, kArray2 outputPhaseMap);

/**
* Sets input/output arrays for tracking and replacing all the intermediate alg states.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @param   input       Input alg states kArrayList<kArray2<k32s>> replacing phaseArray<>
* @param   output      Output alg states kArrayList<kArray2<k32s>> tracking whats inside phaseArray<>
* @return              Operation status.
 */

kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_SetPhaseArrayStates(kS3dEmbeddedPhaseDecoderInt decoder, kArrayList inputPhaseArrayState, kArrayList outputPhaseArrayState);

/**
* Update phase shift sequence image.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @param   image       Phase image (kImage<k8u>)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_Update(kS3dEmbeddedPhaseDecoderInt decoder, kImage image);

/**
* Returns true when all phase sequence images have been processed and the output has been finalized.
*
* @public              @memberof kS3dEmbeddedPhaseDecoderInt
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kBool) kS3dEmbeddedPhaseDecoderInt_IsComplete(kS3dEmbeddedPhaseDecoderInt decoder);



//kVsFx(kStatus) kS3dEmbeddedPhaseDecoderInt_Process(kS3dEmbeddedPhaseDecoderInt decoder, kArray2 inputImages, kArray2 outputBaseMap, kArray2 outputPhaseMap);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kEndHeader()

#include <kVision/S3d/kS3dEmbeddedPhaseDecoderInt.x.h>

#endif  /* kS3D_EMBEDDED_PHASE_DECODER_INT_H */
