/**
 * @file    kS3dEmbeddedPhaseDecoder.h
 * @brief   Declares the ImageFilter class.
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kS3D_EMBEDDED_PHASE_DECODER_H
#define kS3D_EMBEDDED_PHASE_DECODER_H

#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kImage.h>
#include <kVision/S3d/kS3dEncodingMath.h>
#include <kVision/Vs/kVsJobQueue.h>

//////////////////////////////////////////////////////////////////////////
// Embedded Phase Shifting
// Epic:       https://jira.lmi-tech.com:8443/browse/SSW-463
// Doc:        https://docs.google.com/document/d/1rFWbk50PfrkcAe5IHGnkNDYgrNdNGYKUPKwElGUDGzE/edit#
// Paper:      http://mesh.brown.edu/eps/
// Paper SVN:  https://svn1.lmi-tech.com/sensors/Gocator/Documents/3xxx/Papers/Moreno_Embedded_Phase_Shifting_2015_CVPR_paper.pdf
//////////////////////////////////////////////////////////////////////////

#define kS3D_EPD_CONTRAST_THRESHOLD     (10)     // Default contrast (Max - Min)
#define kS3D_EPD_FILTER_WINDOW          (15)     // Smooth window size

#define kS3D_EPD_ATAN_LUT_SIZE          (1 << 16)

#define kS3D_EPD_SUB_BASE_OFFSET        (0.1f)   // Zero offset for base subtraction 

#define kS3D_EPD_NULL_VARIANCE          (0.001f) // 0.1% of max var (average var == 0.00018)
#define kS3D_EPD_ZERO_VARIANCE_GUARD    (1e-6f)  // Avoid zero division 1/var

// values used to convert float shiftMatrix to int (moved here due to: kS3dEmbeddedPhaseDecoder_ConstructShiftMatrix32s())

#define kS3D_EPD_INT_PHASE_MATRIX_SHIFT_UP          (20)                                                // Used to be kS3D_PHASE_SINE_LUT_SHIFT
#define kS3D_EPD_INT_PHASE_MATRIX_SHIFT_DOWN        (kMax_(0, kS3D_EPD_INT_PHASE_MATRIX_SHIFT_UP - 18)) // See kS3dEmbeddedPhaseDecoderInt_MatrixMul_32s_8u() for calc

typedef enum
{
    kS3D_EPD_ORIENTATION_COLUMNS_INCREASING = 0,
    kS3D_EPD_ORIENTATION_COLUMNS_DECREASING,
    kS3D_EPD_ORIENTATION_ROWS_INCREASING,
    kS3D_EPD_ORIENTATION_ROWS_DECREASING
} kS3dEmbeddedPhaseDecoderOrientation;

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

/**
* 
* @class       kS3dEmbeddedPhaseDecoder
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs embedded phase shift sequence decoding
*/
typedef kObject kS3dEmbeddedPhaseDecoder;

/**
* Constructs a kS3dEmbeddedPhaseDecoder object
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/

kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_Construct(kS3dEmbeddedPhaseDecoder* decoder, kAlloc alloc);

//////////////////////////////////////////////////////////////////////////
// Setters & Getters
//////////////////////////////////////////////////////////////////////////

/**
* Sets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                     @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder            Decoder object
* @param   coefficients       Array of coefficients describing embedded periods (k64f).
* @param   stepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   coefficientCount   Size of the "coefficients" and "stepCounts" arrays.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SetSequenceInfo(kS3dEmbeddedPhaseDecoder decoder, const k64f* coefficients, const kSize* stepCounts, kSize coefficientCount);

/**
* Gets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                        @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder               Decoder object
* @param   outCoefficients       Array of coefficients describing embedded periods (k64f).
* @param   outStepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   outCoefficientCount   Number of values in the "coefficients" and "stepCounts" arrays returned
* @param   capacity              Size of "coefficients" and "stepCounts" arrays passed in (capacity should be >= coefficientCount).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SequenceInfo(kS3dEmbeddedPhaseDecoder decoder, k64f* outCoefficients, kSize* outStepCounts, kSize* outCoefficientCount, kSize capacity);

/**
* Gets the number of images in the whole sequence aka total number of images used.
* Sum of "stepCounts" values from the kS3dEmbeddedPhaseDecoder_SetSequenceInfo()
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @return              Number of images in the sequence 
*/
kVsFx(kSize) kS3dEmbeddedPhaseDecoder_SequenceLength(kS3dEmbeddedPhaseDecoder decoder);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SetImageSize(kS3dEmbeddedPhaseDecoder decoder, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of the images used.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_ImageSize(kS3dEmbeddedPhaseDecoder decoder, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the scale of the phase. The resulting phase output is scaled such that [0, 2Pi] phase values are mapped to [0, phaseScale]
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   phaseScale  Phase scale
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SetPhasePeriod(kS3dEmbeddedPhaseDecoder decoder, kSize phaseScale);

/**
* Sets contrast threshold for phase decoding. The range of intensities for a given pixel location among all phase
* images must exceed this threshold value in order for the phase decoding to produce a valid result.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @param   threshold   Contrast threshold for phase decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SetContrastThreshold(kS3dEmbeddedPhaseDecoder decoder, kSize threshold);

/**
* Gets contrast threshold for phase decoding
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @return              Contrast threshold for phase decoding
*/
kVsFx(kSize) kS3dEmbeddedPhaseDecoder_ContrastThreshold(kS3dEmbeddedPhaseDecoder decoder);

/**
* Enables multithreading.  Passes job queue in.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @param   queue       The job queue.
* @return              Operation status
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SetJobQueue(kS3dEmbeddedPhaseDecoder decoder, kVsJobQueue queue);

/**
* Sets the image orientation.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   processor   Processor object
* @param   orientation Orientation
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SetOrientation(kS3dEmbeddedPhaseDecoder decoder, kS3dEmbeddedPhaseDecoderOrientation orientation);

/**
* Gets the image orientation.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @return              Orientation.
*/
kVsFx(kS3dEmbeddedPhaseDecoderOrientation) kS3dEmbeddedPhaseDecoder_Orientation(kS3dEmbeddedPhaseDecoder decoder);

//////////////////////////////////////////////////////////////////////////
// Operations 
//////////////////////////////////////////////////////////////////////////

/**
* Updates working buffers to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution of the algorithm (kS3dEmbeddedPhaseDecoder_Begin).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dEmbeddedPhaseDecoder_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_Setup(kS3dEmbeddedPhaseDecoder decoder);

/**
* Initiates decoding sequence and initializes the output buffer for use.
* Should be called prior to calling kS3dEmbeddedPhaseDecoder_Update.
* The output buffer handle is retained by the decoder object and is used to populate the phase map and the base map.
* Upon completion of the sequence (kS3dEmbeddedPhaseDecoder_IsComplete returns kTRUE)
*
* Each entry in the output phase map is populated with the decoded phase value contrast and intensity of the phase map.
* Each byte of the output base map is populated with a repetition code or k8U_NULL for invalid.
* Pixels which were not decoded are identified by having the phase value of the entry set to k16S_NULL
* Both maps are synchronized in terms of NULL values (NULLS appear in the same places)
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @param   output      Output buffers: Expected phase map kArray2<kPhasePixel> and base map kArray2<k8u> equal in size to the image data
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_Begin(kS3dEmbeddedPhaseDecoder decoder, kArray2 outputBaseMap, kArray2 outputPhaseMap);

 /**
 * Sets input/output arrays for tracking and replacing all the intermediate alg states.
 *
 * @public              @memberof kS3dEmbeddedPhaseDecoder
 * @param   decoder     Decoder object
 * @param   input       Input alg states kArrayList<kArray2<k32s>> replacing phaseArray<>
 * @param   output      Output alg states kArrayList<kArray2<k32s>> tracking whats inside phaseArray<>
 * @return              Operation status.
  */

kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_SetPhaseArrayStates(kS3dEmbeddedPhaseDecoder decoder, kArrayList inputPhaseArrayState, kArrayList outputPhaseArrayState);

/**
* Update phase shift sequence image.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @param   image       Phase image (kImage<k8u>)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_Update(kS3dEmbeddedPhaseDecoder decoder, kImage image);

/**
* Returns true when all phase sequence images have been processed and the output has been finalized.
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kBool) kS3dEmbeddedPhaseDecoder_IsComplete(kS3dEmbeddedPhaseDecoder decoder);


/**
* Outputs the shift matrix used for phase decoding. (for PL use)
*
* Using paper notations here
* M*U = R
* Mt*M*U = Mt*R
* U = inv(Mt*M)*Mt*R
* A = inv(Mt*M)*Mt
* U = A*R
*
* R - intensities vector, no variable
* A - shiftMatrix mangled as M is a true shift matrix
* U - phaseAccumArray
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   out         kArray2<k64f> shift matrix.
* @param   periodCount Number of frequencies (2,3, etc)
* @param   imageCount  Total number of images
* @param   stepCounts  kArray1<kSize> images per period
* @param   alloc       Allocator
* @return              Operation status.
*/
kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_ConstructShiftMatrix64f(kArray2 *outShiftMatrix, kSize periodCount, kSize imageCount, kArray1 stepCounts, kAlloc alloc);

/**
* Outputs the shift matrix used for phase decoding. (for PL use)
* See: kS3dEmbeddedPhaseDecoder_ConstructShiftMatrix64f()
*
* @public              @memberof kS3dEmbeddedPhaseDecoder
* @param   out         kArray2<k32s> shift matrix.
* @param   periodCount Number of frequencies (2,3, etc)
* @param   imageCount  Total number of images
* @param   stepCounts  kArray1<kSize> images per period
* @param   alloc       Allocator
* @return              Operation status.
*/

kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_ConstructShiftMatrix32s(kArray2 *outShiftMatrix, kSize periodCount, kSize imageCount, kArray1 stepCounts, kAlloc alloc);



//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/S3d/kS3dEmbeddedPhaseDecoder.x.h>

#endif  /* kS3D_EMBEDDED_PHASE_DECODER_H */
