/** 
 * @file    kMp3dTracheidAlg.h
 * @brief   Declares the kMp3dTracheidAlg class. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_VISION_MP3D_TRACHEID_ALG_H
#define K_VISION_MP3D_TRACHEID_ALG_H

#include <kVision/Common/kVision.h>
#include <kVision/Mp3d/kMp3dCommon.h>
#include <kVision/Mp3d/kMp3dSensorCal.h>
#include <kFireSync/Data/kSpot.h>

#define kMP3D_TRACHEID_MAX_VIEW_COUNT                             (4)
#define kMP3D_TRACHEID_ALG_DEFAULT_TRACHEID_THRESHOLD             (8)

/**
 * @class       kMp3dTracheidAlg
 * @extends     kObject
 * @ingroup     kVision-Mp3d
 * @brief       Calculates image moments for each spot in an 8-bit image or a compressed binarized image. 
 */
typedef kObject kMp3dTracheidAlg;

/** 
 * Constructs a kMp3dTracheidAlg object.
 *
 * @public                  @memberof kMp3dTracheidAlg
 * @param   tracheid        Destination for the constructed object handle.
 * @param   sensorCal       Profile calibration file. This must live for the duration of kMp3dTracheidAlg.
 * @param   tracheidWindow  Maximum window around each spot centre for calculating moments. Suggested value: 19
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status. 
 */
kVsFx(kStatus) kMp3dTracheidAlg_Construct(kMp3dTracheidAlg* tracheid, kMp3dSensorCal sensorCal, kSize tracheidWindow, kAlloc allocator);

/**
* Updates run to cal window offset information based on the current runtime window settings.
* Refresh must be called after the runtime windows for all cameras are updated via kMp3dTracheidAlg_SetRunWindow.
*
* @public              @memberof kMp3dTracheidAlg
* @param   tracheid    Destination for the constructed object handle.
* @return              Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_Refresh(kMp3dTracheidAlg tracheid);

/**
* Runs tracheid algorithm to produce image moments from a binarized/compressed image.
*
* @public               @memberof kMp3dTracheidAlg
* @param   tracheid     Destination for the constructed object handle.
* @param   binData      Binarize input data.
* @param   binHeight    Height of binarize data.
* @param   binWidth     Width of binarize data.
* @param   spots        Buffer of sorted spot CGs corresponding to image. Sorted spots must be obtained from output of kMp3dProfiler_Run().
* @param   spotCount    Count of elements in spots buffer. Must be equal to kMp3dSensorCal_ViewLightCount * kMp3dSensorCal_LightSpotCount.         
* @param   viewIndex    Camera index.
* @param   moments      2D array for image moment output. Type: 32s. Size: inCount by kMP3D_TRACHEID_TYPE_MOMENT_COUNT.
* @return               Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_Run(kMp3dTracheidAlg tracheid, const k8u* binData, kSize binHeight, kSize binWidth, const kSpot2* spots, kSize spotCount, kSize viewIndex, kArray2 moments);

/**
* Runs tracheid algorithm to produce image moments from an uncompressed image.
*
* @public               @memberof kMp3dTracheidAlg
* @param   tracheid     Destination for the constructed object handle.
* @param   image        Image containing spots (image type k8u).
* @param   spots        Buffer of sorted spot CGs corresponding to image. Sorted spots must be obtained from output of kMp3dProfiler_Run().
* @param   spotCount    Count of elements in spots buffer. Must be equal to kMp3dSensorCal_ViewLightCount * kMp3dSensorCal_LightSpotCount.
* @param   viewIndex    Camera index.
* @param   moments      2D array for image moment output. Type: 32s. Size: inCount by kMP3D_TRACHEID_TYPE_MOMENT_COUNT.
* @return               Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_RunRaw(kMp3dTracheidAlg tracheid, kImage image, const kSpot2* spots, kSize spotCount, kSize viewIndex, kArray2 moments);

/**
* Calculates ellipse fit from 6 image moments and returns the area ,angle, scatter ratio, and minor and major axes.
*
* @public               @memberof kMp3dTracheidAlg
* @param   moments      2D array containing image moments. Type: 32s. Size: spot count by kMP3D_TRACHEID_TYPE_MOMENT_COUNT.
* @param   area         Output array for area of fitted ellipse. Size: kArray2_Length(moments, 1). Type: 64f. Array will be reallocated
                        if size and type are not as expected.
* @param   angle        Output array for angle of fitted ellipse. Size: kArray2_Length(moments, 1). Type: 64f. Array will be reallocated
                        if size and type are not as expected.   
* @param   minor        Output array for minor axis length of fitted ellipse. Size: kArray2_Length(moments, 1). Type: 64f. Array will be reallocated
                        if size and type are not as expected.
* @param   major        Output array for major axis length of fitted ellipse. Size: kArray2_Length(moments, 1). Type: 64f. Array will be reallocated
                        if size and type are not as expected.
* @param   scatter      Output array for minor to major axis ratio of fitted ellipse. Size: kArray2_Length(moments, 1). Type: 64f. Array will be reallocated
                        if size and type are not as expected.
* @return               Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_GetEllipse(kArray2 moments, kArray1 area, kArray1 angle, kArray1 minor, kArray1 major, kArray1 scatter);

/**
* Uncompressed bit array into binarized 8-bit image
*
* @public               @memberof kMp3dTracheidAlg
* @param   in           Pointer to 8u data buffer containing bit-compressed data.
* @param   out          Output 8u binarized image.
* @param   height       
* @param   width        
* @return               Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_UncompressImage(k8u* in, kImage out, kSize height, kSize width);

/**
* Outputs uncompressed binarized image used in tracheid calculations.
*
* @public               @memberof kMp3dTracheidAlg
* @param   tracheid     Destination for the constructed object handle.
* @param   out          8u binarized image.
* @return               Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_BinarizedImage(kMp3dTracheidAlg tracheid, kImage out);

/**
* Sets the number of columns for calculation of current frame backround noise. The backround noise is calculated 
* as the average intensity of all pixels from column 0 to backgroundWidth. Used in raw image input mode only.
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @param   backgroundWidth      Width of background noise calculation. Default: 32
* @return                       Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_SetBackgroundWidth(kMp3dTracheidAlg tracheid, kSize backgroundWidth);

/**
* Returns the number of columns for calculation of current frame backround noise. The backround noise is calculated 
* as the average intensity of all pixels from column 0 to backgroundWidth. Used in raw image input mode only.
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @return                        Width of background noise calculation
*/
kVsFx(kSize) kMp3dTracheidAlg_BackgroundWidth(kMp3dTracheidAlg tracheid);

/**
* If flip index is enabled, the order of spot moment data is reversed such that index 0 of
* moment arrays refers to the last spot's data and vice versa
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @param   flipIndex            Boolean to enable index reversal for moment data. Default: kFALSE
* @return                       Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_FlipIndexes(kMp3dTracheidAlg tracheid, kBool flipIndex);

/**
* Returns toggle for reversing indexes of moment data.
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @return                       Boolean to enable index reversal for moment data.
*/
kVsFx(kBool) kMp3dTracheidAlg_IndexesFlipped(kMp3dTracheidAlg tracheid);

/**
* Sets the tracheid threshold. This value is added to the background level estimate of the image to form
* the overall binarization threshold. For raw image processing only.
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @param   tracheidThreshold    Tracheid threshold. Default: 8
* @return                       Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_SetTracheidThreshold(kMp3dTracheidAlg tracheid, k32u tracheidThreshold);

/**
* Returns tracheid threshold. This value is added to the background level estimate of the image to form
* the overall binarization threshold. For raw image processing only.
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @return                       Black level offset
*/
kVsFx(k32u) kMp3dTracheidAlg_TracheidThreshold(kMp3dTracheidAlg tracheid);

/**
* Sets the runtime imager window for particular camera. Note, binarize data height is one less row than
* the run window; this is handled internally. Run window should not contain that adjustment.
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @param   runWindow            Runtime window. Default: Same as cal window for the same viewIndex
* @param   viewIndex            Camera index
* @return                       Operation status.
*/
kVsFx(kStatus) kMp3dTracheidAlg_SetRunWindow(kMp3dTracheidAlg tracheid, kL3dCameraWindow runWindow, kSize viewIndex);

/**
* Returns the runtime imager window for particular camera.
*
* @public                       @memberof kMp3dTracheidAlg
* @param   tracheid             Destination for the constructed object handle.
* @param   viewIndex            Camera index
* @return                       Runtime window
*/
kVsFx(kL3dCameraWindow) kMp3dTracheidAlg_RunWindow(kMp3dTracheidAlg tracheid, kSize viewIndex);

#include <kVision/Mp3d/kMp3dTracheidAlg.x.h>

#endif  /* K_VISION_MP3D_TRACHEID_ALG_H */
