/**
 * @file kMp3dSensorCal.h
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */

#ifndef K_VISION_MP3D_SENSOR_CAL_H
#define K_VISION_MP3D_SENSOR_CAL_H

#include <kFireSync/Data/kDataTree.h>
#include <kApi/Data/kBytes.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArrayList.h>
#include <kVision/L3d/kL3dCommon.h>
#include <kVision/L3d/kL3dUtilities.h>

/**
 * Calibration file name.
 *
 * @relates kMp3dSensorCal
 */
#define kMP3D_SENSOR_CAL_FILE_NAME    "Cal.kdat"

/**
 * @class       kMp3dSensorCal
 * @extends     kObject
 * @ingroup     kVision-Mp3d
 * @brief       Calibration data.
 */
typedef kObject kMp3dSensorCal;

/**
 * Constructs a kMp3dSensor object.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  stepCount           Count of steps
 * @param  viewCount           Count of views
 * @param  viewLightCount      Count of viewLights
 * @param  lightSpotCount      Count of lightSpots
 * @param  windowWidth         Width of calibration window
 * @param  windowHeight        Height of calibration window
 * @param  allocator           Memory allocator.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dSensorCal_Construct(kMp3dSensorCal* cal, kSize stepCount, kSize viewCount, kSize viewLightCount, kSize lightSpotCount, kSize windowWidth, kSize windowHeight, kAlloc allocator);

/**
 * Load sensor cal. data.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  node                Node to load file from. Use kNULL to load from disk.
 * @param  fileName            Filename.
 * @param  allocator           Memory allocator.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dSensorCal_Load(kMp3dSensorCal* cal, kNode node, const kChar* fileName, kAlloc allocator);

/**
 * Save sensor cal. data.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  node                Node to load file from. Use kNULL to save to disk.
 * @param  fileName            Filename.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dSensorCal_Save(kMp3dSensorCal cal, kNode node, const kChar* filename);

/**
 * Gets sensor ID
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @return                     Sensor ID.
 */
kVsFx(k32u) kMp3dSensorCal_Id(kMp3dSensorCal cal);

/**
 * Gets temperature
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @return                     Temperature.
 */
kVsFx(k64f) kMp3dSensorCal_Temperature(kMp3dSensorCal cal);

/**
 * Sets sensor ID
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  id                  Sensor ID.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dSensorCal_SetId(kMp3dSensorCal cal, k32u id);

/**
 * Sets temperature.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  temperature         Temperature.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dSensorCal_SetTemperature(kMp3dSensorCal cal, k64f temperature);

/**
 * Gets timestamp
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @return                     Timestamp.
 */
kVsFx(const kChar*) kMp3dSensorCal_Timestamp(kMp3dSensorCal cal);

/**
 * Sets timestamp
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  timestamp           Timestamp.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dSensorCal_SetTimestamp(kMp3dSensorCal cal, const kChar* timestamp);

/**
 * Gets calibration window.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           view index.
 * @return                     Calibration window.
 */
kVsFx(const kL3dCameraWindow*) kMp3dSensorCal_CalWindow(kMp3dSensorCal cal, kSize viewIndex);

/**
 * Sets calibration window.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           View index.
 * @param  window              Calibration window.
 * @return                     Operation status.
 */
kVsFx(kStatus) kMp3dSensorCal_SetCalWindow(kMp3dSensorCal cal, kSize viewIndex, const kL3dCameraWindow* window);

/**
 * Gets count of views.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @return                     Count of views.
 */
kVsFx(kSize) kMp3dSensorCal_ViewCount(kMp3dSensorCal cal);

/**
 * Gets Count of steps.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @return                     Count of steps.
 */
kVsFx(kSize) kMp3dSensorCal_StepCount(kMp3dSensorCal cal);

/**
 * Gets count of lights per view.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @return                     Lights per view.
 */
kVsFx(kSize) kMp3dSensorCal_ViewLightCount(kMp3dSensorCal cal);

/**
 * Gets count of spots per light.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @return                     Spots per light.
 */
kVsFx(kSize) kMp3dSensorCal_LightSpotCount(kMp3dSensorCal cal);

/**
 * Get a pointer to the Z values.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           View index.
 * @param  stepIndex           Step index.
 * @param  spotIndex           Spot index.
 * @return                     Pointer to Z values.
 */
kVsFx(k64f*) kMp3dSensorCal_ZAt(kMp3dSensorCal cal, kSize viewIndex, kSize stepIndex, kSize spotIndex);

/**
 * Get a pointer to the X values.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           View index.
 * @param  stepIndex           Step index.
 * @param  spotIndex           Spot index.
 * @return                     Pointer to X values.
 */
kVsFx(k64f*) kMp3dSensorCal_XAt(kMp3dSensorCal cal, kSize viewIndex, kSize stepIndex, kSize spotIndex);

/**
 * Get a pointer to the row values.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           View index.
 * @param  stepIndex           Step index.
 * @param  spotIndex           Spot index.
 * @return                     Pointer to row values.
 */
kVsFx(k64f*) kMp3dSensorCal_RowAt(kMp3dSensorCal cal, kSize viewIndex, kSize stepIndex, kSize spotIndex);
/**
 * Get a pointer to the column values.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           View index.
 * @param  stepIndex           Step index.
 * @param  spotIndex           Spot index.
 * @return                     Pointer to column values.
 */
kVsFx(k64f*) kMp3dSensorCal_ColAt(kMp3dSensorCal cal, kSize viewIndex, kSize stepIndex, kSize spotIndex);

/**
 * Get a pointer to the X perspective scale values.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           View index.
 * @param  row                 Image row (matches calibration window)
 * @param  col                 Image column (matches calibration window)
 * @return                     Pointer to X perspective scale values.
 */
kVsFx(k64f*) kMp3dSensorCal_XPerspectiveScaleAt(kMp3dSensorCal cal, kSize viewIndex, kSize row, kSize col);

/**
 * Get a pointer to the Y perspective scale values.
 * 
 * @public                     @memberof kMp3dSensorCal
 * @param  cal                 kMp3dSensorCal object.
 * @param  viewIndex           View index.
 * @param  row                 Image row (matches calibration window)
 * @param  col                 Image column (matches calibration window)
 * @return                     Pointer to Y perspective scale values.
 */
kVsFx(k64f*) kMp3dSensorCal_YPerspectiveScaleAt(kMp3dSensorCal cal, kSize viewIndex, kSize row, kSize col);



kVsFx(kStatus) kMp3dSensorCal_AddTempEntry(kMp3dSensorCal cal, kSize viewIndex, k64f temperature, k64f z, kArray2 cols, kArray2 rows);
kVsFx(kStatus) kMp3dSensorCal_RemoveTempEntry(kMp3dSensorCal cal, kSize viewIndex, kSize index);
kVsFx(kSize) kMp3dSensorCal_TempEntryCount(kMp3dSensorCal cal, kSize viewIndex);
kVsFx(k64f) kMp3dSensorCal_MinTemperature(kMp3dSensorCal cal, kSize viewIndex);
kVsFx(k64f) kMp3dSensorCal_MaxTemperature(kMp3dSensorCal cal, kSize viewIndex);
kVsFx(k64f) kMp3dSensorCal_NearestTemperature(kMp3dSensorCal cal, kSize viewIndex, k64f temperature);
kVsFx(k64f) kMp3dSensorCal_TempEntryTempAt(kMp3dSensorCal cal, kSize viewIndex, kSize entry);
kVsFx(k64f) kMp3dSensorCal_TempEntryZAt(kMp3dSensorCal cal, kSize viewIndex, kSize entry);
kVsFx(k64f) kMp3dSensorCal_TempEntryColumnAt(kMp3dSensorCal cal, kSize viewIndex, kSize entry, kSize doe, kSize dot);
kVsFx(k64f) kMp3dSensorCal_TempEntryRowAt(kMp3dSensorCal cal, kSize viewIndex, kSize entry, kSize doe, kSize dot);
kVsFx(k64f) kMp3dSensorCal_TempEntryRowDiffAt(kMp3dSensorCal cal, kSize viewIndex, kSize entry, kSize doe, kSize dot);
kVsFx(k64f) kMp3dSensorCal_TempEntryColumnDiffAt(kMp3dSensorCal cal, kSize viewIndex, kSize entry, kSize doe, kSize dot);

#include <kVision/Mp3d/kMp3dSensorCal.x.h>

#endif /* #ifndef K_VISION_MP3D_SENSOR_CAL_H */

