/**
 * @file kMp3dProfiler.x.h
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */

#ifndef K_VISION_MP3D_PROFILER_X_H
#define K_VISION_MP3D_PROFILER_X_H

#include <kApi/Data/kBytes.h>
#include <kApi/Data/kMath.h>
#include <kApi/Threads/kTimer.h>
#include <kVision/Common/kNullMath.h>
#include <kVision/Common/kPolynomial.h>
#include <kVision/Common/kPolynomialFit.h>
#include <kVision/L3d/kL3dTransform2d.h>
#include <kVision/L3d/kL3dUtilities.h>
#include <kVision/Mp3d/kMp3dSensorCal.h>
#include <kVision/Mp3d/kMp3dUtilities.h>
#include <kVision/Vs/kVsSpline.h>

/**
 * Reserved kMp3dProfiler macros.
 *
 * @relates kMp3dProfiler
 * @{ */
#define kMP3D_PROFILER_MAX                         (k16S_MAX)
#define kMP3D_PROFILER_MIN                         (k16S_MIN + 1)
#define kMP3D_PROFILER_CLIP(VALUE)                 ((((VALUE) > (kMP3D_PROFILER_MIN)) && ((VALUE) < (kMP3D_PROFILER_MAX))) ? (VALUE) : k16S_NULL)
#define kMP3D_PROFILER_LUT_ROUND                   (1 << (kSPOT_CENTRE_SHIFT - 1))
#define kMP3D_PROFILER_LUT_MASK                    ((1 << kSPOT_CENTRE_SHIFT) - 1)
#define kMP3D_PROFILER_HEIGHT_SHIFT                (2)
#define kMP3D_PROFILER_HEIGHT_SCALE                (1 << kMP3D_PROFILER_HEIGHT_SHIFT)
#define kMP3D_PROFILER_MAX_TEMP                    (60)
#define kMP3D_PROFILER_MIN_TEMP                    (-10)
#define kMP3D_PROFILER_TEMP_LUT_SHIFT              (4)
#define kMP3D_PROFILER_TEMP_LUT_SCALE              (1 << kMP3D_PROFILER_TEMP_LUT_SHIFT)
#define kMP3D_PROFILER_MIN_TEMP_ENTRIES            (2)
#define kMP3D_PROFILER_MERGE_WEIGHT_SHIFT          (3)
#define kMP3D_PROFILER_RANGE_LUT_MARGIN            (4)
#define kMP3D_PROFILER_ID_LUT_MARGIN               (8)
#define kMP3D_PROFILER_IDLUT_MAX_DIST_FACTOR       (1.25)
#define kMP3D_PROFILER_TEMP_CAL_FIT_ORDER          (3)

/** @} */

typedef struct kMp3dProfilerClass
{
    kObjectClass base;

    kAlloc alloc;

    kBool setupCalledFlag;                                          // setup must be called before using other LUT functions
    kMp3dSensorCal sensorCal;                                       // data from cal file
    kL3dCameraWindow calWindow[kMP3D_PROFILER_MAX_VIEW_COUNT];      // cal. window
    kSize viewCount;                                                // view count
    kSize stepCount;                                                // step count
    kSize viewSpotCount;                                            // spot count per view

    kSize maxIdDuplicateCount;                                      // max spots of same id
    k64f xResolution;                                               // x resolution
    k64f zResolution;                                               // z resolution
    
    kL3dCameraWindow runWindow[kMP3D_PROFILER_MAX_VIEW_COUNT];      // spot window as updated
    k32u yGranularity;                                              // Y granularity of imager
    k32u heightGranularity;                                         // Height granularity of imager
    k32u widthGranularity;                                          // Width granularity of imager
    k32s spotXShift[kMP3D_PROFILER_MAX_VIEW_COUNT];                 // spot x shift for subsampling
    k32s spotYShift[kMP3D_PROFILER_MAX_VIEW_COUNT];                 // spot y shift for subsampling
    k32s spotXOffset[kMP3D_PROFILER_MAX_VIEW_COUNT];                // spot x offset for LUT
    k32s spotYOffset[kMP3D_PROFILER_MAX_VIEW_COUNT];                // spot x offset for LUT
    k32u spotXMax[kMP3D_PROFILER_MAX_VIEW_COUNT];                   // max spot x
    k32u spotYMax[kMP3D_PROFILER_MAX_VIEW_COUNT];                   // max spot y
    kRect3d64f fov;                                                 // sensor field of view
    k32s minX;                                                      // min scaled x
    k32s maxX;                                                      // max scaled x
    k32s minZ;                                                      // min scaled z
    k32s maxZ;                                                      // max scaled z

    k32s rejectWeakMultiplesFactor;                                 // minimum fraction of max spot strength within common ids
    k32u idCorrectionMinSeparation;                                 // x separation distance used in id correction
    kBool flipCoordinates;                                          // flip the coordinates

    kArray2 idLut[kMP3D_PROFILER_MAX_VIEW_COUNT];                   // ID LUT
    kArray2 xLut[kMP3D_PROFILER_MAX_VIEW_COUNT];                    // Full window floating point X LUTs for each spot. Column based.
    kArray2 zLut[kMP3D_PROFILER_MAX_VIEW_COUNT];                    // Full window floating point Z LUTs for each spot. Column based.
    kArray2 zLutRow[kMP3D_PROFILER_MAX_VIEW_COUNT];                 // Full window floating point Z LUTs for each spot. Row based.
    kArray2 xZLut[kMP3D_PROFILER_MAX_VIEW_COUNT];                   // transformed LUT. It is of type <kPoint16s> where the coordinates represent X, Z values
    kArray2 spotBuffer[kMP3D_PROFILER_MAX_VIEW_COUNT];              // used to hold sorted spots
    kArray1 spotCounts[kMP3D_PROFILER_MAX_VIEW_COUNT];              // used with spotBuffer
    kArray2 tempCompLut[kMP3D_PROFILER_MAX_VIEW_COUNT];             /* column offsets per spot by temperature */
    k32s tempLutHeight;                                             /* number of scaled temperature points in temperature compensation LUT */
    k32s minScaledTemp;                                             /* min scaled temperature */
    k32s maxScaledTemp;                                             /* max scaled temperature */
    kBool temperatureCalibrated;

} kMp3dProfilerClass;

kDeclareClassEx(kVs, kMp3dProfiler, kObject)

kStatus kMp3dProfiler_Init(kMp3dProfiler profiler, kMp3dSensorCal sensorCal, kAlloc allocator);
kVsFx(kStatus) kMp3dProfiler_VRelease(kMp3dProfiler profiler);

kStatus kMp3dProfiler_LutPopulateId(kMp3dProfiler profiler, kSize viewIndex);
kStatus kMp3dProfiler_RejectWeakMultiplesFilter(kMp3dProfiler profiler, kSize viewIndex);
kStatus kMp3dProfiler_IdCorrectionFilter(kMp3dProfiler profiler, kSize viewIndex);
kStatus kMp3dProfiler_MergeAllFilter(kMp3dProfiler profiler, kSize viewIndex);
kStatus kMp3dProfiler_LutPopulateXAndZ(kMp3dProfiler profiler, kSize viewIndex);
kStatus kMp3dProfiler_Clip(kMp3dProfiler profiler, kPoint16s* outRanges, kSpot2* outSortedSpots, kSize count);
kStatus kMp3dProfiler_FitColumns1D(kMp3dProfiler profiler, kSize viewIndex, kSize spotIndex, kSize firstValid, kSize validStepCount, k64f cMin, k64f cMax);
kStatus kMp3dProfiler_FitColumns(kMp3dProfiler profiler, kSize viewIndex, kSize spotIndex, kSize firstValid, kSize validStepCount, k64f cMin, k64f cMax);
kStatus kMp3dProfiler_FitRows(kMp3dProfiler profiler, kSize viewIndex, kSize spotIndex, kSize firstValid, kSize validStepCount, k64f rMin, k64f rMax);
kStatus kMp3dProfiler_PopulateTempLut(kMp3dProfiler profiler, kSize viewIndex);
kStatus kMp3dProfiler_VerifyRunWindow(kMp3dProfiler profiler, kSize viewIndex);
kStatus kMp3dProfiler_ActiveAreaRows(kMp3dProfiler profiler, const kRect64f* profileRect, kSize viewIndex, kSSize* minRow, kSSize* maxRow);
kStatus kMp3dProfiler_ActiveAreaCols(kMp3dProfiler profiler, const kRect64f* profileRect, kSize viewIndex, kSSize* minCol, kSSize* maxCol);

#define kxMP3D_PROFILER_INTERP(V0, V1, FRACT)   ((V0) + ((((V1) - (V0)) * (FRACT) + (kMP3D_PROFILER_LUT_ROUND)) >> (kSPOT_CENTRE_SHIFT)))

#define kxMP3D_PROFILER_INTERP_RANGE_LIN(P0, P1, FRACT, OUT)\
do\
{\
    if (((P0)->x == k16S_NULL) || ((P1)->x == k16S_NULL) || ((P0)->y == k16S_NULL) || ((P1)->y == k16S_NULL))\
    {\
        (OUT)->x = k16S_NULL;\
        (OUT)->y = k16S_NULL;\
    }\
    else\
    {\
        (OUT)->x = (k16s)kMP3D_PROFILER_INTERP((P0)->x, (P1)->x, (FRACT)); \
        (OUT)->y = (k16s)kMP3D_PROFILER_INTERP((P0)->y, (P1)->y, (FRACT)); \
    }\
} while (0)

#define kxMP3D_PROFILER_RANGE_LOOKUP(PROFILER, VIEWINDEX, SPOTINDEX, TEMPINDEX, X, Y, XWORLD, ZWORLD)\
do\
{\
    kObjN(kMp3dProfiler, OBJ, PROFILER);\
    k32u xIndex;\
    k32s xFract;\
    kPoint16s* basePt;\
    kPoint16s out;\
    k32s correction;\
    k32u xv;\
    if ((OBJ)->temperatureCalibrated)\
    {\
        correction = *(k32s*)kArray2_At((OBJ)->tempCompLut[VIEWINDEX], TEMPINDEX, SPOTINDEX);\
    }\
    else\
    {\
        correction = 0;\
    }\
    xv = ((X) << (OBJ)->spotXShift[VIEWINDEX]) + (OBJ)->spotXOffset[VIEWINDEX] - correction;\
    if ((OBJ)->setupCalledFlag && xv < (OBJ)->spotXMax[VIEWINDEX])\
    {\
        xIndex = xv >> kSPOT_CENTRE_SHIFT;\
        xFract = xv & kMP3D_PROFILER_LUT_MASK;\
        basePt = (kPoint16s*) kArray2_At(OBJ->xZLut[VIEWINDEX], SPOTINDEX, xIndex);\
        kMP3D_PROFILER_INTERP_RANGE_LIN(basePt, basePt + 1, xFract, &out);\
        *(XWORLD) = out.x;\
        *(ZWORLD) = out.y;\
    }\
    else\
    {\
        *(XWORLD) = *(ZWORLD) = k16S_NULL;\
    }\
} while (0)

#endif /* #ifndef K_VISION_MP3D_PROFILER_X_H */
