/**
* @file    kL3dSegmentSortBlockInfo.h
* @brief   Declares the kL3dSegmentSortBlockInfo class.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/
#ifndef LMITECH_KVISION_L3D_SEGMENT_SORT_BLOCK_INFO_H_INCLUDED
#define LMITECH_KVISION_L3D_SEGMENT_SORT_BLOCK_INFO_H_INCLUDED

#include <kVision/Common/kVision.h>
#include <kApi/Data/kXml.h>

#define kL3D_SEGMENT_SORT_BLOCK_NAME                            "SegmentSort"
#define kL3D_SEGMENT_SORT_BLOCK_TYPE_NAME                       "kL3dSegmentSort"

#define kL3D_SEGMENT_SORT_BLOCK_IN_SPOTS_NAME                   "InSpots"
#define kL3D_SEGMENT_SORT_BLOCK_OUT_SPOTS_NAME                  "OutSpots"
#define kL3D_SEGMENT_SORT_BLOCK_OUT_PLOT_NAME                   "OutPlot"

typedef enum
{
    kL3D_SEGMENT_SORT_BLOCK_IN_SPOTS  = 0,
    kL3D_SEGMENT_SORT_BLOCK_OUT_SPOTS = 1,
    kL3D_SEGMENT_SORT_BLOCK_OUT_PLOT  = 2
} kL3dSegmentSortBlockPort;

/**
* @class   kL3dSegmentSortBlockInfo
* @ingroup kVision
* @brief   Settings class for kL3dSegmentSortBlock
*/
typedef kObject kL3dSegmentSortBlockInfo;

/**
* Constructs a kL3dSegmentSortBlockInfo object
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_Construct(kL3dSegmentSortBlockInfo* info, kAlloc allocator);

/**
* Reads values from a kXml object
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object.
* @param   xml         Source kXml object
* @param   item        Root xml node
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_Read(kL3dSegmentSortBlockInfo info, kXml xml, kXmlItem item);

/**
* Writes values to a kXml object
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object.
* @param   xml         destination kXml object
* @param   item        Root xml node
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_Write(kL3dSegmentSortBlockInfo info, kXml xml, kXmlItem item);

/**
* Loads a new kL3dSegmentSortBlockInfo from a kXml object
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   xml         Source kXml object
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_Load(kL3dSegmentSortBlockInfo* info, kXml xml, kAlloc allocator);

/**
* Saves a kL3dSegmentSortBlockInfo to a new kXml object
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Object to be saved
* @param   xml         Pointer to a destination kXml handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_Save(kL3dSegmentSortBlockInfo info, kXml* xml, kAlloc allocator);

/**
* Set the number of columns
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   count       Number of columns
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_SetColumnCount(kL3dSegmentSortBlockInfo info, kSize count);

/**
* Get the number of columns
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Size of the spot image along the centroid axis (pixels)
*/
kVsFx(kSize) kL3dSegmentSortBlockInfo_ColumnCount(kL3dSegmentSortBlockInfo info);

/**
* Set the size of the spot image along the centroid axis (pixels)
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   range       Size of the spot image along the centroid axis (pixels)
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_SetCentreRange(kL3dSegmentSortBlockInfo info, kSize range);

/**
* Get the size of the spot image along the centroid axis (pixels)
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Size of the spot image along the centroid axis (pixels)
*/
kVsFx(kSize) kL3dSegmentSortBlockInfo_CentreRange(kL3dSegmentSortBlockInfo info);

/**
* Set the maximum number of segments expected by the algorithm. In the worst case should be set to the maximum number of spots
* being output
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   capacity    Maximum number of segments expected by the algorithm
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_SetSegmentCapacity(kL3dSegmentSortBlockInfo info, kSize capacity);

/**
* Get the maximum number of segments expected by the algorithm.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Maximum number of segments expected by the algorithm
*/
kVsFx(kSize) kL3dSegmentSortBlockInfo_SegmentCapacity(kL3dSegmentSortBlockInfo info);

/**
* Set the margin (in pixels) around current centroid of an active segment over which a new spot can be assigned to that segment.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   margin      Margin (in pixels) around current centroid of an active segment over which a new spot can be assigned to that segment.
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_SetCentreMargin(kL3dSegmentSortBlockInfo info, kSize margin);

/**
* Get the margin (in pixels) around current centroid of an active segment over which a new spot can be assigned to that segment.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Margin (in pixels) around current centroid of an active segment over which a new spot can be assigned to that segment.
*/
kVsFx(kSize) kL3dSegmentSortBlockInfo_CentreMargin(kL3dSegmentSortBlockInfo info);

/**
* Set the margin (in pixels) around current slice of an active segment over which a new spot can be assigned to that segment.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   margin      Margin (in pixels) around current slice of an active segment over which a new spot can be assigned to that segment.
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_SetSliceMargin(kL3dSegmentSortBlockInfo info, kSize margin);

/**
* Get the margin (in pixels) around current slice of an active segment over which a new spot can be assigned to that segment.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Margin (in pixels) around current slice of an active active segment over which a new spot can be assigned to that segment.
*/
kVsFx(kSize) kL3dSegmentSortBlockInfo_SliceMargin(kL3dSegmentSortBlockInfo info);

/**
* When enabled, if ovelap between two segments is found, only the spots from the stronger segment (measured by total sum) are kept.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   resolve     Enable spot overlap resolution
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_ResolveSpotOverlap(kL3dSegmentSortBlockInfo info, kBool resolve);

/**
* Query whether spot overlap resolution is enabled
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Flag indicating whether spot overlap resolution is enabled
*/
kVsFx(kBool) kL3dSegmentSortBlockInfo_SpotOverlapResolved(kL3dSegmentSortBlockInfo info);

/**
* Set minimum acceptable segment length (number of slices in pixels including gaps). This feature can be used to eliminate short segments resulting from camera or ambient noise.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   length      Minimum acceptable segment length (number of slices in pixels)
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_SetMinSegmentLength(kL3dSegmentSortBlockInfo info, kSize length);

/**
* Get minimum acceptable segment length (number of slices in pixels including gaps). This feature can be used to eliminate short segments resulting from camera or ambient noise.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Minimum acceptable segment length (number of slices in pixels)
*/
kVsFx(kSize) kL3dSegmentSortBlockInfo_MinSegmentLength(kL3dSegmentSortBlockInfo info);

/**
* Set minimum acceptable number of spots in a segment. This feature can be used to eliminate short segments resulting from camera or ambient noise.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @param   length      Minimum number of spots in a segment
* @return              Operation status
*/
kVsFx(kStatus) kL3dSegmentSortBlockInfo_SetMinSegmentSpotCount(kL3dSegmentSortBlockInfo info, kSize count);

/**
* Get minimum acceptable number of spots in a segment. This feature can be used to eliminate short segments resulting from camera or ambient noise.
*
* @public              @memberof kL3dSegmentSortBlockInfo
* @param   info        Info object
* @return              Minimum number of spots in a segment
*/
kVsFx(kSize) kL3dSegmentSortBlockInfo_MinSegmentSpotCount(kL3dSegmentSortBlockInfo info);

#include <kVision/L3d/kL3dSegmentSortBlockInfo.x.h>

#endif
