/**
 * @file   kL3dProfileSampler.x.h
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kL3D_PROFILE_SAMPLER_X_H
#define kL3D_PROFILE_SAMPLER_X_H

#define kL3D_PROFILE_SAMPLER_INTERPOLANT_SHIFT  14
#define kL3D_PROFILE_SAMPLER_SLOPETHRESH_SCALE  1000
#define kL3D_PROFILE_SAMPLER_RECIP_SHIFT 20
#define kL3D_PROFILE_SAMPLER_RECIP_SCALE (1 << kL3D_PROFILE_SAMPLER_RECIP_SHIFT) // 2^20

#include <math.h>
#include <kApi/Data/kMath.h>

//////////////////////////////////////////////////////////////////////////
// Start, End, Spacing => Start, Count, Step
//////////////////////////////////////////////////////////////////////////

typedef struct kL3dProfileSamplerClass 
{
    kObjectClass base;

    k16s xStart;
    k16s xEnd;
    k16s xStep;

    kSize xCount;

    k16s zMin;
    k16s zMax;

    k32u xToBinShiftMultiplier;

    k16s xStepThreshold; // only works when fillAllAdjacent is OFF
    k16s zTanThreshold;  // only works when fillAllAdjacent is ON

    kBool isBottom;
    kBool fillAllAdjacent;

    kBool updateParams;
} kL3dProfileSamplerClass;

kDeclareClassEx(kVs, kL3dProfileSampler, kObject)

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

typedef struct kL3dProfileSamplerIterator 
{
    const k16s* pSrc; // points
    const k8u* iSrc;  // intensities
    kSize pDim; // point dimension

    // iterator position //
    kSSize pos;
    kSSize posStep; // +/- 1
    kSSize posEnd;
} kL3dProfileSamplerIterator;

//////////////////////////////////////////////////////////////////////////
//semi-private exported functions (virtual override methods)
//////////////////////////////////////////////////////////////////////////

kVsFx(kStatus)kL3dProfileSampler_VRelease(kL3dProfileSampler sampler);

//////////////////////////////////////////////////////////////////////////
//non-exported (private) methods
//////////////////////////////////////////////////////////////////////////

kStatus kL3dProfileSampler_Init(kL3dProfileSampler sampler, kAlloc allocator);

//////////////////////////////////////////////////////////////////////////
//cast macro, virtual table macro
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
//
// inline
//
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// indexing: x to i and i to x
//////////////////////////////////////////////////////////////////////////

kInlineFx(k32s) kL3dProfileSampler_BinToX(kSSize i, k32s xStart, k32s xStep)
{
    return xStart + (k32s)i * xStep;
}

kInlineFx(kSSize) kL3dProfileSampler_XToBin(k32s x, k32s xToBinShiftMultiplier)
{
    return (((k64s)x * (k64s)xToBinShiftMultiplier) >> kL3D_PROFILE_SAMPLER_RECIP_SHIFT);
}

//////////////////////////////////////////////////////////////////////////
// linear interpolation
//////////////////////////////////////////////////////////////////////////

kInlineFx(k32s) kL3dProfileSampler_InterpolationParameter(k32s val, k32s x0, k32s x1)
{
    // val -- 16s
    // x1,x0 -- 16s
    // x1 > x0
    kAssert(x1 - x0 > 0);

    return ((val - x0) << kL3D_PROFILE_SAMPLER_INTERPOLANT_SHIFT) / (x1 - x0);
}

kInlineFx(k32s) kL3dProfileSampler_Interpolation(k32s x0, k32s x1, k32s t)
{
    return x0 + ((t * (x1 - x0)) >> kL3D_PROFILE_SAMPLER_INTERPOLANT_SHIFT);
}

//////////////////////////////////////////////////////////////////////////
// Check that segment slope (P0, P1) is smaller than interpAngle
//
// xScale = tan(interpAngle) * L3D_PROFILE_SAMPLER_SLOPETHRESH_SCALE;
// xScale > zDiff / xDiff * L3D_PROFILE_SAMPLER_SLOPETHRESH_SCALE;
//////////////////////////////////////////////////////////////////////////

kInlineFx(kBool) kL3dProfileSampler_CheckSlope(k32s x0, k32s z0, k32s x1, k32s z1, k32s xScale)
{
    return kAbs_(x1 - x0) * xScale > kAbs_(z1 - z0) * kL3D_PROFILE_SAMPLER_SLOPETHRESH_SCALE;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_ResetOut(kSize dimOut, kSize outputWidth, k16s* outProfile, k8u* outIntensity)
{
    kSize i;
    const kSize zIdxOut = dimOut - 1; // 0,1

    if (zIdxOut == 0)
    {
        for (i = 0; i < outputWidth; i++)
        {
            outProfile[dimOut*i] = k16S_NULL;
        }
    }
    else
    {
        for (i = 0; i < outputWidth; i++)
        {
            outProfile[dimOut*i] = k16S_NULL;
            outProfile[dimOut*i + zIdxOut] = k16S_NULL;
        }
    }

    if (!kIsNull(outIntensity))
    {
        //kZeroItems(kTypeOf(k8u), outIntensity, outputWidth);
        for (i = 0; i < outputWidth; i++)
        {
            outIntensity[i] = 0;
        }
    }

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
// this function figures out if X values in xData are increasing or decreasing
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_DetermineDir(const k16s* xData, kSize xDataStep, kSize count, kBool *increasing)
{
    const k16s* it = xData;
    const k16s* end = xData + xDataStep * count;
    const k16s* p0 = kNULL;
    const k16s* p1 = kNULL;

    // find a valid X on the left //
    while (it != end)
    {
        if (*it != k16S_NULL)
        {
            p0 = it;
            break;
        }
        it += xDataStep;
    }

    // find a valid X on the right //
    it = xData + xDataStep * (count - 1);
    end = xData - xDataStep;
    while (it != end)
    {
        if (*it != k16S_NULL)
        {
            p1 = it;
            break;
        }
        it -= xDataStep;
    }

    // if right X bigger than the left X its "increasing" //
    if (!kIsNull(p0) && !kIsNull(p1) && (*p1 - *p0) < 0)
        *increasing = kFALSE;
    else
        *increasing = kTRUE;

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
// init kL3dProfileSamplerRow struct
//
// kSize dimIn = 2 : 3; // x,z or x,y,z
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_InitIterator(kL3dProfileSamplerIterator* it, const k16s* profileData, const k8u* intensityData, kSize inputCount, kSize dimIn, kBool increasing)
{
    it->pSrc = profileData;
    it->iSrc = intensityData;
    it->pDim = dimIn;

    if (increasing)
    {
        it->posStep = 1;

        it->pos = 0;
        it->posEnd = (kSSize)inputCount; // one behind END
    }
    else
    {
        it->posStep = -1;

        it->pos = (kSSize)inputCount - 1;
        it->posEnd = -1; // one before END
    }

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
// iterator helpers
//////////////////////////////////////////////////////////////////////////

kInlineFx(const k8u*) kL3dProfileSampler_Intensity(kL3dProfileSamplerIterator* it)
{
    return (kIsNull(it->iSrc)) ? kNULL : it->iSrc + it->pos;
}

//////////////////////////////////////////////////////////////////////////
// Is iterator pointing to valid point
//
// kFALSE for NULL or out of FOV
// check x,y,z for NULL
// only check x,z for FOV
//////////////////////////////////////////////////////////////////////////

kInlineFx(kBool) kL3dProfileSampler_IsValid(kL3dProfileSampler sampler, kL3dProfileSamplerIterator* it)
{
    kObj(kL3dProfileSampler, sampler);
    const k16s* point = it->pSrc + (it->pos * it->pDim);

    // pDim = [2,3]
    const kSize zId = it->pDim - 1; // [1,2]

    // x check //
    if (point[0] == k16S_NULL) return kFALSE;
    // X FOV disabled // if (point[0] < obj->xStart || point[0] > obj->xEnd) return kFALSE;

    // y check, if y is not there its just checking z //
    if (point[1] == k16S_NULL) return kFALSE;

    // z check //
    if (point[zId] == k16S_NULL) return kFALSE;
    if (point[zId] < obj->zMin || point[zId] > obj->zMax) return kFALSE;

    return kTRUE;
}

//////////////////////////////////////////////////////////////////////////
// Find next valid point
// Not null and in FOV
// If none exist return NULL
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_NextValidPoint(kL3dProfileSampler sampler, kL3dProfileSamplerIterator* it, const k16s** profilePt, const k8u** intensityPt, kSSize *pos)
{
    // find valid point (not null and in FOV) //
    while(it->pos != it->posEnd && !kL3dProfileSampler_IsValid(sampler, it))
    {
        it->pos += it->posStep;
    }

    if (it->pos != it->posEnd)
    {
        *profilePt = it->pSrc + (it->pos * it->pDim);
        *intensityPt = kL3dProfileSampler_Intensity(it);
        *pos = it->pos;
        it->pos += it->posStep;
    }
    else
    {
        *profilePt = kNULL;
        *intensityPt = kNULL;
        *pos = it->pos;
    }

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
// Step to the next valid resample segment
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_NextSegment(kL3dProfileSampler sampler, const k16s** pSrcP0, const k8u** iSrcP0, kSSize* pos0,
    const k16s** pNextP0, const k8u** iNextP0, kSSize pos1)
{
    *pSrcP0 = *pNextP0;
    *iSrcP0 = *iNextP0;
    *pos0 = pos1;

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_UpdateSample(
    kL3dProfileSampler sampler,
    k16s yValue, k16s zValue, k8u iValue,
    kSSize index, kSize dimOut, k16s* outProfile, k8u* outIntensity)
{
    kObj(kL3dProfileSampler, sampler);

    const kSize zIdxOut = dimOut - 1; // 0,1

    k16s* yValOut = outProfile + (dimOut * index);
    k16s* zValOut = outProfile + (dimOut*index + zIdxOut);
    k8u* iValOut = outIntensity + index;

    if ((*zValOut == k16S_NULL) || (!obj->isBottom && zValue > *zValOut) || (obj->isBottom && zValue < *zValOut))
    {
        *zValOut = zValue;
        if (zIdxOut > 0) *yValOut = yValue;
        if (!kIsNull(outIntensity)) *iValOut = iValue;
    }

    return kOK;
}
//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_Setup(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    kCheckArgs(obj->xStart != k16S_NULL);
    kCheckArgs(obj->xStep != k16S_NULL && obj->xStep > 0);
    kCheckArgs(obj->xCount != kSIZE_NULL && obj->xCount > 1);

    const kSSize segmentCount = (kSSize)obj->xCount - 1;
    const kSSize xEnd = obj->xStart + obj->xStep * segmentCount;

    kCheckArgs(xEnd >= k16S_MIN && xEnd <= k16S_MAX);

    obj->xEnd = (k16s)xEnd;

    kCheckArgs(obj->zMin != k16S_NULL);
    kCheckArgs(obj->zMax != k16S_NULL);
    kCheckArgs(obj->zMin < obj->zMax);

    kCheckArgs(obj->xStepThreshold != k16S_NULL && obj->xStepThreshold > 0);
    kCheckArgs(obj->zTanThreshold != k16S_NULL);

    obj->updateParams = kFALSE;

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
//
// Getters / Setters
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetXStart(kL3dProfileSampler sampler, k16s val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->xStart = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(k16s) kL3dProfileSampler_XStart(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->xStart;
}
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetXCount(kL3dProfileSampler sampler, kSize val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->xCount = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(kSize) kL3dProfileSampler_XCount(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->xCount;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetXStep(kL3dProfileSampler sampler, k16s val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->xStep = val;
    obj->xToBinShiftMultiplier = (kL3D_PROFILE_SAMPLER_RECIP_SCALE / (k32u)val) + 1;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(k16s) kL3dProfileSampler_XStep(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->xStep;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetXStepThreshold(kL3dProfileSampler sampler, k16s val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->xStepThreshold = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(k16s) kL3dProfileSampler_XStepThreshold(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->xStepThreshold;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(k16s) kL3dProfileSampler_CalculateZSlopeThreshold(k64f angle, k64f xResolution, k64f zResolution)
{
    const k64f angleThreshold = kMin_(kAbs_(angle), 89.999);
    const k64f tanThreshold = tan(kMath_DegToRad_(angleThreshold));

    const k64f val = kL3D_PROFILE_SAMPLER_SLOPETHRESH_SCALE * tanThreshold * xResolution / zResolution;

    return kMath_Round16s_(kMath_Clamp_(val, k16S_MIN, k16S_MAX));
}

kInlineFx(kStatus) kL3dProfileSampler_SetZSlopeThreshold(kL3dProfileSampler sampler, k16s val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->zTanThreshold = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(k16s) kL3dProfileSampler_ZSlopeThreshold(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->zTanThreshold;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetZMin(kL3dProfileSampler sampler, k16s val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->zMin = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(k16s) kL3dProfileSampler_ZMin(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->zMin;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetZMax(kL3dProfileSampler sampler, k16s val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->zMax = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(k16s) kL3dProfileSampler_ZMax(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->zMax;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetFillAllAdjacent(kL3dProfileSampler sampler, kBool val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->fillAllAdjacent = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(kBool) kL3dProfileSampler_FillAllAdjacent(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->fillAllAdjacent;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kL3dProfileSampler_SetBottomSensor(kL3dProfileSampler sampler, kBool val)
{
    kObj(kL3dProfileSampler, sampler);

    obj->isBottom = val;
    obj->updateParams = kTRUE;

    return kOK;
}

kInlineFx(kBool) kL3dProfileSampler_BottomSensor(kL3dProfileSampler sampler)
{
    kObj(kL3dProfileSampler, sampler);

    return obj->isBottom;
}

//////////////////////////////////////////////////////////////////////////

kInlineFx(k32s) kL3dProfileSampler_XValue(kL3dProfileSampler sampler, kSize index)
{
    kObj(kL3dProfileSampler, sampler);

    return kL3dProfileSampler_BinToX((kSSize)index, obj->xStart, obj->xStep);
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#endif  /* #ifndef kL3D_PROFILE_SAMPLER_X_H */
