/**
* @file    kG3dPolynomMap.h
* @brief   Declares the kG3dPolynomMap class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_POLYNOMMAP_H
#define KVISION_kG3D_POLYNOMMAP_H

#include <kVision/G3d/kG3dCommon.h>
#include <kVision/G3d/kG3dPolynomial.h>
#include <kVision/Vs/kVsJobQueue.h>

/**
* @class       kG3dPolynomMap
* @extends     kObject
* @ingroup     kMC-Filtering
* @brief       Provides a set of utility functions for calculating polynomial function from a given image.
*/
typedef kObject kG3dPolynomMap;

/**
* Constructs a kG3dPolynomMap object
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_Construct(kG3dPolynomMap* polynMap, kAlloc allocator);

/**
* Calculates polynomial function from a surface dataset
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Output polynomial function. 
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_PolynMap(kG3dPolynomMap polynMap, kArray2 data, kG3dPolynomial polynomial, kS3dArray2DataType type);

/**
* Calculates deformation polynomial function in three directions from a dataset of kArray2<kMCMatchPointPairs>
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2 of kMCMatchPointPairs.
* @param   polynX      Output polynomial function in x direction.
* @param   polynY      Output polynomial function in y direction.
* @param   polynZ      Output polynomial function in z direction.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_DeformationXYZ(kG3dPolynomMap polynMap, kArray1 data, kG3dPolynomial polynX, kG3dPolynomial polynY, kG3dPolynomial polynZ);

/**
* Calculates deformation polynomial function in three directions from a dataset of kArray2<kMCMatchPointPairs>
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2 of kMCMatchPointPairs.
* @param   polyn       Output polynomial function.
* @param   direction   Output polynomial in x (0), y(1) or z(2) direction.
* @param   repeat      
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_DeformationDirection(kG3dPolynomMap polynMap, kArray1 data, kG3dPolynomial polyn, k64f *dev, k32s direction, k32s repeat);

/**
* Calculates polynomial function from a data set with marked area.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   valid       Area-marking.
* @param   polynomial  Output polynomial function.
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_ValidPolynMap(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid, kG3dPolynomial polynomial, kS3dArray2DataType type);

/**
* Harmonization of light-intensity variation by dividing the best-fit polynomial function. Only for grid image.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Polynomial function.
* @param   uniformity  Buffer of k64f values to store the Uniformity of light-intensity variation.
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @param   invers      Determines whether the image should be inverted.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_NormalizeGridImage(kG3dPolynomMap polynMap, kArray2 data, kG3dPolynomial polynomial, k64f* uniformity, kS3dArray2DataType type, kBool invers);

/**
* Iteratively computes the polynomial function to extrapolate the missing data according to existing data.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   valid       Data buffer of valid points of the height map. kArray2 of k8u.
* @param   polynomial  Polynomial function.
* @param   order0      Start order of the polynomial.
* @param   order       Target order of the polynomial.
* @param   repeat      Repetition count
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_RobustValidPolynMap(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid,
    kG3dPolynomial polynomial, k32s order0, k32s order, k32s repeat,  kS3dArray2DataType type);

/**
* Calculates the polynomial function from the derivative image.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   valid       Area-marking.
* @param   polynomial  Polynomial function.
* @param   wavelength  Equivalent wavelength of pattern.
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_Derivative3orderMap(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid, kG3dPolynomial polynomial, k32f wavelength, kS3dArray2DataType type);

/**
* Set order of the polynomial.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   order       Order of the polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_SetOrder(kG3dPolynomMap polynMap, k32s order);

/**
* Set the second order of the polynomial function. It is very rarely used. The extrapolation of invalid 
* region is executed with lowered polynomial
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   order       Second order of the polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_SetOrder2(kG3dPolynomMap polynMap, k32s order);

/**
* Set input sampling step for polynomial fitting.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   step        Step for polynomial fitting
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_SetStep(kG3dPolynomMap polynMap, k32s step);

/**
* Calculates the radial symmetry of the distortion distribution
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Polynomial function.
* @param   Symmetry    Handle of k64f to store the radial symmetry of the distortion distribution.
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_DistortionSymmetry(kG3dPolynomMap polynMap, kArray2 data, kG3dPolynomial polynomial, k64f* Symmetry, kS3dArray2DataType type);

/**
* Draws the polynomial function in a map.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Polynomial function.
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_DrawPolynomial(kG3dPolynomMap polynMap, kArray2 data, kG3dPolynomial polynomial, kS3dArray2DataType type);

/**
* Calculates polynomial function from an array 
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data. kArray1 of kPoint3d32f.
* @param   polynomial  Polynomial function.
* @param   balance     If the boolean value is kTRUE, The z values of the input data are subtracted from the polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_PolynArray1Point3D(kG3dPolynomMap polynMap, kArray1 data, kG3dPolynomial polynomial, kBool balance);

/**
* Calculates polynomial function from an data map with appropriate weighting for each point.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data. kArray2 of kPoint3d32f. 
* @param   polynomial  Polynomial function.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_PolynWeightedPoint3D(kG3dPolynomMap polynMap, kArray2 data, kG3dPolynomial polynomial);

/**
* Converts the polynomial function with new polynomial coordination.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Data buffer.  kArray2 < kPoint3d32f >.
* @param   polynOutX   Output polynomial function in x direction.
* @param   polynOutY   Output polynomial function in y direction.
* @param   polynX      Polynomial function of Coordinate x.
* @param   polynY      Polynomial function of Coordinate Y.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_ConvertPolynomialFunctions(kG3dPolynomMap polynMap, kArray2 data, kG3dPolynomial polynOutX, kG3dPolynomial polynOutY, kG3dPolynomial polynX, kG3dPolynomial polynY);

/**
* Converts the polynomial function with new polynomial coordination.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   output      Output polynomial function.
* @param   polynZ      Input polynomial function. x and y are related to camera pixel.
* @param   polynX      Polynomial function of Coordinate x.
* @param   polynY      Polynomial function of Coordinate Y.
* @param   gScale      Grid spacing in mm.
* @param   imageSizeX  Data size in x direction.
* @param   imageSizeY  Data size in y direction.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_PolynFromPolynCoord(kG3dPolynomMap polynMap, kG3dPolynomial output, kG3dPolynomial polynZ, kG3dPolynomial polynX, kG3dPolynomial polynY, k64f gScale, k32s imageSizeX, k32s imageSizeY);

/**
*  Calculates the scaling factor in the form of a polynomial function based on the known groove depth.
*
* @public                @memberof kG3dPolynomMap
* @param   polynMap      kG3dPolynomMap object.
* @param   data          kArray2 of kPoint3d32f.
* @param   valid         Area-marking.
* @param   polynomial    Output polynomial function.
* @param   depth         Groove depth.
* @param   invalidRatio  Buffer of k64f values to store invalid ratio. 
* @return                Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_GroovePhaseScalingX(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid, kG3dPolynomial polynomial, k64f depth, k64f* invalidRatio);

/**
*  Draws the deviation map between measured and best-fit surface.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        kArray2 of kPoint3d32f.
* @param   polynomial  Input polynomial function.
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input and output data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_DrawPolynomialDifference(kG3dPolynomMap polynMap, kArray2 data, kG3dPolynomial polynomial, kS3dArray2DataType type);

/**
*  Draws the deviation map between measured and best-fit surface.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        kArray2 of kPoint3d32f.
* @param   valid
* @param   polynomial  Input polynomial function.
* @param   inType      Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @param   outType     Value of kS3dArray2DataType. Determines the type and involved component of output data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_DrawValidPolynomialDifference(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid, kG3dPolynomial polynomial, kS3dArray2DataType inType, kS3dArray2DataType outType);

/**
*  The Coordinate in x direction will be changed.
*  xout(xin, y) = polynX(xin, y);
*  xin(xout, y) = output(xout,y);
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   polynX      Input polynomial function.
* @param   output      Output polynomial function.
* @param   imageSizeX  Data size in x direction.
* @param   imageSizeY  Data size in y direction.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_ChangePolynCoordX(kG3dPolynomMap polynMap, kG3dPolynomial output, kG3dPolynomial polynX, k32s imageSizeX, k32s imageSizeY);

/**
*  Calculating the difference value of the specified component of input data.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2.
* @param   valid       Area-marking.     
* @param   polynomial  Input polynomial function.
* @param   type        Value of kS3dArray2DataType. Determines the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_MapDifference(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid, kG3dPolynomial polynomial, kS3dArray2DataType type);

/**
*  Polynomial best-fit with weighting.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input data of kArray2 of kPoint3d32f.  The weight factor for each point is stored in data[].z. 
* @param   valid       Area-marking. of kArray2 of k8u.
* @param   segment     Second area-marking. of kArray2 of k8u.
* @param   polynomial  Output polynomial function.
* @param   bit         Only points are included in the calculation corresponding to the marking map valid.
* @param   invalidBit  point position should be determined.
* @param   bX          Should the data be calculated in x direction?
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_PolynWeightedValidPoint3D(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid, kG3dPolynomial polynomial, k8u bit, k8u invalidBit, kBool bX);

/**
*  Polynomial best-fit in x and y directions in same time from a grid input data. 
*  The output is corresponding to the order of grid with a shift of center.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   grid        Input data of kArray2 of kPoint64f.  Invalid points: k64F_NULL
* @param   polynX      Output polynomial function in x direction.
* @param   polynY      Output polynomial function in y direction.
* @param   centerX     Grid center in x direction.
* @param   centerY     Grid center in y direction.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_PolynFromGridPoints(kG3dPolynomMap polynMap, kArray2 grid, kG3dPolynomial polynX, kG3dPolynomial polynY, k64f centerX, k64f centerY);

/**
*  Set lateral scaling of map pixels
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   scale       lateral scaling
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_SetXyScale(kG3dPolynomMap polynMap, k64f scale);

/**
*  Set z resolution. The coefficient will be only used, if the input data is integer.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   scale       Z resolution.
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_SetZScale(kG3dPolynomMap polynMap, k64f scale);

/**
*  Polynomial best-fit from a height map with area-marking.
*
* @public              @memberof kG3dPolynomMap
* @param   polynMap    kG3dPolynomMap object.
* @param   data        Input height map of kArray2 of k32s.  
* @param   valid       Area-marking. of kArray2 of k8u.
* @param   polynomial  Output polynomial function.
* @param   scale       Digital resolution of the height map.
* @param   bit         Only points are included in the calculation corresponding to the marking map valid.
* @param   outputLaw   Should the surface be outputted with the polynomial function?
* @return              Operation status.
*/
kVsFx(kStatus) kG3dPolynomMap_PolynValidMap32s(kG3dPolynomMap polynMap, kArray2 data, kArray2 valid, kG3dPolynomial polynomial, kPoint3d64f scale, k8u bit, kBool outputLaw);


kVsFx(kStatus) kG3dPolynomMap_SetJobQueue(kG3dPolynomMap polynMap, kVsJobQueue queue);

#include <kVision/G3d/kG3dPolynomMap.x.h>

#endif /* #ifndef KVISION_kG3D_POLYNOMMAP_H */
