/**
* @file    kG3dJumpContour.h
* @brief   Declares the kG3dJumpContour class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kS3D_JUMP_CONTOUR_H
#define KVISION_kS3D_JUMP_CONTOUR_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/G3d/kG3dContourSegment.h>
#include <kVision/G3d/kG3dContourHistogram.h>
#include <kVision/G3d/kG3dContourFitLine.h>
#include <kVision/G3d/kG3dContourClosedLine.h>
#include <kVision/G3d/kG3dContourTemplate.h>

/**
* @class       kG3dJumpContour
* @extends     kObject
* @ingroup     kVision-G3d
* @brief       Class for ...
*/
typedef kObject kG3dJumpContour;

/**
* Constructs a kG3dJumpContour object
*
* @public               @memberof kG3dJumpContour
* @param   jumpCountur  Destination for the constructed object handle.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_Construct(kG3dJumpContour* jumpCountur, kAlloc allocator);

/**
* Set the scaling coeffizients of the corresponding height map.
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   scale          scaling coeffizients of the height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_SetScale(kG3dJumpContour jumpCountur, kPoint3d64f scale);

/**
* Performs the evaluation of jump contours without template
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   input          Data buffer for input height map or intensity map or bitmap.
* @param   mode           Type of the input map.  The definition see kG3dInputImagesOrder.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_Run(kG3dJumpContour jumpCountur, kArray2 input, kG3dInputImagesOrder mode);

/**
* Performs the evaluation of jump contours using template
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   input          Data buffer for input height map or intensity map or bitmap.
* @param   mode           Type of the input map.  The definition see kG3dInputImagesOrder
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_RunTemplate(kG3dJumpContour jumpCountur, kObject input, kG3dInputImagesOrder mode);

/**
* Set a template object which includes the information about the evaluation of magnet glue bands
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   mask           Template object for evaluation of magnet glue bands
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_SetTemplate(kG3dJumpContour jumpCountur, kG3dContourTemplate mask);

/**
* Set type of to be measured sample.
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   mode           Type of to be measured sample: 0 = small magnet 1 = big magnet 2 = unknown
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_SetSampleType(kG3dJumpContour jumpCountur, kSize type);

/**
* Set input height map. This function should only be used if the map to be treated is a intensity image.
* The algorithm tries to figure out the invalid range of the map.
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   map            Data buffer for input height map kArray2 < 16s >.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_SetInputHeightMap(kG3dJumpContour jumpCountur, kArray2 map);

/**
* Set sample index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   index          Sample index of a measurement series
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_SetSampleIndex(kG3dJumpContour jumpCountur, kSize index);

/**
* Set repeat index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public                 @memberof kG3dJumpContour
* @param   jumpCountur    kG3dJumpContour object.
* @param   index          Repeat index of a measurement series
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dJumpContour_SetRepeatIndex(kG3dJumpContour jumpCountur, kSize index);

#include <kVision/G3d/kG3dJumpContour.x.h>

#endif  /* KVISION_kS3D_JUMP_CONTOUR_H */
