/**
* @file    kG3dContourTemplate.h
* @brief   Declares the kG3dContourTemplate class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3DCONTOURTEMPLATE_H
#define KVISION_kG3DCONTOURTEMPLATE_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/G3d/kG3dAffineTransform.h>

//#define kG3DCONTOURTEMPLATE_DEBUG


#define kG3DCONTOURTEMPLATE_TRACK_MAX_COUNT   12
#define kG3DCONTOURTEMPLATE_DEFAUT_OUT_THICK 1
#define kG3DCONTOURTEMPLATE_TRACK_MASK_START_BIT 2
#define kG3DCONTOURTEMPLATE_TRACK_PLANE_SUM_COUNT   9

#define kG3DCONTOURTEMPLATE_TRACK_DEFAULT_OVERFLOW_OFFSET   0.01
#define kG3DCONTOURTEMPLATE_TRACK_DEFAULT_GAPS_OFFSET   -0.12

#define kG3DCONTOUR_ANGLE_MAP_NOT_USED   (-5)
#define kG3DCONTOUR_ANGLE_MAP_INVALID    (-10)

#define kG3D_SEGMENT_PLANES_MAXIMUM_OUTLIER_RATE         0.01
#define kG3D_SEGMENT_PLANES_MINIMUM_OUTLIER_RATE         0.01
#define kG3D_SEGMENT_PLANES_DEFAULT_FILTERWIDTH          2


typedef enum GdkGlueBandSensorMode
{
    GDK_MAGNET_GLUEBAND_BUDDY,
    GDK_MAGNET_GLUEBAND_STANDALONE
} GdkGlueBandSensorMode;

typedef enum GdkGlueBandOperationMode
{
    GDK_MAGNET_GLUEBAND_TEMPLATE_NOT_USED,
    GDK_MAGNET_GLUEBAND_USE_TEMPLATE,
    GDK_MAGNET_GLUEBAND_CREATE_TEMPLATE
} GdkGlueBandOperationMode;

typedef enum GdkGlueBandDefaultMagnetType
{
    GDK_MAGNET_GLUEBAND_SMALL = 0,
    GDK_MAGNET_GLUEBAND_BIG,
    GDK_MAGNET_GLUEBAND_UNKNOWN,
    GDK_MAGNET_GLUEBAND_END
} GdkGlueBandDefaultMagnetType;

typedef enum kG3dAffineTransformRelationMode
{
    kG3D_TRANSFORM_BITMAP_TO_MAIN_HEIGHTMAP = 0,
    kG3D_TRANSFORM_MAIN_HEIGHTMAP_TO_BITMAP,
    kG3D_TRANSFORM_BITMAP_TO_BUDDY_HEIGHTMAP,
    kG3D_TRANSFORM_BUDDY_HEIGHTMAP_TO_BITMAP,
    kG3D_TRANSFORM_MAIN_HEIGHTMAP_TO_BUDDY_HEIGHTMAP,
    kG3D_TRANSFORM_BUDDY_HEIGHTMAP_TO_MAIN_HEIGHTMAP,
    kG3D_TRANSFORM_BITMAP_TO_WORLD,
    kG3D_TRANSFORM_WORLD_TO_BITMAP,
    kG3D_TRANSFORM_MAIN_HEIGHTMAP_TO_WORLD,
    kG3D_TRANSFORM_WORLD_TO_MAIN_HEIGHTMAP,
    kG3D_TRANSFORM_BUDDY_HEIGHTMAP_TO_WORLD,
    kG3D_TRANSFORM_WORLD_TO_BUDDY_HEIGHTMAP,
    kG3D_TRANSFORM_WORLD_CORRECTION,
    kG3D_TRANSFORM_MAIN_HEIGHTMAP_CORRECTION,
    kG3D_TRANSFORM_END
}kG3dAffineTransformRelationMode;


typedef enum kG3dBuddySensorRotationMode
{
    kG3D_BUDDY_ORIGINAL_ORIENTATION = 0,
    kG3D_BUDDY_HORIZONTAL_FLIP,
    kG3D_BUDDY_VERTICAL_FLIP,
    kG3D_BUDDY_ROTATE_180,
    kG3D_BUDDY_FLIPPED_ROTATE_90_LEFT,
    kG3D_BUDDY_ROTATE_90_LEFT,
    kG3D_BUDDY_FLIPPED_ROTATE_90_RIGHT,
    kG3D_BUDDY_ROTATE_90_RIGHT
}kG3dBuddySensorRotationMode;

typedef enum kG3dAffineAlginMode
{
    kG3D_TRANSFORM2D_RECTANGLE_AFFINE,
    kG3D_TRANSFORM2D_RECTANGLE_SCALING,
    kG3D_TRANSFORM2D_RECTANGLE_RIGID,
    kG3D_TRANSFORM_6_POINTS_AFFINE,
    kG3D_TRANSFORM_6_POINTS_SCALING,
    kG3D_TRANSFORM_6_POINTS_RIGID
}kG3dAffineAlginMode;

typedef enum kG3dInputImagesOrder
{
    kG3D_AFFINE_MATCHING_BITMAP = 0,
    kG3D_AFFINE_MATCHING_INTENSITY_MAP,
    kG3D_AFFINE_MATCHING_HEIGHT_MAP,
    kG3D_AFFINE_MATCHING_INTENSITY_MAP2,
    kG3D_AFFINE_MATCHING_HEIGHT_MAP2,
    kG3D_AFFINE_MATCHING_OUTPUT_HEIGHT,
    kG3D_AFFINE_MATCHING_OUTPUT_DIFFERENCE,
    kG3D_AFFINE_MATCHING_OUTPUT_INTENSITY,
    kG3D_AFFINE_MATCHING_OUTPUT_BITMAP,
    kG3D_AFFINE_MATCHING_TEMPLATE,
    kG3D_AFFINE_MATCHING_MAP_END
}kG3dInputImagesOrder;

typedef enum kG3dStraightLine_PassPoint_Type
{
    kG3D_STRAIGHTLINE_BORDER_RECT_BOTTOM_LEFT = 0,
    kG3D_STRAIGHTLINE_BORDER_RECT_BOTTOM_RIGHT,
    kG3D_STRAIGHTLINE_BORDER_RECT_TOP_RIGHT,
    kG3D_STRAIGHTLINE_BORDER_RECT_TOP_LEFT,
    kG3D_STRAIGHTLINE_LONGSIDE_GAPS_LEFT_UPPER,
    kG3D_STRAIGHTLINE_LONGSIDE_GAPS_LEFT_LOWER,
    kG3D_STRAIGHTLINE_LONGSIDE_GAPS_RIGHT_UPPER,
    kG3D_STRAIGHTLINE_LONGSIDE_GAPS_RIGHT_LOWER,
    kG3D_STRAIGHTLINE_SHORTSIDE_TOP_CENTER,
    kG3D_STRAIGHTLINE_SHORTSIDE_BOTTOM_CENTER,
    kG3D_STRAIGHTLINE_END
} kG3dStraightLine_PassPoint_Type;

typedef struct kG3dClosedContourType
{
    kPoint3d64f center;
    kPoint3d64f worldCenter;
    kArray1 contour;
    k64f area;
    k32s trackIndex;
    k32s lineIndex;
    k32s mode; // 0 = not used, 1 = inner contour 2 = outer contour, 3= unknow 
}kG3dClosedContourType;


typedef struct kG3dTrackMaskDataType
{
    k64f thick;          // in pixel  
    k64f targetThick;    // in mm 
    k64f overflowOffset; // in mm
    k64f gapsOffset;     // in mm
    kPoint3d64f center;  // in pixel
    kArray1 contour;     // in pixel
    kArray1 inContour;   // in pixel
    kArray1 outContour;  // in pixel
    kArray1 direction;   // in rad
    kArray1 histogram;   // in rad
    kPoint3d64f inPlane;  // plane z (x, y) = inPlane.x * (x - center.x) + inPlane.y * (y - center.y) + inPlane.z
    kPoint3d64f outPlane; // plane z (x, y) = outPlane.x * (x - center.x) + outPlane.y * (y - center.y) + outPlane.z
    k64f frameThick;      // in mm    
    k64s outSum[kG3DCONTOURTEMPLATE_TRACK_PLANE_SUM_COUNT];
    k64s inSum[kG3DCONTOURTEMPLATE_TRACK_PLANE_SUM_COUNT];
    k16s inDev, outDev;
    k64s inDevSum, outDevSum;
    k32s inDevCount, outDevCount;
    kBool inSuccess;
    kBool outSuccess;
    k16s zmin;
    k16s zmax;
    k64f averageHeight;
    kPoint3d64f maxPosition;
    kPoint3d64f minPosition;
    k64f overflowArea;
    k64f gapsArea;
}kG3dTrackMaskDataType;

typedef struct kG3dContourPointType
{
    k16s x, y;
    k16s value;
    k16s angle;
    k16s groupIndex;
    k16s lineIndex;
    k64f dev;
    k16s cosValue, sinValue;
    k8u isBorder;
    k8u isUsed;
}kG3dContourPointType;

/**
* @class       kG3dContourTemplate
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       This class manages the template information for the application of measuring 
*              magnet glue band height.
*/
typedef kObject kG3dContourTemplate;

/**
* Constructs a kG3dContourTemplate object
*
* @public               @memberof kG3dContourTemplate
* @param   mask         Destination for the constructed object handle.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_Construct(kG3dContourTemplate* mask, kAlloc allocator);

/**
* Load the kG3dContourTemplate object from a file.
*
* @public               @memberof kG3dContourMatching
* @param   mask         Destination for the constructed object handle.
* @param   fileName     file name for the template object.
* @param   allocator    Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_Load(kG3dContourTemplate* mask, const kChar* fileName, kAlloc allocator);

/**
* Store a kG3dContourTemplate object into a file.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   fileName       file name for the template object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_Store(kG3dContourTemplate mask, const kChar* fileName);

/**
* Get the data buffer of interesting points, which are used for calculating the affine
* transformation matrix, for a given image index.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   index          Index of the image series defined in kG3dInputImagesOrder
* @return                 Data buffer of interesting points for given image index.
*/
kVsFx(kArray1) kG3dContourTemplate_MeasurementPoints(kG3dContourTemplate mask, k32s index);

/**
* Get the data buffer of magnet track object, which includes information of glue band geometry for each track.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Data buffer of magnet track object.
*/
kVsFx(kArrayList) kG3dContourTemplate_MeasurementTracks(kG3dContourTemplate mask);

/**
* Get the data buffer of raw glue boundary contour, which is created as intermediate results
* when creating template. This function is used for visualizing the contours to check the
* corretness of contour detection in debug mode.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Data buffer of raw glue boundary contour.
*/
kVsFx(kArrayList) kG3dContourTemplate_InterProfiles(kG3dContourTemplate mask);

/**
* All parameters and data buffer in the tnterProfiles object will be cleared.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_ResetInterProfiles(kG3dContourTemplate mask);

/**
* All affine transform matrix for the main height map will be calculated again.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_CalculateTransformMatrix(kG3dContourTemplate mask);

/**
* All affine transform matrix for the buddy height map will be calculated again.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_CalculateTransformMatrix2(kG3dContourTemplate mask);

/**
* The relationship between sensors, external camera and world coordinate system will be generated.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_CalculateTransformWorld(kG3dContourTemplate mask);

/**
* The glue band for each track will be sorted from the detected contours.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SortTrack(kG3dContourTemplate mask);

/**
* Set number of the magnet tracks for current sample
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   trackCount     Number of the magnet tracks
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetTrackCount(kG3dContourTemplate mask, kSize trackCount);

/**
* Get number of the magnet tracks for current sample
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Number of the magnet tracks
*/
kVsFx(kSize) kG3dContourTemplate_TrackCount(kG3dContourTemplate mask);

/**
* Get a affine transformation matrix with a given index, which indicates the relationship between the images.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   mode           An index which indicates the relationship between the images
* @return                 kG3dContourTemplate object.
*/
kVsFx(kL3dTransform3d*) kG3dContourTemplate_TransformMatix(kG3dContourTemplate mask, kG3dAffineTransformRelationMode mode);

/**
* Set the scaling coeffizients of the main height map.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   scale          scaling coeffizients of the main height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetHeightmapScale(kG3dContourTemplate mask, kPoint3d64f scale);

/**
* Get the scaling coeffizients of the main height map.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 scaling coeffizients of the main height map in the 3 directions.
*/
kVsFx(kPoint3d64f) kG3dContourTemplate_HeightmapScale(kG3dContourTemplate mask);

/**
* Set the scaling coeffizients of the external bitmap.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   scale          scaling coeffizients of the external bitmap in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetBitmapScale(kG3dContourTemplate mask, kPoint3d64f scale);

/**
* Get the scaling coeffizients of the external bitmap.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Scaling coeffizients of the external bitmap in the 3 directions.
*/
kVsFx(kPoint3d64f) kG3dContourTemplate_BitmapScale(kG3dContourTemplate mask);

/**
* Set the mode for the calculation of the affine transformation.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   mode           Mode for the calculation of the affine transformation. See definition in kG3dAffineAlginMode
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetAffineAlginMode(kG3dContourTemplate mask, kG3dAffineAlginMode mode);

/**
* Get the mode for the calculation of the affine transformation.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Mode for the calculation of the affine transformation.
*/
kVsFx(kG3dAffineAlginMode) kG3dContourTemplate_AffineAlginMode(kG3dContourTemplate mask);

/**
* Get actual number of magnet tracks detected from the data.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Actual number of magnet tracks detected from the data.
*/
kVsFx(kSize) kG3dContourTemplate_MeasuredTrackCount(kG3dContourTemplate mask);

/**
* Get actual glue band thick measured from the data for the given track index.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   index          Track index. 
* @return                 actual glue band thick for the given track index.
*/
kVsFx(k64f) kG3dContourTemplate_MeasuredTrackThick(kG3dContourTemplate mask, kSize index);

/**
* Return a message if all specified glue bands have been correctly detected.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 All specified glue bands have been correctly detected?
*/
kVsFx(kBool) kG3dContourTemplate_IsTrackComplete(kG3dContourTemplate mask);

/**
* The parameters and data for each track will be set in preparation state
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_ResetTrackContour(kG3dContourTemplate mask);

/**
* The currently used parameters and data for each track will be replaced by the newly calculated values.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_UpdateTrackContour(kG3dContourTemplate mask);

/**
* Set a sample name for identify the template.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   name           Sample name.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetSampleName(kG3dContourTemplate mask, const kChar* name);

/**
* Set the height and width of the external bitmap.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   sizeX          Bitmap width.
* @param   sizeY          Bitmap height.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetBitmapSize(kG3dContourTemplate mask, kSize sizeX, kSize sizeY);

/**
* Rescaling the glue band width from pixel to millimeter.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_UpdateBandThick(kG3dContourTemplate mask);

/**
* Get the height of the output bitmap in the world coordinate system.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Height of the output bitmap.
*/
kVsFx(kSize) kG3dContourTemplate_OutputImageHeight(kG3dContourTemplate mask);

/**
* Get the width of the output bitmap in the world coordinate system.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Width of the output bitmap.
*/
kVsFx(kSize) kG3dContourTemplate_OutputImageWidth(kG3dContourTemplate mask);

/**
* Calculates correction transform matrix from the given point pairs.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   points         Data buffer for position of reference points
* @param   diff           Data buffer for displacement of the points with respect to reference points.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_CorrectionTransformMatrix(kG3dContourTemplate mask, kArrayList points, kArrayList diff);

/**
* The points will be Transformed with the correction transform matrix.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   points         Data buffer for position of points
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_CorrectionPoints(kG3dContourTemplate mask, kArrayList points);

/**
* Get cage thick. This parameter is used for estimate of the maximum range of cage surface
* outside the glue band, which should be used for the evaluation of the reference plane each track.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Cage thick.
*/
kVsFx(k64f) kG3dContourTemplate_FrameThick(kG3dContourTemplate mask);

/**
* Set cage thick. This parameter is used for estimate of the maximum range of cage surface
* outside the glue band, which should be used for the evaluation of the reference plane each track.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   thick          Cage thick.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetFrameThick(kG3dContourTemplate mask, k64f thick);

/**
* Set window width for calculating the average height value.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   width          Window width for calculating the average height value.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetPointAverageWindowWidth(kG3dContourTemplate mask, k32s width);

/**
* Set minimum outlier rate. This parameter is used to restrict the outlier of the measured data on
* the glue band according to a histogram of height. The height distrubution of points with an area
* proportion to the total area of the glue band, which are determined by this values, are regarded
* as outliers. They are not involved in the calculation.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   rate           Minimum outlier rate.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetOutlierMinRate(kG3dContourTemplate mask, k64f rate);

/**
* Set maximum outlier rate.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   rate           Maximum outlier rate.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetOutlierMaxRate(kG3dContourTemplate mask, k64f rate);

/**
* Get window width for calculating the average height value.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Window width for calculating the average height value.
*/
kVsFx(k32s) kG3dContourTemplate_PointAverageWindowWidth(kG3dContourTemplate mask);

/**
* Get minimum outlier rate. This parameter is used to restrict the outlier of the measured data on
* the glue band according to a histogram of height. The height distrubution of points with an area
* proportion to the total area of the glue band, which are determined by this values, are regarded
* as outliers. They are not involved in the calculation.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Minimum outlier rate.
*/
kVsFx(k64f) kG3dContourTemplate_OutlierMinRate(kG3dContourTemplate mask);

/**
* Get maximum outlier rate.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Maximum outlier rate.
*/
kVsFx(k64f) kG3dContourTemplate_OutlierMaxRate(kG3dContourTemplate mask);

/**
* Get overflow offset height. This parameter is used for the determination of the overflow area.
* The offset height refers to the reference plane. Only points above this limit are considered as
*  overflow area.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Overflow offset height.
*/
kVsFx(k64f) kG3dContourTemplate_OverflowOffset(kG3dContourTemplate mask);

/**
* Get gaps offset height. This parameter is used for the determination of the gaps area.
* The offset height refers to the reference plane. Only points below this limit are considered as
*  gaps area.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Gaps offset height.
*/
kVsFx(k64f) kG3dContourTemplate_GapsOffset(kG3dContourTemplate mask);

/**
* Set overflow offset height. This parameter is used for the determination of the overflow area.
* The offset height refers to the reference plane. Only points above this limit are considered as
*  overflow area.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   offset         Overflow offset height.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetOverflowOffset(kG3dContourTemplate mask, k64f offset);

/**
* Set gaps offset height. This parameter is used for the determination of the gaps area.
* The offset height refers to the reference plane. Only points below this limit are considered as
* gaps area.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   offset         Gaps offset height.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetGapsOffset(kG3dContourTemplate mask, k64f offset);

/**
* Get a boolean value which determines, if the intensity map should be used for the calculation of transformation matrix
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Intensity map should be used?
*/
kVsFx(kBool) kG3dContourTemplate_UseIntensityMap(kG3dContourTemplate mask);

/**
* Set a boolean value to determine, whether the intensity map should be used for the calculation of transformation matrix
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   used           Intensity map should be used?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetUseIntensityMap(kG3dContourTemplate mask, kBool used);

/**
* Get a boolean value which determines, whether the main heightmap and intensity map should be rotated 180 degree.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Main heightmap and intensity map should be rotated?
*/
kVsFx(k32s) kG3dContourTemplate_MainMapRotation(kG3dContourTemplate mask);

/**
* Get a boolean value which determines, whether the buddy heightmap and intensity map should be rotated 180 degree.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Buddy heightmap and intensity map should be rotated?
*/
kVsFx(k32s) kG3dContourTemplate_BuddyMapRotation(kG3dContourTemplate mask);

/**
* Get a boolean value which determines, whether the external bitmap should be rotated 180 degree.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 The external bitmap should be rotated?
*/
kVsFx(k32s) kG3dContourTemplate_BitmapRotation(kG3dContourTemplate mask);

/**
* Set a boolean value to determine, whether the main heightmap and intensity map should be rotated 180 degree.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   rotate         Main heightmap and intensity map should be rotated?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetMainMapRotation(kG3dContourTemplate mask, k32s rotate);

/**
* Set a boolean value to determine, whether the buddy heightmap and intensity map should be rotated 180 degree.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   rotate         Buddy heightmap and intensity map should be rotated?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetBuddyMapRotation(kG3dContourTemplate mask, k32s rotate);

/**
* Set a boolean value to determine, whether the external bitmap should be rotated 180 degree.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   rotate         The external bitmap should be rotated?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetBitmapRotation(kG3dContourTemplate mask, k32s rotate);

/**
* Set type of to be measured sample.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   mode           Type of to be measured sample: 0 = small magnet 1 = big magnet 2 = unknown
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetSampleType(kG3dContourTemplate mask, GdkGlueBandDefaultMagnetType type);

/**
* Set sensor mode to determine, whether the one sensor or two sensors should be used for the evaluation.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   mode           Sensor mode
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetSensorMode(kG3dContourTemplate mask, GdkGlueBandSensorMode mode);

/**
* Set operation mode to determine, whether the template should be used for the evaluation or should be created.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   mode           Operation mode.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetOperationMode(kG3dContourTemplate mask, GdkGlueBandOperationMode mode);

/**
* Get the type of to be measured sample.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Type of to be measured sample.
*/
kVsFx(k32s) kG3dContourTemplate_SampleType(kG3dContourTemplate mask);

/**
* Get sensor mode which determines, whether the one sensor or two sensors should be used for the evaluation.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Sensor mode.
*/
kVsFx(k32s) kG3dContourTemplate_SensorMode(kG3dContourTemplate mask);

/**
* Get operation mode which determines, whether the template should be used for the evaluation or should be created.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Operation mode.
*/
kVsFx(k32s) kG3dContourTemplate_OperationMode(kG3dContourTemplate mask);

/**
* Set a boolean value to determine, whether the customized measuring order should be used.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   rotate         The customized measuring order should be used?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetUseCustomizedOrder(kG3dContourTemplate mask, kBool used);

/**
* Get a boolean value which determines, whether the customized measuring order should be used.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 The customized measuring order should be used?
*/
kVsFx(kBool) kG3dContourTemplate_UseCustomizedOrder(kG3dContourTemplate mask);

/**
* Get the current correction mode which determines, if the correction transfoamtion matrix should be used (0) or not be used (1)
* or be reset to identity.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @return                 Correction mode.
*/
kVsFx(k32s) kG3dContourTemplate_TransCorrectionMode(kG3dContourTemplate mask);

/**
* Set a value to determine, if the correction transfoamtion matrix should be used (0) or not be used (1) 
* or be reset to identity.
*
* @public                 @memberof kG3dContourTemplate
* @param   mask           kG3dContourTemplate object.
* @param   mode           Correction mode.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourTemplate_SetTransCorrectionMode(kG3dContourTemplate mask, k32s mode);

#include <kVision/G3d/kG3dContourTemplate.x.h>

#endif  /* KVISION_kG3DCONTOURTEMPLATE_H */
