/**
* @file    kG3dCommon.h
* @brief   Some commonly used functions.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_COMMON_H
#define KVISION_kG3D_COMMON_H

#include <kApi/kApiDef.h>
#include <kApi/Data/kMath.h>
#include <kApi/Data/kXml.h>
#include <kApi/Io/kFile.h>
#include <kApi/Data/kImage.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArray3.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kBytes.h>
#include <kApi/Data/kString.h>
#include <kApi/Threads/kTimer.h>
#include <kApi/Io/kPath.h>
#include <kApi/Threads/kThread.h>
#include <kApi/Io/kDirectory.h>
#include <kVision/S3d/kS3dVolumeCheckDef.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

#include <math.h>
#include <time.h>
#include <stdio.h>
//#include <conio.h>

/**
* @struct  kMCMatchPointPairs
* @ingroup kMC-Utilities
* @brief   Represents structure information for matching point pairs.
*/
typedef struct kMCMatchPointPairs
{
    kPoint3d64f ref;   // Coordinate of the reference point.
    kPoint3d64f meas;  // Coordinate of the measurement point.
    kPoint3d64f dev;   // Length of the difference vector.  
    k8u validX;
    k8u validY;
    k8u validZ;
    k8u valid;       // The pointpair was involved for the computation of transformation. 
}kMCMatchPointPairs;

#define kG3D_GRIDFFT_DEFAULT_WIDTH (1024)
#define kG3D_GRIDFFT_DEFAULT_EDGEWIDTH (60)
#define kG3D_GRIDFFT_DEFAULT_MIN_FREQUENCE (5)
#define kG3D_GRIDFFT_DEFAULT_EDGESHRINK (4)
#define kG3D_GRIDFFT_DEFAULT_MARKER_RANGE_IN_DEGREE (300)    

#define kG3D_SURFACE_RANGE_MIN (20)

#define kG3D_POLYNOMIALMAX_ORDER   (8)
#define kG3D_POLYNOMIAL_LINE_MAXCOUNT   (9)
#define kG3D_POLYNOMIAL8_COUNT     (45)

#define kG3D_SECTIONLINE_SUBPIXEL (5)    

#define kG3D_LINE_ADF_TIEFPASS_REPEAT (1)  

#define kG3D_BRC_PLATEAU_WINDOW_INPERCENT (40)

#define kG3D_SECTIONLINE_HISTOGRAM_COUNT   (101)
#define kG3D_SURFACE_HISTOGRAM_COUNT   (201)

#define kG3D_LATERAL_SCALE_MINIMUM (0.0001)
#define kG3D_Z_RESOLUTION_SCALEMINIMUM (0.0000001)
#define kG3D_Z_DEFAUT_PROFILE_NAME  ("Profile_%d")
#define kG3D_Z_DEFAUT_PROFILE_COUNT  (8)
#define kG3D_Z_DEFAUT_PROFILE_START_ANGLE  (0)

#define kG3D_AREA_COUNT_MIN (1)
#define kG3D_ROUGH_R3_CONST (3)
#define kG3D_ROUGH_RZISO_CONST (5)
#define kG3D_ROUGH_DEFAULT_MIN_DISTANCE (5)
#define kG3D_ROUGH_DEFAULT_MIN_POINTCOUNT (5)

typedef enum kG3dSurfaceOutputModeType
{
    kG3D_SURFACE_OUTPUT_ORIGINAL = 0,
    kG3D_SURFACE_OUTPUT_RESULT,
    kG3D_SURFACE_OUTPUT_DIFFERENCE,
    kG3D_SURFACE_OUTPUT_GEOMETRY,
    kG3D_SURFACE_CUT_DETECTION,
    kG3D_SURFACE_CUT_OUTSIDE,
    kG3D_SURFACE_SEGMENT
} kG3dSurfaceOutputModeType;

typedef enum kG3dFftFilterType
{
    kG3D_FFT_FILTER_CIRCLE_LORENZ_HIGH = 0,
    kG3D_FFT_FILTER_CIRCLE_GAUSS_HIGH = 1,
    kG3D_FFT_FILTER_CIRCLE_CUTOFF_HIGH = 2,

    kG3D_FFT_FILTER_RECTANGLE_LORENZ_HIGH = 10,
    kG3D_FFT_FILTER_RECTANGLE_GAUSS_HIGH = 11,
    kG3D_FFT_FILTER_RECTANGLE_CUTOFF_HIGH = 12,

    kG3D_FFT_FILTER_CIRCLE_LORENZ_LOW = 20,
    kG3D_FFT_FILTER_CIRCLE_GAUSS_LOW = 21,
    kG3D_FFT_FILTER_CIRCLE_CUTOFF_LOW = 22,

    kG3D_FFT_FILTER_RECTANGLE_LORENZ_LOW = 30,
    kG3D_FFT_FILTER_RECTANGLE_GAUSS_LOW = 31,
    kG3D_FFT_FILTER_RECTANGLE_CUTOFF_LOW = 32,


    kG3D_FFT_FILTER_MASK_MAP = 40
} kG3dFftFilterType;

typedef struct kLine64f
{
    k64f xStart;    
    k64f yStart;    
    k64f xEnd;     
    k64f yEnd;    
} kLine64f;

typedef struct kG3dColorCodingLevel
{
    k64f zmin;
    k64f zmax;
    k8u r;
    k8u g;
    k8u b;
}kG3dColorCodingLevel;


//#define KG3D_JAKOBEIGENVALUE_EPS   (0.00000000001)
//#define KG3D_JAKOBEIGENVALUE_MAXIMUM_PEPEAT   (30000)
//#define KG3D_JAKOBEIGENVALUE_ORDER             (4) 
//#define KG3D_JAKOBEIGENVALUE_MATRIXSIZE        (16) 

/**
* Return the corresponding position with maximum or minimum value from the three neighboring points.
*
* @ingroup             kMC-Utilities
* @param   A           Intensitiy of an adjacent point in an equidistant array with the index -1 from the viewing center.
* @param   B           Intensitiy of an adjacent point in an equidistant array with the index 0 from the viewing center.
* @param   C           Intensitiy of an adjacent point in an equidistant array with the index 1 from the viewing center.
* @return              Subpixel position relative to the viewing center.
*/
kVsFx(k64f) kG3dCommon_MaxPosition3(k64f A, k64f B, k64f C);

/**
* Return the interpolated value at a given point in a kArray2 map preferably in the x direction
*
* @ingroup             kMC-Utilities
* @param   data        input kArray2 object of k32f
* @param   kx          integer coordinate in x direction
* @param   ky          integer coordinate in y direction
* @param   x           Fraction position between 0 and 1 in x direction
* @param   y           Fraction position between 0 and 1 in y direction
* @return              interpolated value.
*/
kVsFx(k32f) kG3dCommon_InterpolateXFromArray2Point3d32f(kArray2 data, k32s kx, k32s ky, k32f x, k32f y);

/**
* Return the quadratic interpolated average value at a given point in a kArray2 map. kArray2<k32f> version
*
* @ingroup             kMC-Utilities
* @param   data        input kArray2 object of k32f
* @param   kx          integer coordinate in x direction
* @param   ky          integer coordinate in y direction
* @param   x           Fraction position between 0 and 1 in x direction
* @param   y           Fraction position between 0 and 1 in y direction
* @return              interpolated value.
*/
kVsFx(k32f) kG3dCommon_InterpolateFromArray2k32f(kArray2 data, k32s kx, k32s ky, k32f x, k32f y);

/**
* Return the quadratic interpolated average value at a given point in a kArray2 map. kArray2 of 32s version
*
* @ingroup             kMC-Utilities
* @param   data        input kArray2 object of k32s
* @param   kx          integer coordinate in x direction
* @param   ky          integer coordinate in y direction
* @param   x           Fraction position between 0 and 1 in x direction
* @param   y           Fraction position between 0 and 1 in y direction
* @return              interpolated value.
*/
kVsFx(k32f) kG3dCommon_InterpolateFromArray2k32s(kArray2 data, k32s kx, k32s ky, k32f x, k32f y);

/**
* Return the linear interpolated average value at a given point in a kArray2 map. kArray2 of k32s version
*
* @ingroup             kMC-Utilities
* @param   data        input kArray2 object of k32s
* @param   kx          integer coordinate in x direction
* @param   ky          integer coordinate in y direction
* @param   x           Fraction position between 0 and 1 in x direction
* @param   y           Fraction position between 0 and 1 in y direction
* @return              interpolated value.
*/
kVsFx(k32f) kG3dCommon_InterpolateFromArray2k32s1(kArray2 data, k32s kx, k32s ky, k32f x, k32f y);

/**
* Return the linear interpolated average value at a given point in a kArray2 map with mask map.
*
* @ingroup             kMC-Utilities
* @param   data        input kArray2 of k32s
* @param   pow         Mask map of kArray2 of 8u
* @param   x           Coordinate in x direction
* @param   y           Coordinate in y direction
* @return              interpolated value.
*/
kVsFx(k32f) kG3dCommon_InterpolateFromValidArray232s1(kArray2 data, kArray2 pow, k32f x, k32f y);

/**
* Return the quadratic interpolated average value with range limitation at a given point in a kArray2 map. kArray2 of k32s version
*
* @ingroup             kMC-Utilities
* @param   data        input kArray2 object of k32s
* @param   kx          integer coordinate in x direction
* @param   ky          integer coordinate in y direction
* @param   x           Fraction position between 0 and 1 in x direction
* @param   y           Fraction position between 0 and 1 in y direction
* @return              interpolated value.
*/
kVsFx(k32f) kG3dCommon_InterpolateFromArray2RestrictK32s(kArray2 data, k32s kx, k32s ky, k32f x, k32f y);

/**
* Return the quadratic interpolated average value at a given point in a kArray1 curve. kArray1<k64f> version
*
* @ingroup             kMC-Utilities
* @param   data        input kArray1 object of k32s
* @param   kx          integer coordinate.
* @param   dx          Fraction position between 0 and 1
* @return              interpolated value.
*/
kVsFx(k64f) kG3dCommon_InterpolateFromArray1K64f(kArray1 data, k32s kx, k64f dx);

/**
* Return the quadratic interpolated average value at a given point in a kArray1 curve. kArray1<kPoint64f> version
*
* @ingroup             kMC-Utilities
* @param   data        input kArray1 object of k32s
* @param   kx          integer coordinate.
* @param   dx          Fraction position between 0 and 1
* @return              interpolated value.
*/
kVsFx(k64f) kG3dCommon_InterpolateFromArray1KPoint64f(kArray1 data, k32s kx, k64f dx);

#include <kVision/G3d/kG3dCommon.x.h>

#endif /* #ifndef KVISION_kG3D_COMMON_H */
