/**
* @file    ${filename_header_public}}
* @brief   Declares the ${classname} class.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kVS_SPARSE_MATRIX_H
#define kVS_SPARSE_MATRIX_H

#include <kVision/Common/kVision.h>
//TODO add any #includes here

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

/**
* @class       kSparseMatrix
* @extends     kObject
* @ingroup     kVision-Common
* @brief       TODO: write this
*/

typedef kObject kSparseMatrix; // TODO templates for index and value

typedef struct kSparseMatrixTriplet
{
    k32u row;
    k32u column;
    k64f value;

} kSparseMatrixTriplet;

kDeclareValueEx(kVs, kSparseMatrixTriplet, kValue)

/**
 * Constructs a kSparseMatrix object.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Destination for the constructed object handle.
 * @param   allocator        Memory allocator (or kNULL for default).
 * @return                   Operation status.
 */

kVsFx(kStatus) kSparseMatrix_Construct(kSparseMatrix* matrix, kAlloc allocator);

/**
 * Construct sparce matrix from a set of triplets (row, column, value).
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   triplets         kArrayList of kSparseMatrixTriplet
 * @param   rowCount         Matrix row count
 * @param   columnCount      Matrix column count
 * @return                   Operation status.
 */

kVsFx(kStatus) kSparseMatrix_Import(kSparseMatrix matrix, kArrayList triplets, kSize rowCount, kSize columnCount);

/**
 * Matrix dimensions (rows, columns)
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @return                   Value.
 */

kInlineFx(kSize) kSparseMatrix_RowCount(kSparseMatrix matrix);
kInlineFx(kSize) kSparseMatrix_ColumnCount(kSparseMatrix matrix);

/**
 * Return the index (position in the entry array) of the first entry a specified column.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   column           Column number
 * @return                   Index (position).
 */

kInlineFx(kSize) kSparseMatrix_ColumnIndex(kSparseMatrix matrix, kSize column);

/**
 * Number of entries (non zero elements) in a specified column.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   column           Column number
 * @return                   Number of entries (nnz count).
 */

kInlineFx(kSize) kSparseMatrix_ColumnEntriesCount(kSparseMatrix matrix, kSize column);

/**
 * Return the value of an entry in question.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   index            Index in quastion (position)
 * @return                   Value of an entry.
 */

kInlineFx(k64f) kSparseMatrix_EntryValueAt(kSparseMatrix matrix, kSize index);

/**
 * Return the row number of an entry in question.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   index            Index in quastion (position)
 * @return                   Row number of an entry.
 */

kInlineFx(kSize) kSparseMatrix_EntryRowAt(kSparseMatrix matrix, kSize index);

/**
 * Calculates the number of entries (nnz values) in the matrix.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @return                   Non zero number
 */

kInlineFx(kSize) kSparseMatrix_NonZeroCount(kSparseMatrix matrix); // max == rows * cols 

/**
 * Inserts a new entry (value) into the matrix.
 * Matrix has to have space reserved.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   row              Which row the value goes to
 * @param   column           Which column
 * @return                   Operation status.
 */

kInlineFx(kStatus) kSparseMatrix_Insert(kSparseMatrix matrix, kSize row, kSize column, k64f value);

/**
 * Creates an empty matrix of given dimentions.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   rowCount         Number of rows
 * @param   columnCount      Number of columns
 * @return                   Operation status.
 */

kVsFx(kStatus) kSparseMatrix_Resize(kSparseMatrix matrix, kSize rowCount, kSize columnCount); // resize and create empty

/**
 * Reserves entries per column for subsequent insertion.
 *
 * @public                   @memberof kSparseMatrix
 * @param   matrix           Matrix object
 * @param   reserveSizes     kArray1 of k32u specifying number of entries per column to reserve
 * @param   columnCount      Number of columns
 * @return                   Operation status.
 */

kVsFx(kStatus) kSparseMatrix_Reserve(kSparseMatrix matrix, kArray1 reserveSizes); // nnz per column

/**
 * Transposes the matrix: inMatrix^T = outMatrix
 *
 * @public                   @memberof kSparseMatrix
 * @param   inMatrix         Input matrix to transpose
 * @param   outMatrix        Output matrix (can't be the same)
 * @return                   Operation status.
 */

kVsFx(kStatus) kSparseMatrix_Transpose(kSparseMatrix inMatrix, kSparseMatrix outMatrix);

/**
* Multiplies sparse matrices:  inMatrixLeft * inMatrixRight = outMatrix
*
* @public                   @memberof kSparseMatrix
* @param   inMatrixLeft     Input left matrix (sparce)
* @param   inMatrixRight    Input right matrix (sparce)
* @param   outMatrix        Output matrix (sparce)
* @return                   Operation status.
*/

kVsFx(kStatus) kSparseMatrix_MultiplySparseSparse(kSparseMatrix inMatrixLeft, kSparseMatrix inMatrixRight, kSparseMatrix outMatrix);

/**
* Multiplies sparse and dense matrices:  inMatrixLeft * inMatrixRight = outMatrix
*
* @public                   @memberof kSparseMatrix
* @param   inMatrixLeft     Input left matrix (sparce)
* @param   inMatrixRight    Input right matrix (dense) 
* @param   outMatrix        Output matrix (dense)
* @return                   Operation status.
*/

kVsFx(kStatus) kSparseMatrix_MultiplySparseDense(kSparseMatrix inMatrixLeft, kArray2 inMatrixRight, kArray2 outMatrix);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/Common/kSparseMatrix.x.h>

#endif  /* kVS_SPARSE_MATRIX_H */
