/**
 * @file    ${filename_header_public}}
 * @brief   Declares the ${classname} class.
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kVS_SOLVER_H
#define kVS_SOLVER_H

#include <kVision/Common/kVision.h>
#include <kVision/Common/kSparseMatrix.h>

/**
 * A direct sparse LDLT Cholesky factorizations without square root.
 * This class provides a LDL^T Cholesky factorizations without square root of sparse matrices that are selfadjoint and positive definite.
 * The factorization allows for solving A.X = B
 * http://eigen.tuxfamily.org/dox/classEigen_1_1SimplicialLDLT.html
 *
 *
 * mA(m,n) * mX(n,k) = mB(m,k) // solve for mX
 *
 * @param   mA               an input kSparseMatrix of size(m,n)  defining the left hand side ofthe linear system
 * @param   mX               an output kArray2 of size(n, k) used to store the solution
 * @param   mB               an input kArray2 of size(m,k) defining the right hand side of the linear system
 * @return                   Operation status.
 */

kVsFx(kStatus) kSolver_SimplicialLdlt(kSparseMatrix mA, kArray2 mX, kArray2 mB);

/**
 * mA(m,n) * mX(n,k) = mB(m,k) // solve for mX
 * where triplets define the mA matrix for fast sparse matrix construction
 *
 * @param   triplets         an input array of triplets defining a sparse matrix mA which is the left hand side ofthe linear system
 * @param   rowCount         rowCount == m
 * @param   columnCount      columnCount == n
 * @param   mX               an output kArray2 of size(n, k) used to store the solution
 * @param   mB               an input kArray2 of size(m,k) defining the right hand side of the linear system
 * @return                   Operation status.
 */

kVsFx(kStatus) kSolver_SimplicialLdltFromTriplets(kArrayList triplets, kSize rowCount, kSize columnCount, kArray2 mX, kArray2 mB);

kVsFx(kStatus) kPoint_Test(kArray2 map);

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#include <kVision/Common/kSolver.x.h>

#endif  /* kVS_SOLVER_H */
