/** 
 * @file    kZstdStream.h
 * @brief   Declares the kZstdStream class. 
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc.
  */
#ifndef K_ZSTD_STREAM_H
#define K_ZSTD_STREAM_H

#include <kApi/kApiDef.h>
#include <kApi/Io/kStream.h>
#include <kFireSync/kFsDef.h>

 /**
 * @struct  kZstdMode
 * @extends kValue
 * @ingroup kFireSync-Utils
 * @brief   Flags to control kZstdStream operation modes.
 */
typedef k32s kZstdMode;

/** @relates kZstdMode @{ */
#define kZSTD_MODE_READ            (0x01)    ///< Open the stream with the ability to read compressed data.
#define kZSTD_MODE_WRITE           (0x02)    ///< Open the stream with the ability to write compressed data.
/** @} */

 /**
 * @struct  kZstdPreset
 * @extends kValue
 * @ingroup kFireSync-Utils
 * @brief   Preset Zstandard compression levels. 
 */
typedef k32s kZstdPreset;

/** @relates kZstdPreset @{ */
#define kZSTD_PRESET_MIN       (1)     ///< Minimum compression supported by Zstandard.
#define kZSTD_PRESET_FAST      (1)     ///< Recommended setting for fast compression.
#define kZSTD_PRESET_DEFAULT   (10)    ///< Recommended setting for a balance of speed and density.
#define kZSTD_PRESET_DENSE     (19)    ///< Recommended setting for dense compression.
#define kZSTD_PRESET_MAX       (22)    ///< Maximum compression supported by Zstandard.
/** @} */

/**
 * @class       kZstdStream
 * @extends     kStream
 * @implements  kCompressor
 * @ingroup     kFireSync-Utils
 * @brief       Implements the Zstandard format for compression and decompression. 
 *
 * The kZstdStream class implements the Zstandard (zstd) compression algorithm. Data is compressed 
 * as it is written and/or decompressed as it is read. The parsing/formatting algorithms implemented 
 * by kZstdStream are compatible with Zstandard-aware compression utilities such as zstd on Linux and 
 * 7zip-zstd (a custom build of 7zip that supports newer codecs) on Windows.
 *
 * The zstd algorithm offers a wide range of speeds and densities through 22 compression presets. 
 * Lower presets are generally faster, while higher presets offer better compression density. 
 * The kZstdPreset enumeration defines numeric limits (i.e., kZSTD_PRESET_MIN, kZSTD_PRESET_MAX)
 * and recommended values for compression presets (i.e., kZSTD_PRESET_FAST, kZSTD_PRESET_DEFAULT, 
 * kZSTD_PRESET_DENSE). 
 * 
 * The amount of memory required for compression varies considerably with the preset, as summarized 
 * in the following table.  
 *
 * Preset | Compression | Decompression
 * :----: | :---------: | :-----------:
 * 1      | 1.4 MB      | 0.3 MB
 * 3      | 2.6 MB      | 0.3 MB
 * 5      | 3.4 MB      | 0.3 MB
 * 7      | 6.0 MB      | 0.3 MB
 * 9      | 9.0 MB      | 0.3 MB 
 * 11     | 24.5 MB     | 0.3 MB 
 * 13     | 29.5 MB     | 0.3 MB 
 * 17     | 50.0 MB     | 0.3 MB 
 * 19     | 59.0 MB     | 0.3 MB 
 * 20     | 198 MB      | 0.3 MB 
 * 22     | 788 MB      | 0.3 MB 
 * 
 * The examples below illustrate the use of kZstdStream. The output of the WriteCompressedObject 
 * sample function would be a ".kdat6.zst" file -- a serialized kdat6 object, encapsulated in a 
 * compressed zst file. This stream-layering approach can be used to compress and decompress 
 * arbitrary content. (An alternative to the stream-layering approach is discussed after these 
 * examples.)
 *
@code {.c}
kStatus WriteCompressedObject(kObject source, const kChar* path)
{
    kFile file = kNULL;
    kZstdStream zstd = kNULL;
    kSerializer writer = kNULL;

    kTry
    {
        //open a file for writing
        kTest(kFile_Construct(&file, path, kFILE_MODE_WRITE, kNULL));

        //construct a kZstdStream object; use fast compression
        kTest(kZstdStream_Construct(&zstd, file, kZSTD_MODE_WRITE, kZSTD_PRESET_FAST, kNULL));

        //construct a kDat6Serializer object
        kTest(kDat6Serializer_Construct(&writer, zstd, kNULL));

        //serialize the object to file
        kTest(kSerializer_WriteObject(writer, source));
    }
    kFinally
    {
        //tear down in correct order
        kObject_Destroy(writer);
        kObject_Destroy(zstd);
        kObject_Destroy(file);

        kEndFinally();
    }

    return kOK;
}

kStatus ReadCompressedObject(kObject *dest, const kChar* path)
{
    kFile file = kNULL;
    kZstdStream zstd = kNULL;
    kSerializer reader = kNULL;

    kTry
    {
        //open a file for writing
        kTest(kFile_Construct(&file, path, kFILE_MODE_READ, kNULL));

        //construct a kZstdStream object
        kTest(kZstdStream_Construct(&zstd, file, kZSTD_MODE_READ, 0, kNULL));

        //construct a kDat6Serializer object
        kTest(kDat6Serializer_Construct(&reader, zstd, kNULL));

        //deserialize the object from file
        kTest(kSerializer_ReadObject(reader, dest, kNULL));
    }
    kFinally
    {
        //tear down in correct order
        kObject_Destroy(reader);
        kObject_Destroy(zstd);
        kObject_Destroy(file);

        kEndFinally();
    }

    return kOK;
}
@endcode
*
* While the stream-layering approach is flexible and efficient, it requires the reader and writer to 
* coordinate on the compression approach. To simplify usage, the kDat6Serializer class offers built-in 
* support for compression algorithms present in the FireSync platform. A version of the WriteCompressedObject 
* sample function, revised to make use of kDat6Serializer's built in compression support, is shown below. 
* The output from this revised example is a "kdat6" file -- a serialized kdat6 object that, internally, 
* contains compressed content.
*
@code {.c}
kStatus WriteCompressedObject(kObject source, const kChar* path)
{
    kFile file = kNULL;
    kSerializer writer = kNULL;

    kTry
    {
        //open a file for writing
        kTest(kFile_Construct(&file, path, kFILE_MODE_WRITE, kNULL));

        //construct a kDat6Serializer object
        kTest(kDat6Serializer_Construct(&writer, file, kNULL));

        //configure compression; note, generic platform compression presets can be used here
        kTest(kDat6Serializer_EnableCompression(writer, kCOMPRESSION_TYPE_ZSTD, kCOMPRESSION_PRESET_DEFAULT));

        //serialize the object to file
        kTest(kSerializer_WriteObject(writer, source));
    }
    kFinally
    {
        //tear down in correct order
        kObject_Destroy(writer);
        kObject_Destroy(file);

        kEndFinally();
    }

    return kOK;
}

kStatus ReadCompressedObject(kObject *dest, const kChar* path)
{
    kFile file = kNULL;
    kSerializer reader = kNULL;

    kTry
    {
        //open a file for writing
        kTest(kFile_Construct(&file, path, kFILE_MODE_READ, kNULL));

        //construct a kDat6Serializer object; no need to specify decompression algorithm
        kTest(kDat6Serializer_Construct(&reader, file, kNULL));

        //deserialize the object from file
        kTest(kSerializer_ReadObject(reader, dest, kNULL));
    }
    kFinally
    {
        //tear down in correct order
        kObject_Destroy(reader);
        kObject_Destroy(file);

        kEndFinally();
    }

    return kOK;
}
@endcode
*
* For applications that only need to write a single compressed object to a "kdat6" file, the kDat6Serializer
* class provides the kDat6Serializer_SaveCompressed method, which can be used to implement the example
* above in a single line of code. Content written using kDat6Serializer_SaveCompressed can be read back
* using standard platform utility methods such as kLoad6.
*
* Note that the layered-streams approach will usually produce more dense/compact compression because it enables
* the algorithm to make use of context <em>across</em> objects. In contrast, when using the built-in 
* compression features in kDat6Serializer, each object is compressed individually. If the source objects are large, 
* there will be little difference in compression density between the two approaches. But if the source objects 
* are small, the layered-streams approach may yield better results.
* 
* The kZstdStream class is available only on host platforms (i.e., Windows, Linux). If support is desired 
* on embedded platforms, please contact FSS to discuss requirements.  
*/

 //typedef kStream kZstdStream;      --forward-declared in kFsDef.x.h 

/** 
 * Constructs a kZstdStream object.
 * 
 * The kZstdStream class requires that its underlying stream object (e.g., kFile) should provide a 
 * a read buffer (if read operations are to be performed). Configure the stream to allocate a read buffer 
 * before passing the stream to this constructor. 
 * 
 * @public                      @memberof kZstdStream
 * @param   zStream             Destination for the constructed object handle. 
 * @param   stream              Underlying stream; see note above on read buffer requirement.
 * @param   mode                Specifies how to construct the stream (reading, writing or both). 
 * @param   level               Compression preset level for writing; accepts kZstdPreset, kCompressionPreset, or integer values.
 * @param   allocator           Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kZstdStream_Construct(kZstdStream* zStream, kStream stream, kZstdMode mode, k32s level, kAlloc allocator); 

/** 
 * Finishes writing the current Zstandard compression frame. 
 * 
 * If the stream was opened in write mode, this method finalizes the current Zstandard compression frame.
 * Output is flushed to the underlying stream, but the underlying stream itself is not flushed. 
 *
 * Additional writes can be performed after calling this method; the additional data will 
 * be part of a new Zstandard frame. 
 *
 * After calling this method, operations (e.g., seek) can be safely performed on the underlying stream
 * until such time as another kZstdStream write is performed. 
 *
 * This method will be called automatically in the kZstdStream destructor, closing the current 
 * frame (if any). To avoid performing writes in the destructor (and/or to detect errors while closing 
 * the frame), this method can optionally be called explicitly before destroying the compression stream.
 * 
 * @public                      @memberof kZstdStream
 * @param   zStream             Stream object.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kZstdStream_FinishWrite(kZstdStream zStream);

/** 
 * Finishes reading the current Zstandard compression frame. 
 * 
 * If the stream was opened in read mode, this method consumes and discards the remainder of the current 
 * Zstandard compression frame.   
 
 * Use of this method is optional. Additional reads can be performed after calling this method, 
 * if additional Zstandard frames are present in the underlying stream. 
 * 
 * After calling this method, operations (e.g., seek) can be safely performed on the underlying stream
 * until such time as another kZstdStream read is performed. 
 *
 * @public                      @memberof kZstdStream
 * @param   zStream             Stream object.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kZstdStream_FinishRead(kZstdStream zStream);

#include <kFireSync/Utils/kZstdStream.x.h>

#endif
