/** 
 * @file    kTar.h
 * @brief   Class for creating and extracting tar files. 
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.
  */
#ifndef K_FIRESYNC_TAR_H
#define K_FIRESYNC_TAR_H

#include <kFireSync/kFsDef.h>

#include <kApi/Io/kFile.h>
#include <kApi/Io/kPath.h>

 /**
 * @struct  kTarMode
 * @extends kValue
 * @ingroup kFireSync-Utils
 * @brief   Flags to control kTar operation modes.
 */
typedef k32s kTarMode;

/** @relates kTarMode @{ */
#define kTAR_MODE_READ            (0x01)        ///< Open the stream with the ability to read data.
#define kTAR_MODE_WRITE           (0x02)        ///< Open the stream with the ability to write data.
/** @} */

#include <kFireSync/Utils/kTar.x.h>

/**
 * @class       kTar
 * @extends     kObject
 * @ingroup     kFireSync-Utils
 * @brief       Class to create and extract tar archives. 
 *
 * kTar class can be used to create or extract tar archives. 
 *
 * There are several tar formats. This class supports the ustar (IEEE Std 1003.1-1988) and the 
 * pax format (IEEE Std 1003.1-2001). It always uses the pax format when creating archives.
 *
 * The class provides some convenient functions to archive a file or directory and to extract
 * a stream or a file. To gain finer control, kTar_BeginWrite and kTar_BeginRead can be used
 * for creating an archive or extracting an archive, respectively.
 * 
 * The following example illustrates the extraction of a tar file using kTar_BeginRead and 
 * streaming operations.
@code {.c}
kStatus ReadTarFile(kStream stream)
{
    kTar tar = kNULL;
    kChar name[kPATH_MAX];
    kBool isDirectory;
    k64u itemSize;
    kByte b;

    kCheck(kTar_Construct(&tar, stream, kTAR_MODE_READ, kNULL));

    while (kSuccess(kTar_BeginRead(tar, name, kCountOf(name), &isDirectory, &itemSize)))
    {
        kLogf("%s: %s, size = %llu", name, isDirectory ? "directory" : "file", itemSize);

        for (kSize i = 0; i < itemSize; i++)
        {
            kCheck(kStream_Read(tar, &b, 1));
        }
    }

    return kOK;
}
@endcode
 * 
 * The following example illustrates how to create a tar file using kTar_BeginWrite and
 * streaming operations.
 @code {.c}
kStatus Archive()
{
    kFile outFile = kNULL;
    kChar fileName[kPATH_MAX];
    kTar tar = kNULL;
    const kSize itemSize = 100;
    const kChar buffer[itemSize] = { 'b' };

    kTry
    {
        kTest(kStrCopy(fileName, kCountOf(fileName), "archive.tar"));

        kTest(kFile_Construct(&outFile, fileName, kFILE_MODE_WRITE, kNULL));
        kTest(kFile_SetWriteBuffer(outFile, 64 * 1024));

        kTest(kTar_Construct(&tar, outFile, kTAR_MODE_WRITE, kNULL));

        kTest(kTar_BeginWrite(tar, "1", kTRUE, 0));
        kTest(kTar_BeginWrite(tar, "1/2", kTRUE, 0));

        kTest(kTar_BeginWrite(tar, "3/4/file", kFALSE, itemSize));

        kTest(kStream_Write(tar, buffer, itemSize));
    }
    kFinally
    {
        kCheck(kObject_Destroy(tar));
        kCheck(kObject_Destroy(outFile));

        kEndFinally();
    }

    return kOK;
}
@endcode
 */
//typedef kObject kTar;      --forward-declared in kFsDef.x.h


/**
* Archives a file or directory to a destination stream.
*
* This convenience method archives a file or directory. If sourcePath is a directory, the directory 
* itself is archived with path names relative to (and including) the top level directory name.
*
* @public                      @memberof kTar
* @param   sourcePath          Source file or directory.
* @param   destStream          Destination stream.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_Archive(const kChar* sourcePath, kStream destStream)
{
    return xkTar_Archive(sourcePath, destStream);
}

/**
* Archives a file or directory to a file.
*
* This method archives a file or directory. If sourcePath is a directory, the directory itself 
* is archived with path names relative to (and including) the top level directory name.
*
* @public                      @memberof kTar
* @param   sourcePath          Source file or directory.
* @param   destFilePath        Destination file path.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_Archive(const kChar* sourcePath, const kChar* destFilePath)
{
    kFile stream = kNULL;

    kTry
    {
        kTest(kFile_Construct(&stream, destFilePath, kFILE_MODE_WRITE, kNULL));

        kTest(kTar_Archive(sourcePath, stream));
    }
    kFinally
    {
        kCheck(kObject_Destroy(stream));

        kEndFinally();
    }

    return kOK;
}

/**
* Extracts a tar archive from a stream to a destination path.
*
* @public                      @memberof kTar
* @param   source              Source stream.
* @param   destParent          Parent directory into which the archive will be extracted.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_Extract(kStream source, const kChar* destParent)
{
    return xkTar_Extract(source, destParent);
}

/**
* Extracts a tar archive from a file path to a destination path.
*
* @public                      @memberof kTar
* @param   sourcePath          Path of the source file.
* @param   destParent          Parent directory into which the archive will be extracted.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_Extract(const kChar* sourcePath, const kChar* destParent)
{
    kFile stream = kNULL;

    kTry
    {
        kTest(kFile_Construct(&stream, sourcePath, kFILE_MODE_READ, kNULL));

        kTest(kTar_Extract(stream, destParent));
    }
    kFinally
    {
        kCheck(kObject_Destroy(stream));

        kEndFinally();
    }

    return kOK;
}

/** 
 * Constructs a kTar object.
 *
 * The tar archive is read from or written to the destination stream. 
 * 
 * It is the caller's reponsiblity to destroy the stream object passed to this constructor. However, 
 * the caller may not read from, write to, or destroy the stream object during the lifetime of the 
 * kTar object. 
 * 
 * @public                      @memberof kTar
 * @param   tar                 Destination for the constructed object handle. 
 * @param   stream              Stream.
 * @param   mode                Operation mode (kTAR_MODE_READ or kTAR_MODE_WRITE).
 * @param   allocator           Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kTar_Construct(kTar* tar, kStream stream, kTarMode mode, kAlloc allocator);

/**
* Prepares a kTar object opened in write mode for the next item.
*
* After a new item is started, the caller must write the specified number of bytes to the kTar object.
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   itemPath            Path of the item.
* @param   isDirectory         Is object a directory.
* @param   itemSize            Size of the item.
* @return                      Operation status.
*/
kFsFx(kStatus) kTar_BeginWrite(kTar tar, const kChar* itemPath, kBool isDirectory, k64u itemSize);

/**
* Adds a file to the tar archive by reading the file contents from a source stream.  
*
* This convenience method can be used instead of the combination of kTar_BeginWrite and 
* streaming operations.
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   itemPath            Relative path of the item in the kTar object.
* @param   stream              Source stream.
* @param   fileSize            Size of the item.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_WriteFile(kTar tar, const kChar* itemPath, kStream stream, k64u fileSize)
{
    return xkTar_WriteFile(tar, itemPath, stream, fileSize);
}

/**
* Adds a file to the tar archive by reading the file contents from a source file.  
*
* This convenience method can be used instead of the combination of kTar_BeginWrite and 
* streaming operations.
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   itemPath            Relative path of the item in the kTar object.
* @param   filePath            File system path to the source file.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_WriteFile(kTar tar, const kChar* itemPath, const kChar* filePath)
{
    kFile file = kNULL;
    k64u itemSize = kFile_Size(filePath);

    kTry
    {
        kTest(kFile_Construct(&file, filePath, kFILE_MODE_READ, kObject_Alloc(tar)));

        kTest(kTar_WriteFile(tar, itemPath, file, itemSize));
    }
    kFinally
    {
        kCheck(kObject_Destroy(file));

        kEndFinally();
    }

    return kOK;
}

/**
* Adds a directory entry to an archive.
*
* This convenience method can be used instead of kTar_BeginWrite to add a directory 
* to an archive.
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   itemPath            Relative path of the directory item in the kTar object.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_WriteDirectory(kTar tar, const kChar* itemPath)
{
    return kTar_BeginWrite(tar, itemPath, kTRUE, 0);
}

/**
* Recursively adds a directory and its contents to an archive.
*
* This convenience method recursively adds items from the directory path, using
* itemPath as the relative base directory name within the tar archive. It can be used 
* in place of multiple kTar_BeginWrite() calls and associated streaming operations.
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   itemPath            Relative path of the directory item in the kTar object.
* @param   directoryPath       File system path of the source directory to be added.
* @return                      Operation status.
*/
kInlineFx(kStatus) kTar_WriteDirectory(kTar tar, const kChar* itemPath, const kChar* directoryPath)
{
    return xkTar_WriteDirectory(tar, itemPath, directoryPath);
}

/**
* Closes a tar stream opened in write mode.
*
* This method finalizes the current tar stream. Output is flushed to the underlying stream, but 
* the underlying stream itself is not flushed. 
* 
* It is not necessary to call this method directly; if this method has not been called, it will be called 
* automatically when the kTar object is destroyed.
* 
* If other kTar methods are called after closing the archive (aside from kObject_Destroy), undefined 
* behavior will result.
* 
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @return                      Operation status.
*/
kFsFx(kStatus) kTar_Close(kTar tar);

/**
* Seeks to the next item in a kTar object opened in read mode.
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   itemPath            Receives the name of the item.
* @param   itemPathCapacity    Maximum number of characters (including null terminator) for item name.
* @param   isDirectory         Receives if item is a directory.
* @param   fileSize            Receives size of the item.
* @return                      Operation status.
*/
kFsFx(kStatus) kTar_BeginRead(kTar tar, kChar* itemPath, kSize itemPathCapacity, kBool* isDirectory, k64u* fileSize);

/**
* Convenience method to find a file or directory in an archive.
*
* This method finds a file or directory in an archive. It can be used instead of manually iterating over all 
* items using kTar_BeginRead. Similar to kTar_BeginRead, it can only forward-seek. 
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   itemPath            Relative path of the item in the kTar object.
* @param   isDirectory         Receives if object a directory.
* @param   fileSize            Receives size of the item.
* @return                      Operation status (kERROR_NOT_FOUND if item not found).
*/
kFsFx(kStatus) kTar_Find(kTar tar, const kChar* itemPath, kBool* isDirectory, k64u* fileSize);

/**
* Convenience method to read a file from an archive and save it to a file system path.
*
* This method reads a file from an archive. It can be used instead of kTar streaming operations 
* after calling kTar_BeginRead (or kTar_Find) to save a tar file item directly to a local file.
*
* @public                      @memberof kTar
* @param   tar                 kTar object.
* @param   filePath            Relative path of the item in the kTar object.
* @param   fileSize            Size of the item.
* @return                      Operation status.
*/
kFsFx(kStatus) kTar_ReadFile(kTar tar, const kChar* filePath, k64u fileSize);

#endif
