/** 
 * @file    kFsUtils.h
 * @brief   Utility functions.
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_UTILS_H
#define K_FIRESYNC_UTILS_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/Utils/kInfo.h>
#include <kFireSync/kNodeDef.h>

/** 
 * Callback signature used with the kLoadFromFile function.
 * 
 * @param   object    Receives object loaded from stream.
 * @param   sender    Stream from which object should be read. 
 * @param   args      Memory allocator (or kNULL for default). 
 * @return            Operation status. 
 */
typedef kStatus (kCall* kLoadStreamFx)(kObject* object, kStream stream, kAlloc allocator);

/** 
 * Callback signature used with the kSaveToFile function.
 * 
 * @param   object    Object to be written to stream.
 * @param   sender    Stream to which object should be written.
 * @return            Operation status. 
 */
typedef kStatus (kCall* kSaveStreamFx)(kObject object, kStream stream);

/**
 * @class   kFsUtils
 * @ingroup kFireSync-Utils
 * @brief   Collection of utility functions. 
 */

/** 
 * Loads a file and parses a user-defined object from the file using the specified callback. 
 *
 * @public                  @memberof kFsUtils
 * @param   object          Receives constructed object. 
 * @param   path            Path to file.
 * @param   loadFunction    Callback function to parse object from file stream.
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kLoadFromFile(kObject* object, const kChar* path, kLoadStreamFx loadFunction, kAlloc allocator);

/** 
 * Formats a user-defined object using the specified callback and saves the result to file.
 *
 * @public                  @memberof kFsUtils
 * @param   object          Object to save.
 * @param   path            Path to file.
 * @param   saveFunction    Callback function to format object to file stream.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kSaveToFile(kObject object, const kChar* path, kSaveStreamFx saveFunction);

/** 
 * Counts the number of bits that are set in a 32-bit unsigned integer.
 *
 * @public          @memberof kFsUtils
 * @param   value   Input value. 
 * @return          Count of '1' bits in input value. 
 */
kFsFx(k32u) kCountBits32u(k32u value); 

/** 
 * This function returns the fully qualified (absolute) path for a given program name
 * based on the controller type.
 *
 * @public                  @memberof kFsUtils
 * @param   programName     Program name, e.g. "kTestApp", "kFireSyncApp" or "kSensorApp".
 * @param   type            Controller type.
 * @param   isDebug         Load debug firmware.
 * @param   path            Receives path of the program name including file name.
 * @param   capacity        Maximum number of characters (including null terminator).   
 * @return                  Operation status. 
 */
kFsFx(kStatus) kFsUtils_FindPsProgram(const kChar* programName, kControllerType type, kBool isDebug, kChar* path, kSize capacity);

/** 
 * Gives the path and the version of a Pl file given a specific configuration.
 *
 * @public                   @memberof kFsUtils
 * @param   configuration    Pl configuration number.
 * @param   expectingVersion Search for specified PL version.
 * @param   path             Receives path of the Pl file.
 * @param   capacity         Maximum number of characters (including null terminator).
 * @param   version          Receives Pl version.
 * @return                   Operation status. 
 */
kFsFx(kStatus) kFindPlProgram(k32u configuration, kVersion expectingVersion, kChar* path, kSize capacity, kVersion* version);

/** 
 * Obtain the absolute path to the OS firmware program and its version.
 *
 * @public                   @memberof kFsUtils
 * @param   type             Controller type.
 * @param   path             Receives path of the OS firmware file.
 * @param   capacity         Maximum number of characters (including null terminator).
 * @param   version          Receives OS version (can be kNULL).
 * @return                   Operation status. 
 */
kFsFx(kStatus) kFindOsProgram(kControllerType type, kChar* path, kSize capacity, kVersion* version);

/** 
 * Calculates the expected compressed subframe capacity given a desired number of subframes and image size.
 * 
 * This method can be used with the compression feature of kCameraPhaseDecoder to calculate the expected 
 * subframe capacity, given specific inputs.
 *
 * @public                          @memberof kFsUtils
 * @param   desiredSubframeCount    Desired number of subframes.
 * @param   pixelCount              Uncompressed image size, in pixels. 
 * @param   subframeCount           Receives actual subframe count.
 * @param   subframeCapacity        Receives subframe capacity. 
 * @return                          Operation status.
 */
kFsFx(kStatus) kCalculateSubframe(kSize desiredSubframeCount, kSize pixelCount, kSize* subframeCount, kSize* subframeCapacity);

#include <kFireSync/Utils/kFsUtils.x.h>

#endif
