/** 
 * @file    kSvNode.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYC_SERVER_NODE_X_H
#define K_FIRESYC_SERVER_NODE_X_H

#include <kFireSync/Client/Network/kDiscoveryProvider.h>
#include <kFireSync/Client/Network/kNodeProvider.h>

#define kSV_NODE_DEVICE_CONFIG_FILE_PATH        "/user/Boot.kdev"                       ///< Device configuration file path (loaded on boot).
#define kSV_NODE_DEVICE_CONFIG_NEXT_FILE_PATH   "/user/BootNext.kdev"                   ///< Device configuration update file path (processed on boot).
#define kSV_NODE_DEVICE_CONFIG_VERSION          (1)                                     ///< Device configuration XML schema version.

#define kSV_NODE_MODE_CONFIG_FILE_PATH          "/user/Boot.kcfg"                       ///< Default mode configuration path (loaded on boot). 
#define kSV_NODE_MODE_CONFIG_NEXT_FILE_PATH     "/user/BootNext.kcfg"                   ///< Default mode configuration update path (processed on boot). 
#define kSV_NODE_MODE_CONFIG_VERSION            (1)                                     ///< Mode configuration XML schema version.

//Priority of watchdog status check thread.
#define kSV_NODE_WATCHDOG_PRIORITY_CLASS        kTHREAD_PRIORITY_CLASS_HIGH             ///< Thread priority class used for watchdog.           
#define kSV_NODE_WATCHDOG_PRIORITY_OFFSET       0                                       ///< Thread priority offset used for watchdog.           

#define kSV_NODE_RECOVERY_QUIT_PERIOD           (100000)                                ///< Recovery thread polls for quit at this interval (us).

#define kSV_NODE_DEFAULT_WEB_PORT               (80)                                    ///< Default web server port. 

typedef kStatus (kCall* kSvNodeParseModuleFx)(kCamera camera, kXml xml, kXmlItem item); 
typedef kStatus (kCall* kSvNodeFormatModuleFx)(kCamera camera, kXml xml, kXmlItem item); 

typedef struct kSvNodeClass
{
    kObjectClass base; 
    
    kObject hwNode;                         //Hardware node (kHxNode).
    kSystem system;                         //System object to which this node belongs.
    kSvDiscoveryProvider discoveryProvider; //Provides discovery integration with kSystem. 

    k32u id;                                //Node id (read-only; read from kHxNode during init). 
    kBool isInitialized;                    //Did node initialization complete successfully?
    kBool isTerminating;                    //Is the node currently being destroyed?
    kBool hasDefaultModeConfig;             //Was a default mode-config file loaded at boot time?
    k64u bootTime;                          //Local time at node construction.
    k64u startTime;                         //Time at which acquisition was most recently started.
    kAtomic32s hasAuthoritativeTime;        //Has the time been set by an authoritative time provider?

    //state relevant only to implementation of node-provider interface
    kCallback providerHealthFx;             //Health callback.
    kSvHealthOutApp providerHealthChannel;  //Health channel. 
    kCallback providerDataFx;               //Data callback.
    kSvDataOutApp providerDataChannel;      //Data channel. 
    kBool providerShouldReset;              //Reset node when control connection closed?
    kBool providerRestartApp;               //Should the sensor reset without rebooting?

    kAtomic32s discoveryPort;               //Current discovery destination UDP port id. 
    k32u nextDiscoveryPort;                 //Discovery destination UDP port id after service restart. 
    kAtomic32s controlPort;                 //Current control TCP port id. 
    k32u nextControlPort;                   //Control TCP port id after service restart. 
    kAtomic32s healthPort;                  //Current health TCP port id. 
    k32u nextHealthPort;                    //Health TCP port id after service restart. 
    kAtomic32s dataPort;                    //Current data TCP port id. 
    k32u nextDataPort;                      //Data TCP port id after service restart. 
    kAtomic32s webPort;                     //Current web server TCP port id. 
    k32u nextWebPort;                       //Web server TCP port id after service restart. 

    k64u configRevision;                //Current device config revision number (shared with discovery service).
    kNodeState actualRunState;          //Real device run state, e.g. running, ready. 
    kNodeState visibleRunState;         //Device run state visible to network.  
    kBool isRunStateSuspended;          //Are updates to the visible run state currently suspended?
    kBool deviceConfigDirty;            //Flags if deviceConfig has unsaved changes.
    kBool isEnabled;                    //Is the node currently enabled?
    kLock stateLock;                    //Lock for state variables shared with discovery service.

    kSvDiscovery discovery;             //Discovery service.
    kSvControl control;                 //Control service. 
    kSvDataOut dataOut;                 //Data output service.
    kSvDataIn dataIn;                   //Data input service.
    kSvHealthOut healthOut;             //Heatlh output service.
    kSvWeb web;                         //Web server. 
    
    kText32 appName;                    //Node application name. 
    kVersion appVersion;                //Node application version.
    kPointer app;                       //Reference to application object; can optionally be set by app, for convenient lookup.
    kMap procedureMap;                  //Maps procedure names to callback info -- kMap<kText64, kCallback>
    kLock procedureLock;                //Provides exclusive access to procedure map. 
                     
    kMap discoveryMap;                  //Maps device id to discovery info, used by data input service -- kMap<k32u, kDiscoveryInfo>. 
    kLock discoveryLock;                //Lock for discovery map. 

    kLock controlLock;                  //Provides exclusive access to most node state for control channel operations.

    kSemaphore shutdownSem;             //Signals that kSvNode_WaitForShutdown function should unblock. 
    kAtomic32s restartApp;              //Should the sensor reset without rebooting?

    kArrayList eventManagers;           //List of event managers -- kArrayList<kSvEventManager>.
    kArrayList cameras;                 //List of cameras -- kArrayList<kSvCamera>.
    kArrayList lights;                  //List of lights -- kArrayList<kSvLight>.
    kArrayList projectors;              //List of projectors -- kArrayList<kSvProjector>.
    kArrayList analogOutputs;           //List of analog outputs -- kArrayList<kSvAnalogOut>.
    kArrayList digitalOutputs;          //List of digital outputs -- kArrayList<kSvDigitalOut>.
    kArrayList serialOutputs;           //List of serial outputs -- kArrayList<kSvSerialOut>.
    kArrayList ioTests;                 //List of I/O tests -- kArrayList<kSvIoTest>.
    kSvEncoder encoder;                 //Encoder module (can be null). 
    kSvTestJig testJig;                 //Test jig module (can be null). 

    kSvPipe pipe;                       //Pipe module. 
    kSvActions actions;                 //Actions module.

    kPeriodic watchdogStatusTimer;      //Periodic callback to assess health for watchdog. 
    kCallback watchdogCallback;         //User callback for watchdog status check. 

    kThread recoveryThread;             //Checks for run-time errors and performs recovery steps as needed. 
    kAtomic32s shouldRecoveryQuit;      //Signals that recovery monitoring thread should quit.
    kBool recoveryEnabled;              //Is automatic error recovery enabled?

    kMap variableMap;                   //Server variables -- kMap<kText64, kObject>. 
    kLock variableLock;                 //Provides exclusive access to server variables.

    kHealthProbe appVersionProbe;            //Health probe for application version number. 
    kHealthProbe platformVersionProbe;       //Health probe for platform version number. 
    kHealthProbe runStateProbe;              //Health probe for run-state. 
    kHealthProbe bootCountProbe;             //Health probe for total boot count (persistent).
    kHealthProbe totalStartCountProbe;       //Health probe for total acquisition start count (persistent).
    kHealthProbe totalRunningTimeProbe;           //Health probe for total acquisition time (persistent).
    kHealthProbe upTimeProbe;                //Health probe for up-time.
    kHealthProbe totalUpTimeProbe;           //Health probe for total up-time (persistent)
    kHealthProbe recoveryCountProbe;         //Health probe to count recovery actions. 
    kHealthProbe totalRecoveryCountProbe;    //Health probe to count total recovery actions (persistent).
    kHealthProbe dateTimeProbe;         //Health probe to report node's current concept of date-time (persistent).
    kHealthProbe watchdogResetCountProbe;    //Health probe for watchdog reset count.
    kHealthProbe cudaStatusProbe;       //Health probe to report overall status of Cuda processing environment.     

    kProfileProbe startTimerProbe;      //Code profiling timer for start operations. 
    kProfileProbe engageTimerProbe;     //Code profiling timer for engage operations. 
    kProfileProbe stopTimerProbe;       //Code profiling timer for stop operations.
} kSvNodeClass; 

kDeclareClassEx(kFs, kSvNode, kObject)

kFsFx(kStatus) xkSvNode_Construct(kSvNode* node, kSystem system, kHxNode hardware, kAlloc allocator);
        
kFsFx(kStatus) kSvNode_Init(kSvNode node, kType type, kSystem system, kHxNode hardware, kAlloc alloc);
kFsFx(kStatus) kSvNode_VRelease(kSvNode node);

kFsFx(kStatus) xkSvNode_Boot(kSvNode node);
kFsFx(kStatus) xkSvNode_ApplySettings(kSvNode node);

kFsFx(kStatus) kSvNode_FormatPath(kSvNode node, const kChar* logicalPath, kChar* physicalPath, kSize capacity); 

kFsFx(kStatus) kSvNode_InitNetwork(kSvNode node); 

kFsFx(kStatus) kSvNode_OnDiscoveryUpdate(kSvNode node, kSize interfaceIndex, kDiscoveryInfo* info);

kFsFx(kStatus) kSvNode_EnumerateHardware(kSvNode node);

kFsFx(kStatus) kSvNode_LoadDeviceConfigFile(kSvNode node); 
kFsFx(kStatus) kSvNode_SaveDeviceConfigFile(kSvNode node); 

kFsFx(kStatus) kSvNode_ParseDeviceInterface(kSvNode node, kXml xml, kXmlItem item, kSize interfaceIndex, kBool isLoading);
kFsFx(kStatus) kSvNode_ParseDeviceInterfaces(kSvNode node, kXml xml, kXmlItem item, kBool isLoading);
kFsFx(kStatus) kSvNode_ParseDeviceConfig(kSvNode node, kXml xml, kBool isLoading); 

kFsFx(kStatus) kSvNode_FormatDeviceInterface(kSvNode node, kXml xml, kXmlItem item, kSize interfaceIndex);
kFsFx(kStatus) kSvNode_FormatDeviceInterfaces(kSvNode node, kXml xml, kXmlItem item);
kFsFx(kStatus) kSvNode_FormatDeviceConfig(kSvNode node, kXml xml); 

kFsFx(kStatus) kSvNode_LoadDefaultModeConfigFile(kSvNode node); 

kFsFx(kStatus) kSvNode_ParseModeInterface(kSvNode node, kXml xml, kXmlItem item, kSize interfaceIndex);
kFsFx(kStatus) kSvNode_ParseModeInterfaces(kSvNode node, kXml xml, kXmlItem item);
kFsFx(kStatus) kSvNode_ParseModeConfig(kSvNode node, kXml xml); 
kFsFx(kStatus) kSvNode_ParseModuleList(kSvNode node, kXml xml, kXmlItem item, kArrayList moduleList, kSvNodeParseModuleFx parseFx); 
kFsFx(kStatus) kSvNode_ParseGpioBanks(kSvNode node, kXml xml, kXmlItem item);

kFsFx(kStatus) kSvNode_FormatModeInterface(kSvNode node, kXml xml, kXmlItem item, kSize interfaceIndex);
kFsFx(kStatus) kSvNode_FormatModeInterfaces(kSvNode node, kXml xml, kXmlItem item);
kFsFx(kStatus) kSvNode_FormatModeConfig(kSvNode node, kXml xml); 
kFsFx(kStatus) kSvNode_FormatModuleList(kSvNode node, kXml xml, kXmlItem item, const kChar* listName, const kChar* itemName, kArrayList moduleList, kSvNodeFormatModuleFx formatFx); 
kFsFx(kStatus) kSvNode_FormatGpioBanks(kSvNode node, kXml xml, kXmlItem item);

kFsFx(kStatus) kSvNode_SetConfigRevision(kSvNode node, k64u revision);
kFsFx(k64u) kSvNode_ConfigRevision(kSvNode node);

kFsFx(kStatus) kSvNode_SetRunState(kSvNode node, kNodeState state);
kFsFx(kStatus) kSvNode_SuspendRunStateUpdates(kSvNode node, kBool suspend);
kFsFx(kNodeState) kSvNode_ActualRunState(kSvNode node);
kFsFx(kNodeState) kSvNode_VisibleRunState(kSvNode node);

kFsFx(kNodeState) kSvNode_Enable(kSvNode node, kBool enabled); 
kFsFx(kBool) kSvNode_IsEnabled(kSvNode node); 

kFsFx(kStatus) kSvNode_OnDiscoveryEnum(kSvNode node, kSvDiscovery discovery, kArrayList info); 
kFsFx(kStatus) kSvNode_FindDataEndPoint(kSvNode node, k32u remoteDevice, kIpAddress* address, k32u* port); 

kFsFx(kSize) kSvNode_CameraCount(kSvNode node); 
kFsFx(kSvCamera) kSvNode_CameraAt(kSvNode node, kSize index); 

kFsFx(kSize) kSvNode_LightCount(kSvNode node); 
kFsFx(kSvLight) kSvNode_LightAt(kSvNode node, kSize index); 

kFsFx(kSize) kSvNode_ProjectorCount(kSvNode node); 
kFsFx(kSvProjector) kSvNode_ProjectorAt(kSvNode node, kSize index); 

kFsFx(k64u) kSvNode_UpTime(kSvNode node); 

kFsFx(kHealth) kSvNode_Health(kSvNode node);

kFsFx(kStatus) kSvNode_RegisterHealth(kSvNode node); 
kFsFx(kStatus) kSvNode_UnregisterHealth(kSvNode node); 
kFsFx(kStatus) kSvNode_OnHealthUpdate(kSvNode node, kObject sender, kPointer unused);

kFsFx(kDateTime) kSvNode_ReportedDateTime(kSvNode node);

kFsFx(kStatus) kSvNode_ClearStats(kSvNode node);

kFsFx(kStatus) kSvNode_OnRecoveryEvent(kSvNode node);
kFsFx(kStatus) kSvNode_RecoveryThreadEntry(kSvNode node);

//methods relevant only to implementation of node-provider interface
kFsFx(kBool) kSvNode_ProviderIsRemote(kSvNode node);
kFsFx(kStatus) kSvNode_ProviderOpenControl(kSvNode node);
kFsFx(kStatus) kSvNode_ProviderCloseControl(kSvNode node);
kFsFx(kStatus) kSvNode_ProviderSetHealthHandler(kSvNode node, kCallbackFx function, kPointer receiver);
kFsFx(kStatus) kSvNode_ProviderOpenHealth(kSvNode node);
kFsFx(kStatus) kSvNode_ProviderCloseHealth(kSvNode node);
kFsFx(kStatus) kSvNode_ProviderSetDataHandler(kSvNode node, kCallbackFx function, kPointer receiver);
kFsFx(kStatus) kSvNode_ProviderOpenData(kSvNode node, const k16u* ports, kSize portCount);
kFsFx(kStatus) kSvNode_ProviderCloseData(kSvNode node);
kFsFx(kStatus) kSvNode_ProviderReset(kSvNode node, kBool restartApp);
kFsFx(kStatus) kSvNode_ClearFirmware(kSvNode node, kNodeFirmwareType type); 
kFsFx(kStatus) kSvNode_WriteFirmware(kSvNode node, kNodeFirmwareType type, kStream stream, kSize size, kCallbackFx progress, kPointer context); 

kFsFx(kStatus) kSvNode_ReloadFirmware(kSvNode node, kNodeFirmwareType type, kStream stream, kSize size);
kFsFx(kStatus) kSvNode_ReloadFirmwareImpl(kSvNode node, kNodeFirmwareType type, kStream stream, kSize size);
kFsFx(kStatus) kSvNode_UnloadHardware(kSvNode node);
kFsFx(kStatus) kSvNode_ReloadHardware(kSvNode node);

kFsFx(kStatus) kSvNode_WriteFile(kSvNode node, const kChar* path, kStream stream, kSize size, kCallbackFx progress, kPointer context);
kFsFx(kStatus) kSvNode_CopyFile(kSvNode node, const kChar* sourcePath, const kChar* destPath, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kSvNode_ReadFile(kSvNode node, const kChar* path, kStream stream, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kSvNode_DeleteFile(kSvNode node, const kChar* path); 
kFsFx(kStatus) kSvNode_GetFileInfo(kSvNode node, const kChar* path, kStorageItem* item);
kFsFx(kStatus) kSvNode_CreateDirectory(kSvNode node, const kChar* path); 
kFsFx(kStatus) kSvNode_DeleteDirectory(kSvNode node, const kChar* path); 
kFsFx(kStatus) kSvNode_ListDirectoryEntries(kSvNode node, const kChar* path, kBool includeFiles, kBool includeDirectories, kArrayList entries); 
kFsFx(kStatus) kSvNode_ListDirectoryEntriesEx(kSvNode node, const kChar* path, kArrayList entries);
kFsFx(kStatus) kSvNode_SetDeviceConfig(kSvNode node, kXml xml, k64u revisionId); 
kFsFx(kStatus) kSvNode_GetDeviceConfig(kSvNode node, kXml xml, k64u* revisionId); 
kFsFx(kStatus) kSvNode_CommitDeviceConfig(kSvNode node); 
kFsFx(kStatus) kSvNode_SetModeConfig(kSvNode node, kXml xml, k64u revisionId); 
kFsFx(kStatus) kSvNode_GetModeConfig(kSvNode node, kXml xml, k64u* revisionId); 
kFsFx(kStatus) kSvNode_SaveModeConfig(kSvNode node, const kChar* path); 
kFsFx(kStatus) kSvNode_LoadModeConfig(kSvNode node, const kChar* path); 
kFsFx(kStatus) kSvNode_CommitModeConfig(kSvNode node); 
kFsFx(kStatus) kSvNode_ValidateBlockConfig(kSvNode node, kXml xml); 
kFsFx(kStatus) kSvNode_SetDateTime(kSvNode node, kDateTime dateTime, kBool isAuthoritative); 
kFsFx(kStatus) kSvNode_IncrementInputCounter(kSvNode node); 
kFsFx(kStatus) kSvNode_ListProcedures(kSvNode node, kArrayList procedures);    
kFsFx(kStatus) kSvNode_InvokeProcedure(kSvNode node, const kChar* name, kObject input, kObject* output, kAlloc alloc);    
kFsFx(kStatus) kSvNode_ValidateActionConfig(kSvNode node, kXml xml); 
kFsFx(kStatus) kSvNode_InvokeAction(kSvNode node, const kChar* typeName, kXml config, kObject input, kObject* output, kAlloc alloc); 
kFsFx(kStatus) kSvNode_InvokeNamedAction(kSvNode node, const kChar* actionName, kObject input, kObject* output, kAlloc alloc); 
kFsFx(kStatus) kSvNode_SetVariable(kSvNode node, const kChar* name, kObject variable); 
kFsFx(kStatus) kSvNode_GetVariable(kSvNode node, const kChar* name, kObject* variable, kAlloc alloc); 
kFsFx(kStatus) kSvNode_GetVariableInfo(kSvNode node, const kChar* name, kNodeVariableInfo* info); 
kFsFx(kStatus) kSvNode_DeleteVariable(kSvNode node, const kChar* name); 
kFsFx(kStatus) kSvNode_ListVariables(kSvNode node, kArrayList variables); 
kFsFx(kStatus) kSvNode_ReadMemory(kSvNode node, kNodeMemorySpace space, k64u offset, kSize length, void* data);
kFsFx(kStatus) kSvNode_WriteMemory(kSvNode node, kNodeMemorySpace space, k64u offset, kSize length, const void* data);
kFsFx(kStatus) kSvNode_ReadRegisters(kSvNode node, kRegisterModule moduleType, kSize moduleIndex, kArrayList registers);
kFsFx(kStatus) kSvNode_ReadRegisterOverrides(kSvNode node, kRegisterModule moduleType, kArrayList registers);
kFsFx(kStatus) kSvNode_WriteRegisterOverrides(kSvNode node, kRegisterModule moduleType, kArrayList registers);
kFsFx(kStatus) kSvNode_I2cRead(kSvNode node, k32u deviceId, k32u address, kByte* data, kSize size);
kFsFx(kStatus) kSvNode_I2cWrite(kSvNode node, k32u deviceId, k32u address, const kByte* data, kSize size);
kFsFx(kStatus) kSvNode_BeginSpi(kSvNode node, kSpiDeviceType deviceType);
kFsFx(kStatus) kSvNode_EndSpi(kSvNode node);
kFsFx(kStatus) kSvNode_SpiRead(kSvNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);
kFsFx(kStatus) kSvNode_SpiWrite(kSvNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);
kFsFx(kStatus) kSvNode_ReadLogHistory(kSvNode node, kArrayList logItems);
kFsFx(kStatus) kSvNode_ReadCrashLog(kSvNode node, kString* crashLog, kAlloc allocator);
kFsFx(kStatus) kSvNode_ClearCrashLog(kSvNode node);
kFsFx(kStatus) kSvNode_Start(kSvNode node);    
kFsFx(kStatus) kSvNode_Stop(kSvNode node, kBool synchronizeData);    
kFsFx(kStatus) kSvNode_Engage(kSvNode node, k64u startTime, k64s startEncoder);    
kFsFx(kStatus) kSvNode_EngageEx(kSvNode node, k64u startTime, k64s startEncoder, kBool maintainPhase);    
kFsFx(kStatus) kSvNode_StartReplay(kSvNode node); 
kFsFx(kStatus) kSvNode_Replay(kSvNode node, kObject input, kArrayList output); 

kFsFx(kStatus) kSvNode_CameraOverride(kSvNode node, kSize index, const kCameraOverrideItem* overrides, kSize count);
kFsFx(kStatus) kSvNode_LightOverride(kSvNode node, kSize index, const kLightOverrideItem* overrides, kSize count);

kFsFx(kStatus) kSvNode_ScheduleEvent(kSvNode node, k32u eventManagerId, k64s target);    
kFsFx(kStatus) kSvNode_TriggerEvent(kSvNode node, k32u eventManagerId);   
kFsFx(kStatus) kSvNode_WriteRamImage(kSvNode node, kSize index, kSize stateIndex, kSize imageIndex, kImage image);
kFsFx(kStatus) kSvNode_WritePrnu(kSvNode node, kSize index, kImage white, kImage black);
kFsFx(kStatus) kSvNode_WriteFpn(kSvNode node, kSize index, kImage offsets);
kFsFx(kStatus) kSvNode_WriteRangeLut(kSvNode node, kSize index, const kRangeLutParams* params, const kRangeLutArray* xArray,
                                      const kRangeLutArray* zArray, const kRangeLutArray* validArray);
kFsFx(kStatus) kSvNode_WritePhaseDecoderLut(kSvNode node, kSize index, kCameraPhaseDecoderLutType type, kArray1 lut);
kFsFx(kStatus) kSvNode_SetCameraLvdsPower(kSvNode node, kSize index, k32u power);
kFsFx(kStatus) kSvNode_GetCameraLvdsPower(kSvNode node, kSize index, k32u* power);
kFsFx(kStatus) kSvNode_WriteAccelerationTestData(kSvNode node, kSize index, kObject data); 
kFsFx(kStatus) kSvNode_ReadAccelerationTestResult(kSvNode node, kSize index, kObject* data, kAlloc allocator); 

kFsFx(kStatus) kSvNode_TriggerCamera(kSvNode node, kSize index);
kFsFx(kStatus) kSvNode_TriggerLight(kSvNode node, kSize index);
kFsFx(kStatus) kSvNode_SetLightState(kSvNode node, kSize index, kBool enabled);
kFsFx(kStatus) kSvNode_LoadPatterns(kSvNode node, kSize index, kArray2 patterns);
kFsFx(kStatus) kSvNode_LoadProjectorRowGains(kSvNode node, kSize index, kArray1 gains);
kFsFx(kStatus) kSvNode_LoadProjectorColGains(kSvNode node, kSize index, kArray1 gains);
kFsFx(kStatus) kSvNode_SetDigitalOutState(kSvNode node, kSize index, kBool state);
kFsFx(kStatus) kSvNode_TriggerDigitalOut(kSvNode node, kSize index);
kFsFx(kStatus) kSvNode_EnqueueAnalogOut(kSvNode node, kSize index, k32u value);
kFsFx(kStatus) kSvNode_EnqueueAnalogOutCurrent(kSvNode node, kSize index, k64f value);
kFsFx(kStatus) kSvNode_WriteAnalogOutCalibration(kSvNode node, kSize index, kArray2 calibration, k64u revisionId);
kFsFx(kStatus) kSvNode_WriteSerialOut(kSvNode node, kSize index, const void* buffer, kSize size);
kFsFx(kStatus) kSvNode_ResetEncoder(kSvNode node); 
kFsFx(kStatus) kSvNode_EnableLed(kSvNode node, kLed instance, kBool enabled); 
kFsFx(kStatus) kSvNode_SetLedMode(kSvNode node, kLed instance, kLedMode mode);
kFsFx(kStatus) kSvNode_SetGpioState(kSvNode node, kSize bankIndex, k64u mask, k64u state);
kFsFx(kStatus) kSvNode_GetGpioState(kSvNode node, kSize bankIndex, k64u* state);

kFsFx(kStatus) kSvNode_GetInfo(kSvNode node, kNodeInfo* info); 
kFsFx(kStatus) kSvNode_GetState(kSvNode node, kNodeStateInfo* info); 
kFsFx(kStatus) kSvNode_GetHealth(kSvNode node, kArrayList stats); 
kFsFx(kStatus) kSvNode_ReadHealthLog(kSvNode node, kHealthSummary* summary, kAlloc alloc); 
kFsFx(kStatus) kSvNode_ClearHealthLog(kSvNode node); 
kFsFx(kStatus) kSvNode_GetNodeStats(kSvNode node, kNodeStats* stats);  
kFsFx(kStatus) kSvNode_GetEventStats(kSvNode node, k32u eventManagerId, kEventStats* stats); 
kFsFx(kStatus) kSvNode_GetCameraStats(kSvNode node, kSize index, kCameraStats* stats);
kFsFx(kStatus) kSvNode_GetLightStats(kSvNode node, kSize index, kLightStats* stats);
kFsFx(kStatus) kSvNode_GetProjectorStats(kSvNode node, kSize index, kProjectorStats* stats);
kFsFx(kStatus) kSvNode_GetAnalogOutStats(kSvNode node, kSize index, kAnalogOutStats* stats);
kFsFx(kStatus) kSvNode_GetDigitalOutStats(kSvNode node, kSize index, kDigitalOutStats* stats);
kFsFx(kStatus) kSvNode_GetSerialOutStats(kSvNode node, kSize index, kSerialOutStats* stats);

kFsFx(kStatus) kSvNode_GetIoTestStats(kSvNode node, kSize index, kIoTestStats* stats);
kFsFx(kStatus) kSvNode_ResetIoTest(kSvNode node, kSize index);
kFsFx(kStatus) kSvNode_ReadIoTestSerialData(kSvNode node, kSize index, void* buffer, kSize capacity, kSize* bytesRead);

kFsFx(kStatus) kSvNode_TestJigCommand(kSvNode node, kTestJigCommand command, k64u data);
kFsFx(kStatus) kSvNode_GetTestJigStats(kSvNode node, kTestJigStats *stats);

kFsFx(kStatus) kSvNode_LockOutLight(kSvNode node, kSize index, kBool shouldLock);
kFsFx(kStatus) kSvNode_SetLightDriverControl(kSvNode node, kSize index, k64u key, kBool enabled);
kFsFx(kStatus) kSvNode_GetLightDriverControl(kSvNode node, kSize index, k64u key, kBool* enabled);
kFsFx(kStatus) kSvNode_SetLightDriverPower(kSvNode node, kSize index, k64u key, k32u power, kBool commit);
kFsFx(kStatus) kSvNode_GetLightDriverPower(kSvNode node, kSize index, k64u key, k32u* power);
kFsFx(kStatus) kSvNode_SetLightDriverCurrentLimit(kSvNode node, kSize index, k64u key, k32u currentLimit, kBool commit);
kFsFx(kStatus) kSvNode_GetLightDriverCurrentLimit(kSvNode node, kSize index, k64u key, k32u* currentLimit);
kFsFx(kStatus) kSvNode_GetLightDriverRemainingSlotCount(kSvNode node, kSize index, k64u key, kSize* remainingCount);
kFsFx(kStatus) kSvNode_SetLightDriverInfo(kSvNode node, kSize index, k64u key, kLightModel model, kVersion revision, k32u deviceId);
kFsFx(kStatus) kSvNode_GetLightDriverInfo(kSvNode node, kSize index, k64u key, kLightModel* model, kVersion* revision, k32u* deviceId);
kFsFx(kStatus) kSvNode_SetLightDriverCalibration(kSvNode node, kSize index, k64u key, kDataTree ldCal, kBool commit);
kFsFx(kStatus) kSvNode_GetLightDriverCalibration(kSvNode node, kSize index, k64u key, kDataTree* ldCal, kAlloc alloc);
kFsFx(kStatus) kSvNode_ReadLightDriverTemperature(kSvNode node, kSize index, k64u key, k32s* temperature);

kInlineFx(kSvDiscovery) kSvNode_Discovery(kSvNode node)
{
    kObj(kSvNode, node); 

    return obj->discovery;
}

kInlineFx(kSvHealthOut) kSvNode_HealthOut(kSvNode node)
{
    kObj(kSvNode, node); 

    return obj->healthOut;
}

kInlineFx(kSvDataOut) kSvNode_DataOut(kSvNode node)
{
    kObj(kSvNode, node); 

    return obj->dataOut;
}

kInlineFx(kSvPipe) kSvNode_Pipe(kSvNode node)
{
    kObj(kSvNode, node); 

    return obj->pipe;
}

kInlineFx(kHxNode) kSvNode_Hardware(kSvNode node)
{
    kObj(kSvNode, node); 

    return obj->hwNode;
}

kFsFx(kStatus) kSvNode_PrimeHardware(kSvNode node);
kFsFx(k32u) kSvNode_CameraPrimeCount(kSvNode node);
kFsFx(k32u) kSvNode_LightPrimeCount(kSvNode node);
kFsFx(kStatus) kSvNode_PrepareForPrime(kSvNode node, kSvEventManager eventManager, k32u primeCount, k64u* period);
kFsFx(kStatus) kSvNode_RunPrime(kSvNode node, kSvEventManager eventManager, k32u discardCount, k64u framePeriod);
kFsFx(kStatus) kSvNode_RestoreAfterPrime(kSvNode node, kSvEventManager eventManager);

kFsFx(kStatus) kSvNode_CalibrateCameras(kSvNode node);

kFsFx(kStatus) xkSvNode_Log(kSvNode node, kLogOption options, const kChar* source, const kChar* message);
kFsFx(kStatus) xkSvNode_Logf(kSvNode node, kLogOption options, const kChar* source, const kChar* format, ...);
kFsFx(kStatus) xkSvNode_Logvf(kSvNode node, kLogOption options, const kChar* source, const kChar* format, kVarArgList argList);

/* 
* Deprecated (Stage 1): not recommended for further use, but not yet announced via kDeprecate
*/
#define kSvNode_Client kSvNode_System 

#endif
