/** 
 * @file    kPxPipe.h
 * @brief   Declares the kPxPipe class and related types. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PX_PIPE_H
#define K_FIRESYNC_PX_PIPE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kPxPipe
 * @extends kObject
 * @ingroup kFireSync-Pipe
 * @brief   Abstract base class for a pipe execution engine.
 */
//typedef kObject kPxPipe;         --forward-declared in kFsDef.x.h

/** 
 * Sets the maximum input queue data size, in bytes. 
 *
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @param   size            Maximum input queue data size, in bytes.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_SetMaxQueueSize(kPxPipe pipe, kSize size);

/** 
 * Sets the maximum input queue item count. 
 *
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @param   count           Maximum input queue item count.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_SetMaxQueueCount(kPxPipe pipe, kSize count);

/**
 * Sets the desired number of threads used by the pipe for data processing. 
 * 
 * This setting applies only to concurrent pipe implementations. 
 *
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object.
 * @param   count           Desired thread count (1 - N). 
 * @return                  Operation status.
 */
kFsFx(kStatus) kPxPipe_SetThreadCount(kPxPipe pipe, kSize count);

/** 
 * Clears existing pipe configuration (blocks, routes). 
 *
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_Clear(kPxPipe pipe);

/** 
 * Adds block instances to the pipe. 
 * 
 * Ownership of the blocks is not transferred; the caller is responsible for destroying 
 * the blocks when they are no longer needed. 
 *
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @param   blocks          Block list. 
 * @param   count           Block count. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_AddBlocks(kPxPipe pipe, const kPxBlock* blocks, kSize count);

/** 
 * Adds message routes to the pipe. 
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @param   routes          Message routes. 
 * @param   count           Route count. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_AddRoutes(kPxPipe pipe, const kRouteEntry* routes, kSize count);

/** 
 * Starts the pipe execution environment. 
 * 
 * The pipe takes responsibility for starting and controlling its individual blocks. 
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_Start(kPxPipe pipe);

/** 
 * Engages the pipe execution environment. 
 * 
 * This method should be called after all startup activities have completed and the environment 
 * is ready to begin receiving pipe-generated messages. 
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_Engage(kPxPipe pipe);

/** 
 * Stops the pipe execution environment. 
 * 
 * The pipe takes responsibility for stopping its individual blocks. 
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_Stop(kPxPipe pipe);

/** 
 * Pauses the pipe execution environment. 
 * 
 * The pipe takes responsibility for pausing its individual blocks. 
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_Pause(kPxPipe pipe);

/** 
 * Resumes the pipe execution environment. 
 * 
 * The pipe takes responsibility for resuming its individual blocks. 
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_Resume(kPxPipe pipe);

/** 
 * Inserts a message into the pipe for re-processing.
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @param   msgInfo         kMsgInfo object (ownership is transferred). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_Replay(kPxPipe pipe, kMsgInfo msgInfo);

/** 
 * Updates any periodic pipe statistics. 
 * 
 * The pipe itself does not maintain an internal thread to periodically update health/statistics; 
 * if updates are desired, this function should be called periodically by external code. 
 * 
 * This function is thread-safe. 
 * 
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxPipe_UpdateStats(kPxPipe pipe);

/**
 * Gets a reference to the (optional) pipe environ provider. 
 *
 * @public                  @memberof kPxPipe
 * @param   pipe            Pipe object.
 * @return                  Pipe environ provider (or null, if not available). 
 */
kFsFx(kPxEnviron) kPxPipe_Environ(kPxPipe pipe);

#include <kFireSync/Pipe/kPxPipe.x.h>

#endif
