/** 
 * @file    kPxEnviron.h
 * @brief   Declares the kPxEnviron interface. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PX_ENVIRON_H
#define K_FIRESYNC_PX_ENVIRON_H

#include <kFireSync/kNodeDef.h>
#include <stdarg.h>

/**
 * @interface   kPxEnviron
 * @ingroup     kFireSync-Pipe
 * @brief       Represents an object that provides external services to a pipe execution engine.
 */
//typedef kObject kPxEnviron;            --forward-declared in kFsDef.x.h  

/** 
 * Gets a reference to a named object. 
 *
 * The object returned by this function should be treated as read-only and should be disposed when 
 * no longer needed. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @param   name            Object name. 
 * @param   object          Receives object reference.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxEnviron_FindVariable(kPxEnviron pipeEnviron, const kChar* name, kObject* object);

/** 
 * Gets the full path for a given file name used within a pipe execution environment. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @param   fileName        File name. 
 * @param   filePath        Receive fully-qualified file path. 
 * @param   capacity        Capacity of the filePath argument. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxEnviron_FilePath(kPxEnviron pipeEnviron, const kChar* fileName, kChar* filePath, kSize capacity);

/** 
 * Prints a diagnostic message. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @param   format          Format string.   
 * @param   args            Variable argument list. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPxEnviron_Printvf(kPxEnviron pipeEnviron, const kChar* format, va_list args);

/** 
 * Gets an allocator that should be used to construct output messages. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @return                  Message allocator. 
 */
kFsFx(kAlloc) kPxEnviron_MessageAlloc(kPxEnviron pipeEnviron);

/** 
 * Gets the health service provider for the pipe environment. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @return                  Health service (may be null).
 */
kFsFx(kHealth) kPxEnviron_Health(kPxEnviron pipeEnviron);

/** 
 * Within an embedded node environment, this method can be used to access the current
 * FireSync synchnronization time. 
 *
 * Use of this method is not recommended, as it can produce unexpected results during 
 * simulation/testing.
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @return                  Current FireSync time (us). 
 */
kFsFx(k64u) kPxEnviron_Timestamp(kPxEnviron pipeEnviron); 

/**
 * Reports the configured pipe thread priority class. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @return                  Thread priority class. 
 */
kFsFx(kThreadPriorityClass) kPxEnviron_ThreadPriorityClass(kPxEnviron pipeEnviron);

/**
 * Reports the configured pipe thread priority offset. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @return                  Thread priority offset. 
 */
kFsFx(k32s) kPxEnviron_ThreadPriorityOffset(kPxEnviron pipeEnviron);

/**
 * Reports the configured pipe thread affinity. 
 *
 * @public                  @memberof kPxEnviron
 * @param   pipeEnviron     Environment provider. 
 * @return                  Thread affinity (may be kNULL, signifying no affinity; do not modify).
 */
kFsFx(kBitArray) kPxEnviron_ThreadAffinity(kPxEnviron pipeEnviron);

#include <kFireSync/Pipe/kPxEnviron.x.h>

#endif
