    /** 
 * @file    kDhcpServer.h
 * @brief   Declares the kDhcpServer class. 
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc.
  */
#ifndef K_FIRESYNC_DHCP_SERVER_H
#define K_FIRESYNC_DHCP_SERVER_H

#include <kApi/kApiDef.h>
#include <kApi/Io/kNetwork.h>
#include <kFireSync/kFsDef.h>


/**
 * @class       kDhcpServer
 * @extends     kObject
 * @ingroup     kFireSync-Net
 * @brief       Implements simplest possible kDhcpServer to test DHCP functionality. 
 */

//typedef kObject kDhcpServer;      --forward-declared in kFsDef.x.h
  
/** 
 * Constructs a kDhcpServer object.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              Destination for the constructed object handle. 
 * @param   allocator           Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kDhcpServer_Construct(kDhcpServer* server, kAlloc allocator);

/** 
 * Sets the DHCP IP address pool start and size.
 *
 * This method can only be called before calling kDhcpServer_Start. One have 
 * to set DHCP pool start address (e.g. 192.168.1.10) and the size of the DHCP address pool (e.g. 10, 
 * so address server can lease address in the range from 192.168.1.10 to 192.168.1.20).
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @param   poolStart           Start IP address of pool.
 * @param   poolSize            Size of the pool.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kDhcpServer_SetPool(kDhcpServer server, kIpAddress poolStart, k32u poolSize);

/** 
 * Reports DHCP pool start IP address.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @return                      Pool start. 
 */
kFsFx(kIpAddress) kDhcpServer_PoolStart(kDhcpServer server);

/** 
 * Reports DHCP pool size.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @return                      Pool size. 
 */
kFsFx(k32u) kDhcpServer_PoolSize(kDhcpServer server);

/** 
 * Sets the DHCP prefix length option.
 *
 * This method can only be called before calling kDhcpServer_Start.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @param   mask                DHCP prefix length option.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kDhcpServer_SetPrefix(kDhcpServer server, k32u mask);

/** 
 * Reports the DHCP prefix length option.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @return                      Prefix length. 
 */
kFsFx(k32u) kDhcpServer_Prefix(kDhcpServer server);

/** 
 * Sets the DHCP gateway option, e.g. 192.168.1.254. 
 *
 * This method can only be called before calling kDhcpServer_Start.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @param   gateway             DHCP gateway option.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kDhcpServer_SetGateway(kDhcpServer server, kIpAddress gateway);

/** 
 * Reports the DHCP gateway option.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @return                      Gateway. 
 */
kFsFx(kIpAddress) kDhcpServer_Gateway(kDhcpServer server);

/** 
 * Starts the DHCP server process.
 *
 * One have to set pool start address and the size of the pool and DHCP mask before 
 * calling this function. The lease time is infinite.
 * 
 * @public                      @memberof kDhcpServer
 * @param   server              kDhcpServer object.
 * @param   serverIp            IP of the server, must be assigned to one of the NICs in the host system. 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kDhcpServer_Start(kDhcpServer server, kIpAddress serverIp);

#include <kFireSync/Net/kDhcpServer.x.h>

#endif
