/** 
 * @file    kHealth.h
 * @brief   Declares the kHealth class and related types. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HEALTH_H
#define K_FIRESYNC_HEALTH_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kHealth
 * @ingroup kFireSync-Health
 * @brief   Base class for a health statistics collection service.
 * 
 * Note, this class is intended to be used primarily in the implementation of the FireSync platform, 
 * rather than directly by applications. However, there may be rare occasions in which it would be 
 * appropriate to use this class directly in an application (e.g., standalone desktop application 
 * that manages its own health functionality). 
 */
//typedef kObject kHealth;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a kHealth object. 
 *
 * @public                  @memberof kHealth
 * @param   health          Destination for the constructed object handle. 
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_Construct(kHealth* health, kAlloc allocator); 

/** 
 * Registers a callback for periodic health polling. 
 * 
 * This function can be used to register a callback that should be invoked periodically 
 * to poll for health updates. Multiple callbacks can be registered. 
 *
 * This method is thread-safe.
 * 
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @param   function        Callback function.
 * @param   receiver        Context pointer for callback function (can be kNULL). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_AddUpdateHandler(kHealth health, kCallbackFx function, kPointer receiver); 

/** 
 * Unregisters a callback from periodic health polling. 
 * 
 * This method is thread-safe.
 * 
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @param   function        Callback function.
 * @param   receiver        Context pointer for callback function (can be kNULL). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_RemoveUpdateHandler(kHealth health, kCallbackFx function, kPointer receiver); 

/** 
 * Invokes health update handlers.
 *
 * This method can be called to invoke all registered health update handlers.
 * 
 * This method is thread-safe.
 * 
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_UpdateHealthStats(kHealth health); 

/** 
 * Gets latest streamed health statistics. 
 * 
 * This method does not automatically call update-handlers to refresh probe values. If desired, 
 * call kHealth_UpdateHealthStats to ensure that this occurs. 
 * 
 * This method is thread-safe.
 *
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @param   stats           Receives health statistics -- kArrayList<kHealthStat>. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_HealthStats(kHealth health, kArrayList stats); 

/** 
 * Gets latest persistent health statistics. 
 * 
 * This method does not automatically call update-handlers to refresh probe values. If desired, 
 * call kHealth_UpdateHealthStats to ensure that this occurs. 
 * 
 * This method is thread-safe.
 *
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @param   stats           Receives health statistics -- kArrayList<kHealthStat>. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_PersistentHealthStats(kHealth health, kArrayList stats); 

/** 
 * Gets current code profiling statistics. 
 * 
 * This method is thread-safe.
 *
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @param   stats           Receives profiling statistics -- kArrayList<kProfileStat>. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_ProfileStats(kHealth health, kArrayList stats); 

/** 
 * Installs a health logger. 
 *
 * This method can be used to install a health logger that provides persistence for 
 * health indicator data. After setting the log, future calls to kHealth_SaveLogSnapshot will 
 * have the effect of generating new log snapshots.
 * 
 * Note that the values or baselines of any already-existent probes that were constructed with 
 * the kHEALTH_OPTION_RESTORE or kHEALTH_OPTION_BASELINE options (respectively) may be modified 
 * upon log installation. This nuance may require some attention in the design of health probes 
 * that are constructed prior to log installation. 
 *
 * This method is thread-safe.
 *  
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @param   log             Health log object (or kNULL, to uninstall the existing logger). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_SetLog(kHealth health, kHealthLog log); 

/** 
 * Gets a reference to the health log (if installed).
 *
 * The health log object returned by this method should be destroyed by the caller 
 * when no longer needed.
 * 
 * This method is thread-safe.
 *  
 * @public                  @memberof kHealth
 * @param   health          Health service.
 * @param   log             Receives health log object or kNULL if not installed.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHealth_Log(kHealth health, kHealthLog* log); 

#include <kFireSync/Health/kHealth.x.h>

#endif
