/** 
 * @file    kHxNullNode.h
 * @brief   Declares the kHxNullNode class.
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HX_NULL_NODE_H
#define K_FIRESYNC_HX_NULL_NODE_H

#include <kFireSync/kNodeDef.h>
#include <kApi/Data/kString.h>
#include <kFireSync/Hardware/kHxNullNode.x.h>

/**
 * @class       kHxNullNode
 * @extends     kObject
 * @ingroup     kFireSync-Hardware
 * @brief       Provides a software-only backend for kSvNode.
 * 
 * kHxNullNode is intended for use by applications to create a kSvNode instance without
 * a strong association to the underlying device on which the server is running (a software-only,
 * or "null" device). This kind of node is unable to perform most kHxNode operations, such as 
 * firmware upgrades or system resets, but is able to support network communication. Multiple kHxNullNode 
 * instances can be created on the same device. 
 * 
 * By default, kHxNullNode instances behave as though they are configured to use the loopback network 
 * adapter. To participate in communication beyond the local device, use one of the available kHxNullNode 
 * configuration methods (e.g., kHxNullNode_AddNetInterfaceByAddress) to create an association between this 
 * node and a different network adapter. 
 */
//typedef kObject kHxNullNode;      --forward-declared in kFsDef.x.h

/** 
 * Constructs a kHxNullNode instance.
 * 
 * @public                  @memberof kHxNullNode
 * @param   node            Receives constructed object. 
 * @param   nodeId          Node identifier that will be used in communication. 
 * @param   health          Health provider to be used by the node (if kNULL, node will create dedicated provider).  
 * @param   storagePath     Absolute file system path to the directory that should be used to support storage (or kNULL for no storage).
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status.
 */
kFsFx(kStatus) kHxNullNode_Construct(kHxNullNode* node, k32u nodeId, kHealth health, const kChar* storagePath, kAlloc allocator); 

/** 
 * Creates an association between this node and a particular network adapter, specified by adapter name. 
 * 
 * If this method will be used, it should be called prior to kSvNode construction.  
 * 
 * @public              @memberof kHxNullNode
 * @param   node        Receives constructed object. 
 * @param   name        Network adapter name (e.g., "eth0").  
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNullNode_AddNetInterfaceByName(kHxNullNode node, const kChar* name); 

/** 
 * Creates an association between this node and a particular network adapter, specified by adapter IP address. 
 * 
 * If this method will be used, it should be called prior to kSvNode construction.  
 * 
 * @public              @memberof kHxNullNode
 * @param   node        Receives constructed object. 
 * @param   address     Network adapter IP address. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNullNode_AddNetInterfaceByAddress(kHxNullNode node, kIpAddress address); 

/** 
 * Creates an association between this node and a particular network adapter, specified by adapter network.
 * 
 * If this method will be used, it should be called prior to kSvNode construction.  
 * 
 * @public                  @memberof kHxNullNode
 * @param   node            Receives constructed object. 
 * @param   address         Network address (host portion will be ignored).
 * @return                  Operation status.
 */
kFsFx(kStatus) kHxNullNode_AddNetInterfaceByNetwork(kHxNullNode node, kIpAddress address); 

/** 
 * Enables this node to capture global log messages. 
 * 
 * By default, null nodes are not integrated with global/application logging infrastructure. This method can 
 * be used to enable global log integration. 
 *
 * @public              @memberof kHxNullNode
 * @param   node        Node object. 
 * @param   enable      TRUE to enable global log integration. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kHxNullNode_EnableGlobalLog(kHxNullNode node, kBool enable);

#endif
