/** 
 * @file    kHxNode.x.h
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HX_NODE_X_H
#define K_FIRESYNC_HX_NODE_X_H

#define kHX_NODE_LOG_QUEUE_CAPACITY                 (1024)          // Log queue capacity. 
#define kHW_NODE_MAX_IFACES                         (4)             // Maximum number of network interfaces.

#define kHX_NODE_ENVIRONMENT_UPDATE_PERIOD          (500000)        // Period of environment update callbacks (us).

#define kHX_NODE_DEFAULT_MIN_HEALTH_LOGGING_PERIOD  (10*60*1000*1000)   // Default period (us) for health logging.

#define kHX_NODE_LED_COUNT                          (kLED_RANGE+1)
#define kHX_NODE_LED_BLINK_CALLBACK_PERIOD          (62500) //half the minimum blink period. All blink periods are multiples of this number.

typedef struct kHxNodeVTable
{
    kObjectVTable base; 

    //should be overridden only by platform singleton node types
    kStatus (kCall* VConstructInstance)(k32s argc, const kChar** argv);
    kStatus (kCall* VLoadApp)(kHxNode node, kDynamicLib* library);
    kStatus (kCall* VSetMessageAlloc)(kHxNode node, kAlloc allocator); 
    kHealth (kCall* VHealth)(kHxNode node);
    kStatus (kCall* VEnableWatchdog)(kHxNode node, kBool enable, k64u timeoutPeriod);  
    kStatus (kCall* VUpdateWatchdog)(kHxNode node, kRestartReason reason); 
    kStatus (kCall* VClear)(kHxNode node);
    kStatus (kCall* VStart)(kHxNode node); 
    kStatus (kCall* VEngage)(kHxNode node, k64u startTime, k64s startEncoder, kBool maintainPhase); 
    kStatus (kCall* VDisengage)(kHxNode node); 
    kStatus (kCall* VStop)(kHxNode node); 
    kStatus (kCall* VWaitForRecoveryEvent)(kHxNode node, k64u timeout); 
    kStatus (kCall* VClearStats)(kHxNode node); 
    kStatus (kCall* VSetNextId)(kHxNode node, k32u id); 
    kStatus (kCall* VSetNextIpConfig)(kHxNode node, kSize index, const kIpConfig* config); 
    kStatus (kCall* VWaitForNetworkOnline)(kHxNode node);
    kStatus (kCall* VDynamicIpConfig)(kHxNode node, kSize index, kIpConfig* config);
    kStatus (kCall* VSetBootConfig)(kHxNode node, const kBootConfig* config);
    kStatus (kCall* VSetNextNetworkSpeed)(kHxNode node, kSize index, kNetworkSpeed speed);
    kStatus (kCall* VSetNetworkTransmitRate)(kHxNode node, kSize index, k32u rate); 
    k32u    (kCall* VNetworkTransmitRate)(kHxNode node, kSize index);
    kStatus (kCall* VCommitConfig)(kHxNode node); 
    kStatus (kCall* VClearFirmware)(kHxNode node, kNodeFirmwareType type); 
    kStatus (kCall* VWriteFirmware)(kHxNode node, kNodeFirmwareType type, kStream stream, kSize size, kCallbackFx progress, kPointer context); 
    kStatus (kCall* VReloadFirmware)(kHxNode node, kNodeFirmwareType type, kStream stream, kSize size);
    kStatus (kCall* VLogEx)(kHxNode node, kLogOption options, const kChar* source, const kChar* message);
    kStatus (kCall* VGetCrashLog)(kHxNode node, kString* crashLog, kAlloc allocator);
    kStatus (kCall* VClearCrashLog)(kHxNode node);
    kStatus (kCall* VSetCrashLogAppInfo)(kHxNode node, kVersion version, k32u nodeId);
    kStatus (kCall* VSetCrashLogDateTime)(kHxNode node, k64s bootDateTime);
    kStatus (kCall* VStats)(kHxNode node, kNodeStats* stats); 
    kBool (kCall* VPlLoaded)(kHxNode node); 
    kBool (kCall* VPl2Loaded)(kHxNode node); 
    kStatus (kCall* VReadMemory)(kHxNode node, kNodeMemorySpace space, k64u offset, kSize length, void* data);
    kStatus (kCall* VWriteMemory)(kHxNode node, kNodeMemorySpace space, k64u offset, kSize length, const void* data);
    kStatus (kCall* VReadRegisters)(kHxNode node, kRegisterModule moduleType, kSize moduleIndex, kArrayList registers);
    kStatus (kCall* VReadRegisterOverrides)(kHxNode node, kRegisterModule moduleType, kArrayList registers);
    kStatus (kCall* VWriteRegisterOverrides)(kHxNode node, kRegisterModule moduleType, kArrayList registers);
    kStatus (kCall* VI2cRead)(kHxNode node, k32u deviceId, k32u address, kByte* data, kSize size);
    kStatus (kCall* VI2cWrite)(kHxNode node, k32u deviceId, k32u address, const kByte* data, kSize size);
    kStatus (kCall* VBeginSpi)(kNode node, kSpiDeviceType deviceType);
    kStatus (kCall* VEndSpi)(kNode node);
    kStatus (kCall* VSpiRead)(kNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);
    kStatus (kCall* VSpiWrite)(kNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);
    kStatus (kCall* VSetTempControl)(kHxNode node, const kTempControlEx* temp);
    kStatus (kCall* VTempControl)(kHxNode node, kTempControlEx* temp);
    kStatus (kCall* VEnableLed)(kHxNode node, kLed instance, kBool enabled);
    k64s    (kCall* VEncoderCounter)(kHxNode node);
    k64u    (kCall* VTimeCounter)(kHxNode node); 
    kStatus (kCall* VSetSyncSource)(kHxNode node, kSyncSource source);
    kStatus (kCall* VSetEventHandler)(kHxNode node, kCallbackFx function, kPointer receiver, kAlloc eventAlloc);
    kStatus (kCall* VEnableLightDriverHighPower)(kHxNode node, kBool enabled);
    kStatus (kCall* VSetLightDriverControl)(kHxNode node, k64u key, kSize index, kBool enabled);
    kStatus (kCall* VGetLightDriverControl)(kHxNode node, k64u key, kSize index, kBool* enabled);
    kStatus (kCall* VSetLightDriverPower)(kHxNode node, k64u key, kSize index, k32u power, kBool commit);
    kStatus (kCall* VGetLightDriverPower)(kHxNode node, k64u key, kSize index, k32u* power);
    kStatus (kCall* VSetLightDriverCurrentLimit)(kHxNode node, k64u key, kSize index, k32u currentLimit, kBool commit);
    kStatus (kCall* VGetLightDriverCurrentLimit)(kHxNode node, k64u key, kSize index, k32u* currentLimit);
    kStatus (kCall* VGetLightDriverRemainingSlotCount)(kHxNode node, k64u key, kSize index, kSize* remainingCount);
    kStatus (kCall* VSetLightDriverInfo)(kHxNode node, k64u key, kSize index, kLightModel model, kVersion revision, k32u deviceId);
    kStatus (kCall* VGetLightDriverInfo)(kHxNode node, k64u key, kSize index, kLightModel* model, kVersion* revision, k32u* deviceId);
    kStatus (kCall* VSetLightDriverCalibration)(kHxNode node, kSize index, k64u key, kDataTree ldCal, kBool commit);
    kStatus (kCall* VGetLightDriverCalibration)(kHxNode node, kSize index, k64u key, kDataTree* ldCal, kAlloc alloc);
    kStatus (kCall* VReadLightDriverTemperature)(kHxNode node, kSize index, k64u key, k32s* temperature);
} kHxNodeVTable;

typedef struct kHxNodeClass
{
    kObjectClass base; 

    /* 
     * Read-only attributes that must be established by descendant during descendant init.
     */

    k32u id;                                    //Node id.
    kBool hasStorage;                           //Does this node have local storage? True by default.
    kString storagePath;                        //Path to storage root (constructecd by base and set to "/"; can be modified by descendant).
    kSize netInterfaceCount;                    //Number of network interfaces
    kIpConfig ipConfig[kHW_NODE_MAX_IFACES];    //Current IP configuration settings.
    kNodeType nodeType;                         //Device category.
    kControllerType controllerType;             //Controller hardware type. 
    kVersion controllerVersion;                 //Controller hardware version. 
    kSocType socType;                           //Controller SoC type.
    k32u socSpeedGrade;                         //Controller SoC speed grade.
    k32u partNumber;                            //Controller part number.
    k32u cpuCount;                              //Count of CPUs in device (0 if unknown). 
    k64u cpuFrequency;                          //CPU clock frequency (Hz; 0 if unknown).
    k64u memorySize;                            //Amount of volatile memory (bytes; 0 if unknown). 
    k64u storageSize;                           //Amount of non-volatile memory (bytes; 0 if unknown).
    k64u minHealthLogPeriod;                    //Minimum health logging period (us).
    kThreadPriorityClass healthLogPriorityClass;//Recommended priority class for health logging.
    k32s healthLogPriorityOffset;               //Recommended priority offset for health logging.
    kBool controllerFactoryResult;              //Controller factory test result.
    kVersion osVersion;                         //Operating system version.
    kVersion bootloaderVersion;                 //Bootloader program version.
    kBootMode bootMode;                         //Method by which software was loaded. 
    kVersion platformVersion;                   //FireSync platform firmware version.
    k32u mainPlConfig;                          //Main programmable logic configuration identifier. 
    kVersion mainPlVersion;                     //Main programmable logic program version. 
    k32u secondaryPlConfig;                     //Secondary programmable logic configuration identifier.
    kVersion secondaryPlVersion;                //Secondary programmable logic program version. 
    kDaughterboardModel daughterboardModel;     //Daughterboard model.
    kVersion daughterboardVersion;              //Daughterboard version.
    kRestartReason restartReason;               //Reports restart details if rebooted by watchdog.
    k32u crashCount;                            //Reports number of crashes encountered by the device.
    kBool hasPeltier;                           //Does this node have peltier temperature control?
    kBool hasExternalTempProbe;                 //Does this node have an external temperature probe? 
    k32u fanCount;                              //How many fan controllers does this node have? 
    kBool hasWatchdog;                          //Can this platform support a watchdog timer (must be set by descendant classes).
    k64u cameraDataBandwidth;                   //What is the total memory bandwidth available for moving camera data into DDR?.
    kMacAddress macAddress[kHW_NODE_MAX_IFACES];                    //MAC address.
    kNetworkSpeed networkSpeedCapabilities[kHW_NODE_MAX_IFACES];    //Network speed capability bitset.
    kNetworkSpeed defaultNetworkSpeed[kHW_NODE_MAX_IFACES];         //Default network speed setting.
    kNetworkSpeed networkSpeed[kHW_NODE_MAX_IFACES];                //Network speed.
    kBool networkTransmitRateSupported[kHW_NODE_MAX_IFACES];        //Can the network transmit rate be adjusted?
    
    /* 
     * Dynamic properties that must be established by descendant during descendant init.
     */

    k32u nextId;                                //Future id setting (to take effect after reboot). 
    kIpConfig nextIpConfig[kHW_NODE_MAX_IFACES];                    //Future IP configuration settings (to take effect after reboot). 
    kNetworkSpeed nextNetworkSpeed[kHW_NODE_MAX_IFACES];            //Future network speed, after reboot (to take effect after reboot). 
    kBootConfig bootConfig;                     //Node boot configuration settings (to take effect after reboot).     
    kSyncSource syncSource;                     //Preferred synchronization source setting. 
    kTempProbeId mainProbe;                     //Setting that determines temperature probe used for message stamp. 
    k64u powerSaverTimeout;                     //Power saver timeout setting (us).
    k64u powerSaverThreshold;                   //Power saver encoder threshold setting (ticks).
    kBool lightDriverHighPowerEnabled;          //Are higher-powered light-driver set points enabled?
    kBool restartRequested;                     //Can be used by application to request application restart instead of full reboot. Set by kHxNode_RequestAppRestart.

    kArrayList eventManagers;                   //Event managers; list list constructed by base, elements added by descendant -- kArrayList<kHwEvent>.
    kArrayList cameras;                         //Cameras; list list constructed by base, elements added by descendant -- kArrayList<kHwCamera>.
    kArrayList lights;                          //Lights; list list constructed by base, elements added by descendant -- kArrayList<kHwLight>.
    kArrayList projectors;                      //Projectors; list list constructed by base, elements added by descendant -- kArrayList<kHwProjector>.
    kArrayList digitalOutputs;                  //Digital outputs; list constructed by base, elements added by descendant -- kArrayList<kHwDigital>.
    kArrayList analogOutputs;                   //Analog outputs; list constructed by base, elements added by descendant -- kArrayList<kHwAnalog>.
    kArrayList serialOutputs;                   //Serial outputs; list constructed by base, elements added by descendant -- kArrayList<kHwSerial>.
    kArrayList ioTests;                         //IoTests; list constructed by base, elements added by descendant -- kArrayList<kHwIoTest>.
    kHxEncoder encoder;                         //Encoder; constructed by descendant.
    kHxTestJig testJig;                         //Test jig; constructed by descendant.
    kHxGpio gpio;                               //GPIO services; constructed by descendant.
 
    /* 
     * Logging infrastructure
     */

    kQueue logQueue;                            //Log queue -- kQueue<kLogItem>
    kLock logLock;                              //Log queue lock.
    k64u logId;                                 //Log message id.
    kSize logItemCount;                         //New log item count.

    /* 
     * Calendar date-time infrastructure
     */
    kLock dateTimeLock;                         //Provides exclusive access to bootDateTime.
    k64s bootDateTime;                          //Offset from Jan 1, 1 CE to node boot time (UTC).

    /* 
     * Environment update infrastructure 
     */

    kLock environmentUpdateLock;                //Provides excusive access to environmentUpdateEvent.
    kEvent environmentUpdateEvent;              //Periodic update event, on which external entities can register for notification.
    kPeriodic environmentUpdateTimer;           //Periodic thread to perform environment updates.

    kHealthProbe leakProbe;                     //Health probe for memory leak tracking

    /* 
    * LED management infrastructure 
    */

    kPeriodic blinkTimer;                           //Periodic background timer to manage LED blink.
    k64u ledBlinkEnabledBitset;                     //Bitset reporting whether specific LEDs are currently configured to blink. Bit indices corresponds to kLed enumerators.
    kLock ledLock;                                  //LED control lock.
    kBool ledStates[kHX_NODE_LED_COUNT];            //Array of LED states (on/off), indexed by kLed enumerator.
    k64u blinkPeriodTicks[kHX_NODE_LED_COUNT];      //Array of LED blink periods (in microseconds), indexed by kLed enumerator.
    k64u blinkTickCounter;                          //Count of blink timer iterations; incremented each time that the periodic blink timer callback executes.

    /*
    * Input counter infrastructure
    */
    kAtomic32s inputCounter;                    //Current input counter value; optionally used in camera stamps.

} kHxNodeClass;

typedef struct kHxNodeStatic
{
    kHxNode platformInstance;                   //Hardware platform singleton, or kNULL if none.
} kHxNodeStatic;

kDeclareFullClassEx(kFs, kHxNode, kObject) 

//Foward declarations
kInlineFx(kHealth) kHxNode_Health(kHxNode node);

kInlineFx(kStatus) xkHxNode_InitStatic() { return kOK; }
kInlineFx(kStatus) xkHxNode_ReleaseStatic() { return kOK; }

kInlineFx(kStatus) xkHxNode_ConstructInstance(kType type, k32s argc, const kChar** argv)
{
    return kType_VTableT(type, kHxNode)->VConstructInstance(argc, argv); 
}

kInlineFx(kStatus) xkHxNode_VConstructInstance(k32s argc, const kChar** argv)
{
    return kERROR_UNIMPLEMENTED;
}

kFsFx(kStatus) xkHxNode_Init(kHxNode node, kType type, kAlloc alloc);
kFsFx(kStatus) xkHxNode_VRelease(kHxNode node);

kFsFx(kStatus) kHxNode_ClearNetModeConfig(kHxNode node, kSize index);
kFsFx(kStatus) kHxNode_ClearModeConfig(kHxNode node);

kFsFx(kStatus) kHxNode_ClearNetDeviceConfig(kHxNode node, kSize index);
kFsFx(kStatus) kHxNode_ClearDeviceConfig(kHxNode node);

kFsFx(kStatus) xkHxNode_StartEnvironUpdates(kHxNode node); 
kFsFx(kStatus) xkHxNode_StopEnvironUpdates(kHxNode node);

kFsFx(kStatus) xkHxNode_OnEnvironUpdate(kHxNode node, kPeriodic timer); 

kFsFx(kStatus) xkHxNode_AddLogEntry(kHxNode node, const kLogArgs* args);

kFsFx(kStatus) xkHxNode_ApplyLedMode(kHxNode node, kLed instance, kLedMode mode, k64u period);
kFsFx(kStatus) xkHxNode_UpdateBlinkTimer(kHxNode node, kLed instance, kLedMode mode);
kFsFx(kStatus) xkHxNode_StopLedBlink(kHxNode node);
kFsFx(kStatus) xkHxNode_OnBlinkTimer(kHxNode node, kPeriodic timer);
kFsFx(kSize) xkHxNode_LedIdBit(kLed instance);
kFsFx(kBool) xkHxNode_IsLedModeBlink(kHxNode node, kLed instance);

kFsFx(k64s) xkHxNode_OnDateTimeQuery();

kInlineFx(kStatus) xkHxNode_VLoadApp(kHxNode node, kDynamicLib* library)
{
    return kERROR_UNIMPLEMENTED;
}

kInlineFx(kStatus) xkHxNode_VSetMessageAlloc(kHxNode node, kAlloc allocator)
{
    return kOK; 
}

kInlineFx(kHealth) xkHxNode_VHealth(kHxNode node)
{
    kAssert(kFALSE);
    return kNULL; 
}

kInlineFx(kStatus) xkHxNode_VEnableWatchdog(kHxNode node, kBool enable, k64u timeoutPeriod)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VUpdateWatchdog(kHxNode node, kRestartReason reason)
{
    return kOK; 
}

kFsFx(kStatus) xkHxNode_VClear(kHxNode node);

kInlineFx(kStatus) xkHxNode_VStart(kHxNode node)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VEngage(kHxNode node, k64u startTime, k64s startEncoder, kBool maintainPhase)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VDisengage(kHxNode node)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VStop(kHxNode node)
{
    return kOK; 
}

kFsFx(kStatus) xkHxNode_VWaitForRecoveryEvent(kHxNode node, k64u timeout);

kInlineFx(kStatus) xkHxNode_VClearStats(kHxNode node)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VSetNextId(kHxNode node, k32u id)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetNextIpConfig(kHxNode node, kSize index, const kIpConfig* config)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VWaitForNetworkOnline(kHxNode node)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VDynamicIpConfig(kHxNode node, kSize index, kIpConfig* config)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetBootConfig(kHxNode node, const kBootConfig* config)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetNextNetworkSpeed(kHxNode node, kSize index, kNetworkSpeed speed)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetNetworkTransmitRate(kHxNode node, kSize index, k32u rate)
{
    return kOK; 
}

kInlineFx(k32u) xkHxNode_VNetworkTransmitRate(kHxNode node, kSize index)
{
    return 100; 
}

kInlineFx(kStatus) xkHxNode_VCommitConfig(kHxNode node)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VClearFirmware(kHxNode node, kNodeFirmwareType type)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VWriteFirmware(kHxNode node, kNodeFirmwareType type, kStream stream, kSize size, kCallbackFx progress, kPointer context)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VReloadFirmware(kHxNode node, kNodeFirmwareType type, kStream stream, kSize size)
{
    return kERROR_UNIMPLEMENTED;
}

kInlineFx(kStatus) xkHxNode_VLogEx(kHxNode node, kLogOption options, const kChar* source, const kChar* message)
{
    return kLog(options, source, message);
}

kInlineFx(kStatus) xkHxNode_VGetCrashLog(kHxNode node, kString* crashLog, kAlloc allocator)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VClearCrashLog(kHxNode node)
{
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VSetCrashLogAppInfo(kHxNode node, kVersion version, k32u nodeId)
{
    return kOK;
}

kInlineFx(kStatus) xkHxNode_VSetCrashLogDateTime(kHxNode node, k64s bootDateTime)
{
    return kOK;
}

kInlineFx(kStatus) xkHxNode_VStats(kHxNode node, kNodeStats* stats)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kBool) xkHxNode_VPlLoaded(kHxNode node)
{    
    return kFALSE; 
}

kInlineFx(kBool) xkHxNode_VPl2Loaded(kHxNode node)
{    
    return kFALSE; 
}

kInlineFx(kStatus) xkHxNode_VReadMemory(kHxNode node, kNodeMemorySpace space, k64u offset, kSize length, void* data)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VWriteMemory(kHxNode node, kNodeMemorySpace space, k64u offset, kSize length, const void* data)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VReadRegisters(kHxNode node, kRegisterModule moduleType, kSize moduleIndex, kArrayList registers)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VReadRegisterOverrides(kHxNode node, kRegisterModule moduleType, kArrayList registers)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VWriteRegisterOverrides(kHxNode node, kRegisterModule moduleType, kArrayList registers)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VI2cRead(kHxNode node, k32u deviceId, k32u address, kByte* data, kSize size)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VI2cWrite(kHxNode node, k32u deviceId, k32u address, const kByte* data, kSize size)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VBeginSpi(kHxNode node, kSpiDeviceType deviceType)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VEndSpi(kHxNode node)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSpiRead(kHxNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSpiWrite(kHxNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetTempControl(kHxNode node, const kTempControlEx* temp)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VTempControl(kHxNode node, kTempControlEx* temp)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VEnableLed(kHxNode node, kLed instance, kBool enabled)
{
    return kOK; 
}

kInlineFx(k64s) xkHxNode_VEncoderCounter(kHxNode node)
{
    return 0; 
}

kInlineFx(k64u) xkHxNode_VTimeCounter(kHxNode node)
{
    k64u now = kTimer_Now();

    return kTimeToFsTime_(now);
}

kInlineFx(kStatus) xkHxNode_VSetSyncSource(kHxNode node, kSyncSource source)
{
    kObj(kHxNode, node); 
    
    obj->syncSource = source;
    
    return kOK; 
}

kInlineFx(kStatus) xkHxNode_VSetEventHandler(kHxNode node, kCallbackFx function, kPointer receiver, kAlloc eventAlloc)
{
    return kOK;
}

kInlineFx(kStatus) xkHxNode_VEnableLightDriverHighPower(kHxNode node, kBool enabled)
{
    return kOK;
}

kInlineFx(kStatus) xkHxNode_VSetLightDriverControl(kHxNode node, k64u key, kSize index, kBool enabled)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VGetLightDriverControl(kHxNode node, k64u key, kSize index, kBool* enabled)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetLightDriverPower(kHxNode node, k64u key, kSize index, k32u power, kBool commit)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VGetLightDriverPower(kHxNode node, k64u key, kSize index, k32u* power)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetLightDriverCurrentLimit(kHxNode node, k64u key, kSize index, k32u currentLimit, kBool commit)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VGetLightDriverCurrentLimit(kHxNode node, k64u key, kSize index, k32u* currentLimit)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VGetLightDriverRemainingSlotCount(kHxNode node, k64u key, kSize index, kSize* remainingCount)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetLightDriverInfo(kHxNode node, k64u key, kSize index, kLightModel model, kVersion revision, k32u deviceId)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VGetLightDriverInfo(kHxNode node, k64u key, kSize index, kLightModel* model, kVersion* revision, k32u* deviceId)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VSetLightDriverCalibration(kHxNode node, kSize index, k64u key, kDataTree ldCal, kBool commit)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VGetLightDriverCalibration(kHxNode node, kSize index, k64u key, kDataTree* ldCal, kAlloc alloc)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_VReadLightDriverTemperature(kHxNode node, kSize index, k64u key, k32s* temperature)
{
    return kERROR_UNIMPLEMENTED; 
}

kInlineFx(kStatus) xkHxNode_SetCrashLogDateTime(kHxNode node, k64s bootDateTime)
{
    return xkHxNode_VTable(node)->VSetCrashLogDateTime(node, bootDateTime);
}

//Deprecated: Use kHxNode_Health to access the node's health service
kInlineFx(kStatus) kHxNode_HealthStats(kHxNode node, kArrayList stats)
{
    kObj(kHxNode, node);

    kCheck(kHealth_UpdateHealthStats(kHxNode_Health(node))); 

    return kHealth_HealthStats(kHxNode_Health(node), stats);
}
kDeprecate(kHxNode_HealthStats)


//Deprecated: Use kHxNode_Health to access the node's health service
kInlineFx(kStatus) kHxNode_ProfileStats(kHxNode node, kArrayList stats)
{
    kObj(kHxNode, node); 
    
    return kHealth_ProfileStats(kHxNode_Health(node), stats); 
}
kDeprecate(kHxNode_ProfileStats)

#endif
