/** 
 * @file    kHxNode.h
 * @brief   Declares the kHxNode class.
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HX_NODE_H
#define K_FIRESYNC_HX_NODE_H

#include <kFireSync/kNodeDef.h>

#include <kApi/Data/kString.h>
#include <kApi/Threads/kTimer.h>
#include <kApi/Utils/kDateTime.h>

#include <kFireSync/Health/kHealth.h>
#include <kFireSync/Health/kHealthProbe.h>

#include <kFireSync/Hardware/kHxNode.x.h>

/** 
 * Dynamic entry point to user application. 
 * 
 * @see         kMain
 * @return      Application exit status. 
 */
typedef kStatus (kCall* kHxNodeMainFx)();

/**
 * @class       kHxNode
 * @extends     kObject
 * @ingroup     kFireSync-Hardware
 * @brief       Abstract base class for hardware node classes.
 * 
 * For the most part, the methods defined in this abstract base class are not intended for 
 * direct use by sensor application software. kHxNode (and its subordinate control objects) 
 * are intended for use in the FireSync Server framework (i.e., kFireSync/Server classes) to 
 * interface with underlying node hardware. The only exceptions to this rule are the 
 * kHxNode_PlatformInstance/kHxNode_FindPlatformInstance methods, which can be used to locate 
 * a suitable kHxNode object. 
 * 
 * In general, the public methods in kHxNode and related classes are not thread-safe; it 
 * is up to the caller to guarantee exclusive access. kHxNode and its subordinate control 
 * objects should be considered as a single handle, designed for control by a single agent. 
 * In specific cases where methods are designed for concurrent access, this guarantee is 
 * noted in method documentation.
 */
//typedef kObject kHxNode;      --forward-declared in kFsDef.x.h

/** 
 * Constructs a hardware node object representing the underlying platform. 
 *
 * Sensor applications that instantiate a server node representing underlying platform hardware 
 * should typically load an assembly that implements a kHxNode-derived singleton object. If such 
 * an assembly has been loaded, this method can be used to construct the kHxNode-derived singleton. 
 * This kHxNode instance can then be accessed via the kHxNode_FindPlatformInstance method.
 * 
 * The node object created by this method does not need to be explicitly destroyed. It will be destroyed 
 * automatically when its assembly is disposed.
 * 
 * @public              @memberof kHxNode
 * @param   argc        Number of arguments in argv.  
 * @param   argv        Arguments; meaning is node-specific.  
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNode_ConstructPlatformInstance(k32s argc, const kChar** argv);

/** 
 * Returns a reference to the hardware platform singleton, if it exists.
 *
 * Sensor applications that instantiate a server node representing underlying platform hardware 
 * should typically load an assembly that implements a kHxNode-derived singleton object. If such 
 * an assembly has been loaded and the singleton has been constructed, this method can be used to 
 * access the kHxNode-derived singleton. This kHxNode instance can then be passed to kSystem_AddServerNode, 
 * to be used as the hardware node to which the server node will bind. 
 * 
 * The node object returned by this method does not need to be destroyed.
 * 
 * @public            @memberof kHxNode
 * @return              Platform hardware singleton, or kNULL if none.
 */
kInlineFx(kHxNode) kHxNode_PlatformInstance()
{
    return kStaticOf(kHxNode)->platformInstance;
}

/** 
 * Locates a hardware node object representing the underlying platform. 
 *
 * This method provides the same result as kHxNode_PlatformInstance, but returns 
 * kERROR_NOT_FOUND if a platform hardware singleton does not exist.
 * 
 * @public              @memberof kHxNode
 * @param   node        Receives hardware node singleton object, if one exists. 
 * @return              Operation status.
 */
kInlineFx(kStatus) kHxNode_FindPlatformInstance(kHxNode* node)
{
    *node = kHxNode_PlatformInstance(); 

    return kIsNull(*node) ? kERROR_NOT_FOUND : kOK; 
}

/** 
 * Uses platform-specific methodology to load the library containing the application. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   library     Receives application library object; should be destoyed by caller.
 * @return              Operation status.
 */
kInlineFx(kStatus) kHxNode_LoadApp(kHxNode node, kDynamicLib* library)
{
   return xkHxNode_VTable(node)->VLoadApp(node, library);
}

/** 
 * Reports whether this node provides storage support.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              kTRUE if file storage is supported.
 */
kInlineFx(kBool) kHxNode_HasStorage(kHxNode node)
{
    kObj(kHxNode, node);

    return obj->hasStorage;
}

/** 
 * Reports the number of network interfaces.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              number of network interfaces.
 */
kInlineFx(kSize) kHxNode_NetInterfaceCount(kHxNode node)
{
    kObj(kHxNode, node);

    return obj->netInterfaceCount;
}

/** 
 * Provides the root storage path for this node (if storage is supported).
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              Root storage path.
 */
kInlineFx(const kChar*) kHxNode_StoragePath(kHxNode node)
{
    kObj(kHxNode, node);

    return kString_Chars(obj->storagePath);
}

/** 
 * Sets the memory allocator used to generate camera messages. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   allocator   Memory allocator.  
 * @return              Operation status.
 */
kInlineFx(kStatus) kHxNode_SetMessageAlloc(kHxNode node, kAlloc allocator)
{
    return xkHxNode_VTable(node)->VSetMessageAlloc(node, allocator);
}

/** 
 * Sets the callback for camera data messages. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   function    Callback function. 
 * @param   receiver    Callback context pointer. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNode_SetDataHandler(kHxNode node, kCallbackFx function, kPointer receiver); 

/** 
 * Registers a callback function for periodic health updates. 
 * 
 * This function allows an application to take advantage of the health polling infrastructure that 
 * already exists within the kHardware library, rather than duplicating that infrastructure. Registered
 * callbacks should be fast; non-negligible blocking will interfere with health processsing.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   function    Callback function. 
 * @param   receiver    Callback context pointer. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNode_AddHealthUpdateHandler(kHxNode node, kCallbackFx function, kPointer receiver);

/** 
 * Unregisters a callback function that was previously registered using kHxNode_AddHealthUpdateHandler. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   function    Callback function. 
 * @param   receiver    Callback context pointer. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNode_RemoveHealthUpdateHandler(kHxNode node, kCallbackFx function, kPointer receiver);

/** 
 * Registers a callback function for periodic environment updates. 
 * 
 * This function allows an application to take advantage of the environment polling infrastructure that 
 * already exists within the kHardware library, rather than duplicating that infrastructure. Registered 
 * callbacks can perform I/O that would normally be unsuitable for frequent health callbacks. (The environment 
 * callback period will typically be 0.2 - 1 Hz, while the health callback period might be 5 Hz or more.)
 * 
 * Note, environmental monitoring callbacks may be disabled in rescue firmware builds, to increase system 
 * stability. In this case, callback registration will succeed, but the specified callbacks will never be 
 * invoked. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   function    Callback function. 
 * @param   receiver    Callback context pointer. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNode_AddEnvironUpdateHandler(kHxNode node, kCallbackFx function, kPointer receiver);

/** 
 * Unregisters a callback function that was previously registered using kHxNode_AddEnvironUpdateHandler. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   function    Callback function. 
 * @param   receiver    Callback context pointer. 
 * @return              Operation status.
 */
kFsFx(kStatus) kHxNode_RemoveEnvironUpdateHandler(kHxNode node, kCallbackFx function, kPointer receiver);

/** 
* Starts health services related to the base node object.
* 
* Must be called after kHxNode_Init and after the health service has been constructed.
* 
* @internal            @memberof kHxNode
* @param   node        Node object.
* @param   health      Node health service.
* @param   alloc       Memory allocator.
* @return              Operation status.
*/
kFsFx(kStatus) kHxNode_StartHealth(kHxNode node, kHealth health, kAlloc alloc);

/** 
* Destroys health services related to the base node object.
* 
* Must be called before the node object is destroyed.
* 
* @internal            @memberof kHxNode
* @param   node        Node object.
* @return              Operation status.
*/
kFsFx(kStatus) kHxNode_StopHealth(kHxNode node);

/** 
* Sets the given LED to the given operational state.
* 
* @internal            @memberof kHxNode
* @param   node        Node object.
* @param   instance    LED instance ID.
* @param   mode        LED operational state.
* @return              Operation status
*/
kFsFx(kStatus) kHxNode_SetLedMode(kHxNode node, kLed instance, kLedMode mode);

/** 
 * Returns the recommended minimum health logging period. 
 * 
 * The underlying hardware may only support health logging at up to a specific rate, depending
 * on capabilities of the storage medium. Application logic responsible for generating health snapshots
 * must respect the recommended minimum period, else risk premature storage failure. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              Minimum supported health logging period (us).
 */
kInlineFx(k64u) kHxNode_MinHealthLogPeriod(kHxNode node)
{
    kObj(kHxNode, node);

    return obj->minHealthLogPeriod; 
}

/** 
 * Returns the recommended thread priority for health logging. 
 * 
 * Ideally, health logging is performed at a lower than normal priority to prevent disruption 
 * of other activities. However, a mixture of low and high priority threads can result in priority 
 * inversion, which in turn can result in deadlock under high CPU load. Accordingly, care must 
 * be taken in the design of any system that makes use of thread priorities for CPU workload 
 * management. For systems that have been audited for adherence to the necessary rules, 
 * a lower than normal priority may be recommended; otherwise, normal priority will be recommended. 
 * 
 * @internal                @memberof kHxNode
 * @param   node            Node object. 
 * @param   priorityClass   Receives recommended priority class.  
 * @param   priorityOffset  Receives recommended priority offset.  
 * @return                  Operation status
 */
kInlineFx(kStatus) kHxNode_HealthLogPriority(kHxNode node, kThreadPriorityClass* priorityClass, k32s* priorityOffset)
{
    kObj(kHxNode, node);
    
    *priorityClass = obj->healthLogPriorityClass;
    *priorityOffset = obj->healthLogPriorityOffset;

    return kOK;
}

/** 
 * Returns health service associated with this node.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              Health service.
 */
kInlineFx(kHealth) kHxNode_Health(kHxNode node)
{
    return xkHxNode_VTable(node)->VHealth(node);
}

/** 
 * Requests that the application should be restarted after exiting. 
 * 
 * Application restarts enable information from error-checking that normally occurs
 * during shutdown to be preserved so that it can be accessed after restart.  It can 
 * also be used (with some risk) as a faster form of reset when debugging. 
 * 
 * This feature may not be supported on every platform. If this feature is not supported, 
 * this method has no effect. 
 * 
 * This method is not thread safe. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 */
kInlineFx(void) kHxNode_RequestAppRestart(kHxNode node)
{
    kObj(kHxNode, node);

    obj->restartRequested = kTRUE;
}

/**
 * Gets the current state of the application restart flag.
 * 
 * The application restart flag is used to track if the application should
 * perform reset or a full reboot.
 * 
 * @param   node        Node object. 
 * @return              KTRUE if the application should perform a soft reset.
 */
kInlineFx(kBool) kHxNode_GetRestartRequest(kHxNode node)
{
    kObj(kHxNode, node);

    return obj->restartRequested;
}

/** 
 * Sets the reported amount of memory that has been leaked on this node. 
 * 
 * In environments that support software-restart (lightweight, software-only reset), this method can be used 
 * during node initialization to record the amount of memory that was leaked during previous sessions. 
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   amount      Total size of memory leaked in previous sessions.  
 */
kInlineFx(void) kHxNode_SetLeakedMemory(kHxNode node, kSize amount)
{
    kObj(kHxNode, node);

    kHealthProbe_SetValue(obj->leakProbe, (k64s)amount);
}

/** 
 * Adds to the total reported amount of memory leaked by this node.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   amount      Size of memory leak. 
 */
kInlineFx(void) kHxNode_AddLeakedMemory(kHxNode node, kSize amount)
{
    kObj(kHxNode, node);

    kHealthProbe_AddValue(obj->leakProbe, (k64s)amount);
}

/** 
 * Reports the total amount of leaked memory.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object.
 */
kInlineFx(kSize) kHxNode_LeakedMemory(kHxNode node)
{
    kObj(kHxNode, node);

    return (kSize) kHealthProbe_Value(obj->leakProbe);
}

/** 
 * Reports whether watchdog services are available on this platform.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              kTRUE if watchdog is available.  
 */
kInlineFx(kBool) kHxNode_HasWatchdog(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->hasWatchdog; 
}

/** 
 * Enables or disables the watchdog timer.
 * 
 * Watchdog timers are disabled by default. Use this method to enable the watchdog timer
 * with the specified timeout period.  
 * 
 * When the watchdog timer is enabled, the kHxNode_UpdateWatchdog method must be called before 
 * the watchdog times out. If the watchdog times out before kHxNode_UpdateWatchdog is called, 
 * the device will reset. 
 * 
 * This method is thread safe.
 * 
 * @internal                @memberof kHxNode
 * @param   node            Node object. 
 * @param   enable          kTRUE to enable the watchdog. 
 * @param   timeoutPeriod   Timeout period (microseconds).  
 * @return                  Operation status. 
 */
kInlineFx(kStatus) kHxNode_EnableWatchdog(kHxNode node, kBool enable, k64u timeoutPeriod)
{
    return xkHxNode_VTable(node)->VEnableWatchdog(node, enable, timeoutPeriod);
}

/** 
 * Updates the watchdog status, optionally resetting the watchdog countdown timer.
 * 
 * If the reason argument is kRESTART_REASON_NONE, this method resets the watchdog countdown timer 
 * using the timeout period specified in the most recent call to kHxNode_EnableWatchdog. Otherwise, the 
 * reason code is logged (if suported by the underlying platform) and the device is allowed to reboot. 
 * 
 * This method is thread safe.
 * 
 * @internal                @memberof kHxNode
 * @param   node            Node object. 
 * @param   reason          Reason code (kRESTART_REASON_NONE to prevent reboot).
 * @return                  Operation status. 
 */
kInlineFx(kStatus) kHxNode_UpdateWatchdog(kHxNode node, kRestartReason reason)
{
    return xkHxNode_VTable(node)->VUpdateWatchdog(node, reason);
}

/** 
 * Resets most mode configuration to defaults.
 * 
 * This function will clear mode settings and device settings, and is intended to be used 
 * in kHardware unit testing only.  
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Operation status.
 */
kInlineFx(kStatus) kHxNode_Clear(kHxNode node)
{
    return xkHxNode_VTable(node)->VClear(node);
}

/** 
 * Prepare for node acquisition and output activities. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Operation status.
 */
kInlineFx(kStatus) kHxNode_Start(kHxNode node)
{
    return xkHxNode_VTable(node)->VStart(node);
}

/** 
 * Begin node acquisition and output activities. 
 *
 * @internal                @memberof kHxNode
 * @param   node            Node object. 
 * @param   startTime       Node start time (FS us), or k64U_NULL for auto-synchronized start.
 * @param   startEncoder    Node start encoder (ticks), or k64S_NULL for auto-synchronized start. 
 * @param   maintainPhase   kTRUE for auto-synchronized start that remains in phase with previous operation.
 * @return                  Operation status.
 */
kInlineFx(kStatus) kHxNode_Engage(kHxNode node, k64u startTime, k64s startEncoder, kBool maintainPhase)
{
    return xkHxNode_VTable(node)->VEngage(node, startTime, startEncoder, maintainPhase);
}

/** 
 * Begins termination of node acquisition and output activities. 
 *  
 * The purpose of this function is to begin the cancellation of any asynchronous background activities. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Operation status.
 */
kInlineFx(kStatus) kHxNode_Disengage(kHxNode node)
{
    return xkHxNode_VTable(node)->VDisengage(node);
} 

/** 
 * Ends node acquisition and output activities. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Operation status.
 */
kInlineFx(kStatus) kHxNode_Stop(kHxNode node)
{
    return xkHxNode_VTable(node)->VStop(node);
} 

/** 
 * Clears cumulative run-time counters. 
 * 
 * Because kHxNode_Start/kHxNode_Stop are used in the implementation of pause/resume, as well 
 * as during error-handling, most counters are not automatically reset during kHxNode_Start. 
 * This function can be used to explicity reset counters. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Operation status.
 */
kInlineFx(kStatus) kHxNode_ClearStats(kHxNode node)
{
    return xkHxNode_VTable(node)->VClearStats(node);
} 

/** 
 * Reports whether the node has an error condition requiring it to be stopped and started again.
 *
 * This method is thread-safe.
 *
 * @internal          @memberof kHxNode
 * @param   node      Node object. 
 * @param   timeout   Amount of time to wait for an error conidition to arise (us). 
 * @return            kOK if a recovery event has been detected. 
 */
kInlineFx(kStatus) kHxNode_WaitForRecoveryEvent(kHxNode node, k64u timeout)
{
    kObj(kHxNode, node);    

    return xkHxNode_VTable(node)->VWaitForRecoveryEvent(node, timeout);
}

/** 
 * Gets the node's device identifier (serial number).
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Device identifier.
 */
kInlineFx(k32u) kHxNode_Id(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->id; 
}

/** 
 * Gets the type of node. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Node type. 
 */
kInlineFx(kNodeType) kHxNode_NodeType(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->nodeType;
}

/** 
 * Gets the node's future device identifier (after reboot).
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Device identifier.
 */
kInlineFx(k32u) kHxNode_NextId(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->nextId; 
}

/** 
 * Sets the node's future device identifier (after reboot).
 * 
 * This setting is established by reading node configuration files during kHxNode construction 
 * and does need to be explicity set. 
 *
 * Call kHxNode_CommitConfig to commit changes to non-volatile memory. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   id      Device identifier.
 * @return          Operation status.
 */
kInlineFx(kStatus) kHxNode_SetNextId(kHxNode node, k32u id)
{
    return xkHxNode_VTable(node)->VSetNextId(node, id);
} 

/** 
 * Gets a value indicating the stamp temperature source. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Returns a value indicating the stamp temperature source. 
 */
kInlineFx(kTempProbeId) kHxNode_MainTempProbe(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->mainProbe; 
}

/** 
 * Specifies whether the internal temperature is selected to present in message stamp. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   probeId Specifies whether the internal temperature or external temperature probe is used. 
 * @return          Operation status.
 */
kInlineFx(kStatus) kHxNode_SetMainTempProbe(kHxNode node, kTempProbeId probeId)
{
    kObj(kHxNode, node); 

    obj->mainProbe = probeId;
    
    return kOK; 
}

/** 
 * Gets the IP configuration with which the device booted. 
 * 
 * Use kHxNode_DynamicIpConfig to get the current/actual IP configuration, which may have changed
 * since boot time due to DHCP activity. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @param   config  Receives IP configuration.
 * @return          Operation status. 
 * @see             kHxNode_DynamicIpConfig
 */
kInlineFx(kStatus) kHxNode_IpConfig(kHxNode node, kSize index, kIpConfig* config)
{
    kObj(kHxNode, node); 

    *config = obj->ipConfig[index]; 
    
    return kOK; 
}

/** 
 * Gets the node's future IP configuration (after reboot).
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @param   config  Receives IP configuration.
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_NextIpConfig(kHxNode node, kSize index, kIpConfig* config)
{
    kObj(kHxNode, node); 

    *config = obj->nextIpConfig[index]; 
    
    return kOK; 
}

/** 
 * Sets the node's future IP configuration (after reboot).
 * 
 * This setting is established by reading node configuration files during kHxNode construction 
 * and does need to be explicity set. 
 *
 * Call kHxNode_CommitConfig to commit changes to non-volatile memory. 
 * 
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @param   config  IP configuration.
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_SetNextIpConfig(kHxNode node, kSize index, const kIpConfig* config)
{
    return xkHxNode_VTable(node)->VSetNextIpConfig(node, index, config);
} 

/** 
 * Waits for network startup complation. The Dynamic Host Configuration Protocol requires
 * some time to retrieve initial address configuration. This function may block and return control
 * once DHCP has finished.
 *
 * This function blocks until startup has completed. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_WaitForNetworkOnline(kHxNode node)
{
    return xkHxNode_VTable(node)->VWaitForNetworkOnline(node);
} 

/** 
 * Retrieve IP configuration by querying the stack.
 * 
 * Call kHxNode_WaitForNetworkOnline prior to this function.
 * 
 * Note, this method can fail if the network adapter is currently being reconfigured. 
 * 
 * This method is thread-safe. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @param   config  IP configuration.
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_DynamicIpConfig(kHxNode node, kSize index, kIpConfig* config)
{
    return xkHxNode_VTable(node)->VDynamicIpConfig(node, index, config);
} 

/** 
 * Gets the node's Ethernet MAC address.
 *
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @param   address Receives MAC address.
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_MacAddress(kHxNode node, kSize index, kMacAddress* address)
{
    kObj(kHxNode, node); 

    *address = obj->macAddress[index]; 

    return kOK; 
} 

/** 
 * Sets boot configuration parameters. 
 *
 * This setting is established by reading node configuration files during kHxNode construction 
 * and does need to be explicity set. 
 *
 * Call kHxNode_CommitConfig to commit changes to non-volatile memory. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   config  Boot configuration parameters. 
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_SetBootConfig(kHxNode node, const kBootConfig* config)
{
    return xkHxNode_VTable(node)->VSetBootConfig(node, config);
} 

/** 
 * Gets the node's power saver timeout.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Power saver timeout. 
 */
kInlineFx(k64u) kHxNode_PowerSaverTimeout(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->powerSaverTimeout;
} 

/** 
 * Sets the node's power saver timeout.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   timeout Timeout parameter. 
 * @return          Power saver timeout. 
 */
kInlineFx(kStatus) kHxNode_SetPowerSaverTimeout(kHxNode node, k64u timeout)
{
    kObj(kHxNode, node); 

    obj->powerSaverTimeout = timeout; 

    return kOK; 
} 

/** 
 * Gets the node's power saver threshold.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Power saver threshold. 
 */
kInlineFx(k64u) kHxNode_PowerSaverThreshold(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->powerSaverThreshold;
} 

/** 
 * Sets the node's power saver threshold.
 *
 * @internal          @memberof kHxNode
 * @param   node      Node object. 
 * @param   threshold Threshold parameter. 
 * @return            Power saver threshold. 
 */
kInlineFx(kStatus) kHxNode_SetPowerSaverThreshold(kHxNode node, k64u threshold)
{
    kObj(kHxNode, node); 

    obj->powerSaverThreshold = threshold; 

    return kOK; 
} 

/** 
 * Reports whether use of higher-powered light-driver set points is enabled.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          kTRUE if enabled.
 */
kInlineFx(kBool) kHxNode_LightDriverHighPowerEnabled(kHxNode node)
{
    kObj(kHxNode, node); 

    return obj->lightDriverHighPowerEnabled;
} 

/** 
 * Enables use of higher-powered light-driver set points.
 *
 * @internal          @memberof kHxNode
 * @param   node      Node object. 
 * @param   enabled   kTRUE to enable. 
 * @return            Power saver threshold. 
 */
kInlineFx(kStatus) kHxNode_EnableLightDriverHighPower(kHxNode node, kBool enabled)
{
    kObj(kHxNode, node); 

    obj->lightDriverHighPowerEnabled = enabled; 

    return xkHxNode_VTable(node)->VEnableLightDriverHighPower(node, enabled);
} 

/** 
 * Gets the node's boot configuration parameters.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   config  Receives boot configuration parameters.
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_BootConfig(kHxNode node, kBootConfig* config)
{
    kObj(kHxNode, node);

    *config = obj->bootConfig;

    return kOK;
}

/** 
 * Gets current network speed.  
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @return          Current network speed. 
 */
kInlineFx(kNetworkSpeed) kHxNode_NetworkSpeed(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return obj->networkSpeed[index];
}

/** 
 * Gets network speed that should take effect on next reboot.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @return          Network speed after reboot.
 */
kInlineFx(kNetworkSpeed) kHxNode_NextNetworkSpeed(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return obj->nextNetworkSpeed[index];
}

/** 
 * Sets network speed that should take effect on next reboot.
 *
 * This setting is established by reading node configuration files during kHxNode construction 
 * and does need to be explicity set. 
 *
 * Call kHxNode_CommitConfig to commit changes to non-volatile memory. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @param   speed   Network speed after reboot.
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_SetNextNetworkSpeed(kHxNode node, kSize index, kNetworkSpeed speed)
{
    return xkHxNode_VTable(node)->VSetNextNetworkSpeed(node, index, speed);
} 

/** 
 * Gets the network speed settings supported by this device. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @return          Bitset of supported network speed settings.
 */
kInlineFx(kNetworkSpeed) kHxNode_NetworkSpeedCapabilities(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return obj->networkSpeedCapabilities[index];
}

/** 
 * Gets the default network speed settings for this device. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   index   Interface index. 
 * @return          Default network speed setting. 
 */
kInlineFx(kNetworkSpeed) kHxNode_DefaultNetworkSpeed(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return obj->defaultNetworkSpeed[index];
}

/**
* Sets the maximum Ethernet transmission rate.
*
* @internal         @memberof kHxNode
* @param   node     Node object.
* @param   index    Interface index. 
* @param   rate     Maximum transmission rate (percentage).
* @return           Operation status.
*/
kInlineFx(kStatus) kHxNode_SetNetworkTransmitRate(kHxNode node, kSize index, k32u rate)
{
    return xkHxNode_VTable(node)->VSetNetworkTransmitRate(node, index, rate);
}

/**
* Reports the maximum Ethernet transmission rate.
*
* @internal         @memberof kHxNode
* @param   node     Node object.
* @param   index    Interface index. 
* @return           Maximum transmission rate (percentage).
*/
kInlineFx(k32u) kHxNode_NetworkTransmitRate(kHxNode node, kSize index)
{
    return xkHxNode_VTable(node)->VNetworkTransmitRate(node, index);
} 

/**
* Reports whether the TransmitRate setting is supported by this device.
*
* @internal         @memberof kHxNode
* @param   node     Node object.
* @param   index    Interface index. 
* @return           kTRUE if supported.
*/
kInlineFx(kBool) kHxNode_NetworkTransmitRateSupported(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return obj->networkTransmitRateSupported[index];
}

/** 
 * Commits changes to device configuration. 
 * 
 * This function notifies the kHxNode instance to write changes to any board-specific files 
 * that must be maintained in addition to the node device configuration file.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_CommitConfig(kHxNode node)
{
    return xkHxNode_VTable(node)->VCommitConfig(node);
}

/** 
 * Clear the firmware region specified. 
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   type    Firmware type. 
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_ClearFirmware(kHxNode node, kNodeFirmwareType type)
{
    return xkHxNode_VTable(node)->VClearFirmware(node, type);
}

/** 
 * Writes the firmware region specified. 
 *
 * @see kNode_LoadFirmwareFromStream.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   type        Firmware type. 
 * @param   stream      Stream from which the firmware is read. 
 * @param   size        Size of the fimware data in the stream. 
 * @param   progress    Progress callback, @see kNode_LoadFirmwareFromStream. 
 * @param   context     Progress context, @see kNode_LoadFirmwareFromStream. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_WriteFirmware(kHxNode node, kNodeFirmwareType type, kStream stream, kSize size, kCallbackFx progress, kPointer context)
{
    return xkHxNode_VTable(node)->VWriteFirmware(node, type, stream, size, progress, context);
}

/**
 * Reloads firmware without affecting non-volatile records or rebooting. 
 *
 * @see kNode_ReloadFirmwareFromStream.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object.
 * @param   type        Firmware type.
 * @param   stream      Stream from which the firmware is read.
 * @param   size        Size of the fimware data in the stream.
 * @return              Operation status.
 */
kInlineFx(kStatus) kHxNode_ReloadFirmware(kHxNode node, kNodeFirmwareType type, kStream stream, kSize size)
{
    return xkHxNode_VTable(node)->VReloadFirmware(node, type, stream, size);
}

/** 
 * Fills in the given node info structure with any known information about this node.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   info        Receives node info; 
 * @return              Operation status. 
 */
kFsFx(kStatus) kHxNode_Info(kHxNode node, kNodeInfo* info);

/** 
 * Reports whether this firmware is operating under rescue mode restrictions (bootloader or rescue image). 
 *
 * @private         @memberof kHxNode
 * @param   node    Node object. 
 * @return          kTRUE if the firmware is operating under rescue mode restrictions. 
 */
kInlineFx(kBool) kHxNode_IsRescue(kHxNode node)
{
    kObj(kHxNode, node);

    return (obj->bootMode == kBOOT_MODE_RESCUE) || (obj->bootMode == kBOOT_MODE_FALLBACK);
} 

/** 
 * Reports whether this node is a virtual (simulated) node. 

 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          kTRUE if the node is a virtual node; kFALSE otherwise. 
 */
kInlineFx(kBool) kHxNode_IsVirtual(kHxNode node)
{    
    kObj(kHxNode, node);

    return (obj->controllerType == kCONTROLLER_TYPE_VIRTUAL); 
}

/** 
 * Reports the number of available processor cores. 

 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Processsor count. 
 */
kInlineFx(kSize) kHxNode_ProcessorCount(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->cpuCount;
}

/** 
 * Adds an item to the log. 
 *
 * @internal            @memberof kHxNode
 * @param    node       Node object. 
 * @param    options    Log options.
 * @param    source     Message source.
 * @param    message    Message to log. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_LogEx(kHxNode node, kLogOption options, const kChar* source, const kChar* message)
{
    return xkHxNode_VTable(node)->VLogEx(node, options, source, message);
}

/** 
 * Adds an item to the log. 
 *
 * @internal            @memberof kHxNode
 * @param    node       Node object. 
 * @param    message    Message to log. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_Log(kHxNode node, const kChar* message)
{
    return kHxNode_LogEx(node, 0, "", message);
}

/** 
 * Destructively reads log messages from the log history queue.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   logItems    Log items kArrayList<kLogItem>. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kHxNode_ReadLogItems(kHxNode node, kArrayList logItems);

/** 
 * Gets all log messages in the log history queue.
 *
 * @internal                @memberof kHxNode
 * @param   node            Node object. 
 * @param   logItems        Log items kArrayList<kLogItem>. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kHxNode_GetLogItems(kHxNode node, kArrayList logItems);

/** 
 * Gets all log messages in the log history records.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   crashLog    Receives crash log object. 
 * @param   allocator   Object allocator. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_GetCrashLog(kHxNode node, kString* crashLog, kAlloc allocator)
{
    return xkHxNode_VTable(node)->VGetCrashLog(node, crashLog, allocator);
} 

/** 
 * Clears the crash log history records.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_ClearCrashLog(kHxNode node)
{
    return xkHxNode_VTable(node)->VClearCrashLog(node);
} 

/**
 * Sets the appropriate fields in the crash log object.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object.
 * @param   version     Node application version.
 * @param   nodeId      Node ID.
 * @return              Operation status.
 */
kInlineFx(kStatus) kHxNode_SetCrashLogAppInfo(kHxNode node, kVersion version, k32u nodeId)
{
    return xkHxNode_VTable(node)->VSetCrashLogAppInfo(node, version, nodeId);
}

/** 
 * Fills in the node statistics items that are known by the kHardware library. 
 *
 * This method is thread-safe.
 *
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   stats       Receives node run-time statistics. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_Stats(kHxNode node, kNodeStats* stats)
{
    return xkHxNode_VTable(node)->VStats(node, stats);
} 

/** 
 * Reports whether the node's primary PL has been successfully programmed.
 * 
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          kTRUE if primary PL successfully programmed.
 */
kInlineFx(kBool) kHxNode_PlLoaded(kHxNode node)
{    
    kObj(kHxNode, node);

    return xkHxNode_VTable(node)->VPlLoaded(node);
}

/** 
 * Reports the node's primary PL firmware version.
 * 
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Primary PL firmware version. 
 */
kInlineFx(kVersion) kHxNode_PlVersion(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->mainPlVersion;
}

/** 
 * Reports the node's primary PL configuration id.
 * 
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Primary PL configuration id. 
 */
kInlineFx(k32u) kHxNode_PlConfig(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->mainPlConfig;
}

/** 
 * Reports whether the node's secondary PL has been successfully programmed.
 * 
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          kTRUE if secondary PL successfully programmed.
 */
kInlineFx(kBool) kHxNode_Pl2Loaded(kHxNode node)
{    
    kObj(kHxNode, node);

    return xkHxNode_VTable(node)->VPl2Loaded(node);
}

/** 
 * Reports the node's secondary PL firmware version.
 * 
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Secondary PL firmware version. 
 */
kInlineFx(kVersion) kHxNode_Pl2Version(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->secondaryPlVersion;
}

/** 
 * Reports the node's secondary PL configuration id.
 * 
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Secondary PL configuration id. 
 */
kInlineFx(k32u) kHxNode_Pl2Config(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->secondaryPlConfig;
}

/** 
 * Reports the node's daughterboard model. 
 * 
 * This method is thread-safe.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Daughterboard model. 
 */
kInlineFx(kDaughterboardModel) kHxNode_DaughterboardModel(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->daughterboardModel;
}

kInlineFx(kStatus) kHxNode_ReadMemory(kHxNode node, kNodeMemorySpace space, k64u offset, kSize length, void* data)
{
    return xkHxNode_VTable(node)->VReadMemory(node, space, offset, length, data);
} 

kInlineFx(kStatus) kHxNode_WriteMemory(kHxNode node, kNodeMemorySpace space, k64u offset, kSize length, const void* data)
{
    return xkHxNode_VTable(node)->VWriteMemory(node, space, offset, length, data);
} 

kInlineFx(kStatus) kHxNode_ReadRegisters(kHxNode node, kRegisterModule moduleType, kSize moduleIndex, kArrayList registers)
{
    return xkHxNode_VTable(node)->VReadRegisters(node, moduleType, moduleIndex, registers);
} 

kInlineFx(kStatus) kHxNode_ReadRegisterOverrides(kHxNode node, kRegisterModule moduleType, kArrayList registers)
{
    return xkHxNode_VTable(node)->VReadRegisterOverrides(node, moduleType, registers);
} 

kInlineFx(kStatus) kHxNode_WriteRegisterOverrides(kHxNode node, kRegisterModule moduleType, kArrayList registers)
{
    return xkHxNode_VTable(node)->VWriteRegisterOverrides(node, moduleType, registers);
} 

kInlineFx(kStatus) kHxNode_I2cRead(kHxNode node, k32u deviceId, k32u address, kByte* data, kSize size)
{
    return xkHxNode_VTable(node)->VI2cRead(node, deviceId, address, data, size);
} 

kInlineFx(kStatus) kHxNode_I2cWrite(kHxNode node, k32u deviceId, k32u address, const kByte* data, kSize size)
{
    return xkHxNode_VTable(node)->VI2cWrite(node, deviceId, address, data, size);
} 

kInlineFx(kStatus) kHxNode_BeginSpi(kHxNode node, kSpiDeviceType deviceType)
{
    return xkHxNode_VTable(node)->VBeginSpi(node, deviceType);
}

kInlineFx(kStatus) kHxNode_EndSpi(kHxNode node)
{
    return xkHxNode_VTable(node)->VEndSpi(node);
}

kInlineFx(kStatus) kHxNode_SpiRead(kHxNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize)
{
    return xkHxNode_VTable(node)->VSpiRead(node, opCode, opSize, data, dataSize);
}

kInlineFx(kStatus) kHxNode_SpiWrite(kHxNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize)
{
    return xkHxNode_VTable(node)->VSpiWrite(node, opCode, opSize, data, dataSize);
}

kInlineFx(kStatus) kHxNode_SetTempControl(kHxNode node, const kTempControlEx* temp)
{
    return xkHxNode_VTable(node)->VSetTempControl(node, temp);
} 

kInlineFx(kStatus) kHxNode_TempControl(kHxNode node, kTempControlEx* temp)
{
    return xkHxNode_VTable(node)->VTempControl(node, temp);
} 

kInlineFx(kSize) kHxNode_EventCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->eventManagers);
}

kInlineFx(kHxEvent) kHxNode_EventAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->eventManagers, index, kHxEvent);
}

kInlineFx(kSize) kHxNode_CameraCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->cameras);
}

kInlineFx(kHxCamera) kHxNode_CameraAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->cameras, index, kHxCamera);
}

kInlineFx(kSize) kHxNode_LightCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->lights);
}

kInlineFx(kHxLight) kHxNode_LightAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->lights, index, kHxLight);
}

kInlineFx(kSize) kHxNode_DigitalCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->digitalOutputs);
}

kInlineFx(kHxDigitalOut) kHxNode_DigitalAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->digitalOutputs, index, kHxDigitalOut);
}

kInlineFx(kSize) kHxNode_AnalogCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->analogOutputs);
}

kInlineFx(kHxAnalogOut) kHxNode_AnalogAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->analogOutputs, index, kHxAnalogOut);
}

kInlineFx(kSize) kHxNode_SerialCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->serialOutputs);
}

kInlineFx(kHxSerialOut) kHxNode_SerialAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->serialOutputs, index, kHxSerialOut);
}

kInlineFx(kSize) kHxNode_ProjectorCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->projectors);
}

kInlineFx(kHxProjector) kHxNode_ProjectorAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->projectors, index, kHxProjector);
}

kInlineFx(kSize) kHxNode_IoTestCount(kHxNode node)
{
    kObj(kHxNode, node);

    return kArrayList_Count(obj->ioTests);
}

kInlineFx(kHxIoTest) kHxNode_IoTestAt(kHxNode node, kSize index)
{
    kObj(kHxNode, node);

    return kArrayList_AsT(obj->ioTests, index, kHxIoTest);
}

kInlineFx(kHxEncoder) kHxNode_Encoder(kHxNode node)
{
    kObj(kHxNode, node);

    return obj->encoder;
}

kInlineFx(kHxGpio) kHxNode_Gpio(kHxNode node)
{
    kObj(kHxNode, node);

    return obj->gpio;
}

kInlineFx(kHxTestJig) kHxNode_TestJig(kHxNode node)
{
    kObj(kHxNode, node);

    return obj->testJig;
}

kInlineFx(kStatus) kHxNode_EnableLed(kHxNode node, kLed instance, kBool enabled)
{
    return xkHxNode_VTable(node)->VEnableLed(node, instance, enabled);
} 

kInlineFx(kControllerType) kHxNode_ControllerType(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->controllerType;
}

kInlineFx(k64s) kHxNode_EncoderCounter(kHxNode node)
{
    return xkHxNode_VTable(node)->VEncoderCounter(node);
} 

kInlineFx(k64u) kHxNode_TimeCounter(kHxNode node)
{
    return xkHxNode_VTable(node)->VTimeCounter(node);
} 

kInlineFx(kSize) kHxNode_LocalDigitalInputCount(kHxNode node)
{    
    //currently hardcoded, based on PL documentation; should we read this from somewhere?
    return 2; 
}

kInlineFx(kSize) kHxNode_NetworkDigitalInputCount(kHxNode node)
{
    //currently hardcoded, based on PL documentation; should we read this from somewhere?
    return 4; 
}

kInlineFx(kBool) kHxNode_HasPeltier(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->hasPeltier;
}

kInlineFx(kBool) kHxNode_HasExternalTempProbe(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->hasExternalTempProbe;
}

kInlineFx(kSize) kHxNode_FanCount(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->fanCount;
}

kInlineFx(k64u) kHxNode_CameraDataBandwidth(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->cameraDataBandwidth; 
}

/** 
 * Sets the node's preferred synchronization source.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @param   source  Synchronization source. 
 * @return          Operation status. 
 */
kInlineFx(kStatus) kHxNode_SetSyncSource(kHxNode node, kSyncSource source)
{
    return xkHxNode_VTable(node)->VSetSyncSource(node, source);
} 

/** 
 * Gets the node's preferred synchronization source.
 *
 * @internal        @memberof kHxNode
 * @param   node    Node object. 
 * @return          Synchronization source. 
 */
kInlineFx(kSyncSource) kHxNode_SyncSource(kHxNode node)
{    
    kObj(kHxNode, node);

    return obj->syncSource; 
}

kInlineFx(kStatus) kHxNode_SetEventHandler(kHxNode node, kCallbackFx function, kPointer receiver, kAlloc eventAlloc)
{
    return xkHxNode_VTable(node)->VSetEventHandler(node, function, receiver, eventAlloc);
} 

kInlineFx(kStatus) kHxNode_SetLightDriverControl(kHxNode node, k64u key, kSize index, kBool enabled)
{
    return xkHxNode_VTable(node)->VSetLightDriverControl(node, key, index, enabled);
} 

kInlineFx(kStatus) kHxNode_GetLightDriverControl(kHxNode node, k64u key, kSize index, kBool* enabled)
{
    return xkHxNode_VTable(node)->VGetLightDriverControl(node, key, index, enabled);
} 

kInlineFx(kStatus) kHxNode_SetLightDriverPower(kHxNode node, k64u key, kSize index, k32u power, kBool commit)
{
    return xkHxNode_VTable(node)->VSetLightDriverPower(node, key, index, power, commit);
} 

kInlineFx(kStatus) kHxNode_GetLightDriverPower(kHxNode node, k64u key, kSize index, k32u* power)
{
    return xkHxNode_VTable(node)->VGetLightDriverPower(node, key, index, power);
} 

kInlineFx(kStatus) kHxNode_SetLightDriverCurrentLimit(kHxNode node, k64u key, kSize index, k32u currentLimit, kBool commit)
{
    return xkHxNode_VTable(node)->VSetLightDriverCurrentLimit(node, key, index, currentLimit, commit);
} 

kInlineFx(kStatus) kHxNode_GetLightDriverCurrentLimit(kHxNode node, k64u key, kSize index, k32u* currentLimit)
{
    return xkHxNode_VTable(node)->VGetLightDriverCurrentLimit(node, key, index, currentLimit);
} 

kInlineFx(kStatus) kHxNode_GetLightDriverRemainingSlotCount(kHxNode node, k64u key, kSize index, kSize* remainingCount)
{
    return xkHxNode_VTable(node)->VGetLightDriverRemainingSlotCount(node, key, index, remainingCount);
} 

kInlineFx(kStatus) kHxNode_SetLightDriverInfo(kHxNode node, k64u key, kSize index, kLightModel model, kVersion revision, k32u deviceId)
{
    return xkHxNode_VTable(node)->VSetLightDriverInfo(node, key, index, model, revision, deviceId);
} 

kInlineFx(kStatus) kHxNode_GetLightDriverInfo(kHxNode node, k64u key, kSize index, kLightModel* model, kVersion* revision, k32u* deviceId)
{
    return xkHxNode_VTable(node)->VGetLightDriverInfo(node, key, index, model, revision, deviceId);
} 

kInlineFx(kStatus) kHxNode_SetLightDriverCalibration(kHxNode node, kSize index, k64u key, kDataTree ldCal, kBool commit)
{
    return xkHxNode_VTable(node)->VSetLightDriverCalibration(node, index, key, ldCal, commit);
} 

kInlineFx(kStatus) kHxNode_GetLightDriverCalibration(kHxNode node, kSize index, k64u key, kDataTree* ldCal, kAlloc alloc)
{
    return xkHxNode_VTable(node)->VGetLightDriverCalibration(node, index, key, ldCal, alloc);
} 

kInlineFx(kStatus) kHxNode_ReadLightDriverTemperature(kHxNode node, kSize index, k64u key, k32s* temperature)
{
    return xkHxNode_VTable(node)->VReadLightDriverTemperature(node, index, key, temperature);
} 

/** 
 * Set the current calendar date/time for the node.
 * 
 * This method adjusts the date/time maintained by the node instance. The effect is not persistent; 
 * underlying operating system and/or hardware clocks are not modified.  
 * 
 * For hardware singleton nodes, this method will affect the future results of kDateTime_Now. 
 * 
 * This method is thread-safe.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @param   dateTime    Current calender date-time.
 * @return              Operation status. 
 */
kFsFx(kStatus) kHxNode_SetDateTime(kHxNode node, kDateTime dateTime);

/** 
 * Gets the current calendar date/time for the node.
 * 
 * This method accesses the date/time maintained by the node instance. For hardware
 * singleton nodes, the value reported by this method is equivalent to the value 
 * reported by kDateTime_Now. 
 * 
 * This method is thread-safe.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              Current calender date-time. 
 */
kFsFx(kDateTime) kHxNode_DateTime(kHxNode node);

/** 
 * Increments the node input counter.
 * 
 * Refer to @ref kHxNode_InputCounter for details. 
 * 
 * This method is thread-safe.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_IncrementInputCounter(kHxNode node)
{    
    kObj(kHxNode, node);

    kAtomic32s_Increment(&obj->inputCounter);

    return kOK;
}

/** 
 * Reports the current input counter value.
 * 
 * Cameras have the ability to track and report input events that have occurred since the previous 
 * camera frame. The type of event to be tracked is specified individually for each camera via 
 * the kCamera_SetInputCounterSource method. Cameras with an input counter source of type 
 * kCAMERA_INPUT_COUNTER_SOURCE_SOFTWARE will track changes to the node's software input counter. 
 * The software input counter can be incremented by calling this method. 
 * 
 * Input counter changes are reported in camera message stamps. The InputCounter bits within the 
 * kStamp.status field reflect the number of events recorded since the previous camera frame (or since 
 * acquisition start, for the first frame). The reported event count is limited by the number of available 
 * stamp status bits (as of this writing, 2). If the number of events exceeds the available bits, the 
 * reported count will saturate.  
 * 
 * Software input counters are sampled when camera messages are processed by camera drivers 
 * rather than at frame acquisition time. As such, software input counter sample timing is somewhat 
 * less deterministic than hardware input counter sample timing.
 * 
 * This method is thread-safe.
 * 
 * @internal            @memberof kHxNode
 * @param   node        Node object. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kHxNode_InputCounter(kHxNode node)
{    
    kObj(kHxNode, node);

    return kAtomic32s_Get(&obj->inputCounter);
}

#endif
