/** 
 * @file    kPlot.h
 * @brief   Declares the kPlot class. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_API_PLOT_H
#define K_API_PLOT_H

#include <kApi/Data/kImage.h>
#include <kFireSync/Data/kGraphic.h>
#include <kFireSync/Data/kPlot.x.h>

/**
* @class       kPlot
* @extends     kObject
* @ingroup     kFireSync-Data
* @brief       Represents a multi-layer collection of images and/or vector graphics.
* 
* Note that unlike many kApi data collections, kPlot owns the data items that are added to it. 
* Accordingly, kObject_Destroy will destroy both a plot object and any plot data items. 
* 
* kPlot supports the kObject_Clone and kObject_Size methods.
* 
* kPlot supports the kdat5 and kdat6 serialization protocols. 
*/
//typedef kObject kPlot;            --forward-declared in kFsDef.x.h

/** 
 * Constructs a plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Destination for the constructed object handle.  
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_Construct(kPlot* plot, kObject allocator);

/** 
 * Adds a new item to the plot. 
 *
 * Items are rendered in the order that they are added to the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   label       Label for the new plot item.
 * @param   item        Pointer to data object handle (note: ownership is transferred).
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_Add(kPlot plot, const kChar* label, const kObject* item);

/** 
 * Defines the world coordinate system for the plot. 
 *
 * World coordinates determine axis scaling and affect the rendering of graphic objects. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   x           World coorindate x-origin.   
 * @param   y           World coorindate y-origin.   
 * @param   width       World coorindate width.   
 * @param   height      World coorindate height.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetWorld(kPlot plot, k32f x, k32f y, k32f width, k32f height);

/** 
 * Defines the world coordinate system for the plot in 3D. 
 *
 * World coordinates determine axis scaling and affect the rendering of graphic objects. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   x           World coorindate x-origin.   
 * @param   y           World coorindate y-origin.   
 * @param   z           World coorindate z-origin.   
 * @param   width       World coorindate width.   
 * @param   height      World coorindate height.   
 * @param   depth       World coorindate depth.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetWorld3d(kPlot plot, k32f x, k32f y, k32f z, k32f width, k32f height, k32f depth);

/** 
 * Sets the background color for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   color       Background color.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetBackColor(kPlot plot, kColor color);

/** 
 * Sets titles for the main heading, x-axis, and y-axis. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   title       Heading label.   
 * @param   xTitle      X-axis label.   
 * @param   yTitle      Y-axis label.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetTitles(kPlot plot, const kChar* title, const kChar* xTitle, const kChar* yTitle);

/** 
 * Sets titles for the main heading
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   title       main heading title.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetTitle(kPlot plot, const kChar* title);

/** 
 * Sets label for the x-axis
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   title       X-axis label.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetXTitle(kPlot plot, const kChar* title);

/** 
 * Sets label for the y-axis
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   title       Y-axis label.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetYTitle(kPlot plot, const kChar* title);

/** 
 * Sets label for the z-axis
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   title       Z-axis label.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetZTitle(kPlot plot, const kChar* title);

/** 
 * Sets title for the profiler
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   title       profiler label.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kPlot_SetProfilerTitle(kPlot plot, const kChar* title);

/** 
 * Returns the world coordinates for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the world coordinates for the plot. 
 */
kInlineFx(k32f) kPlot_X(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->x;
}

/** 
 * Returns the world coordinates for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the world coordinates for the plot. 
 */
kInlineFx(k32f) kPlot_Y(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->y;
}

/** 
 * Returns the world coordinates for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the world coordinates for the plot. 
 */
kInlineFx(k32f) kPlot_Z(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->z;
}

/** 
 * Returns the world coordinates for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the world coordinates for the plot. 
 */
kInlineFx(k32f) kPlot_Width(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->width;
}

/** 
 * Returns the world coordinates for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the world coordinates for the plot. 
 */
kInlineFx(k32f) kPlot_Height(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->height;
}

/** 
 * Returns the world coordinates for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the world coordinates for the plot. 
 */
kInlineFx(k32f) kPlot_Depth(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->depth;
}

/** 
 * Returns the background color for the plot. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the background color for the plot. 
 */
kInlineFx(kColor) kPlot_BackColor(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->backColor;
}

/** 
 * Returns the main heading. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the main heading. 
 */
kInlineFx(const kChar*) kPlot_Title(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->title;
}

/** 
 * Returns the x-axis label. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the x-axis label. 
 */
kInlineFx(const kChar*) kPlot_XTitle(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->xTitle;
}

/** 
 * Returns the y-axis label. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the y-axis label. 
 */
kInlineFx(const kChar*) kPlot_YTitle(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->yTitle;
}

/** 
 * Returns the z-axis label. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the z-axis label. 
 */
kInlineFx(const kChar*) kPlot_ZTitle(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->zTitle;
}

/** 
 * Returns the profiler title. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the profiler title. 
 */
kInlineFx(const kChar*) kPlot_ProfilerTitle(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->profilerTitle;
}

/** 
 * Returns the total count of items in the plot.
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @return              Returns the total count of items in the plot.
 */
kInlineFx(kSize) kPlot_Count(kPlot plot)
{
    kObj(kPlot, plot); 

    return obj->count;
}

/** 
 * Returns plot item name at the specified index. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   index       Item index.   
 * @return              Returns plot item name at the specified index. 
 */
kInlineFx(const kChar*) kPlot_NameAt(kPlot plot, kSize index)
{
    return xkPlot_ItemAt(plot, index)->name;
}

/** 
 * Returns plot item data at the specified index. 
 *
 * @public              @memberof kPlot
 * @param   plot        Plot object.   
 * @param   index       Item index.   
 * @return              Returns plot item data at the specified index. 
 */
kInlineFx(kObject) kPlot_DataAt(kPlot plot, kSize index)
{
    return xkPlot_ItemAt(plot, index)->object;
}

#endif
