/** 
 * @file    kMsgInfo.h
 * @brief   Declares the kMsgInfo type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_MSG_INFO_H
#define K_FIRESYNC_MSG_INFO_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Data/kMsgInfo.x.h>

/**
 * @class   kMsgInfo
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Represents message source information and an attached message.
 * 
 * kMsgInfo is a top-level container for a FireSync data message. Message content can be accessed via the 
 * Message property.  
 * 
 * Each kMsgInfo instance also contains a small amount of metadata related to message routing. The Source property 
 * describes the origin of a message (node id, block id, port id). The Tag property is provided for use within data 
 * processing envionments, for implementation-specific purposes; this property should typically be ignored 
 * in application code. 
 * 
 * The following example illustrates how to log various information related to a standard FireSync video message. 
 @code {.c}
kStatus PrintVideoMessage(kMsgInfo msgInfo)
{
    kMsgSet message = kMsgInfo_Message(msgInfo); 
    kMsgSource source = kMsgInfo_Source(msgInfo); 

    kLogf("Received %s from Node-%u, Block-%u, Port-%u", kType_Name(kObject_Type(message)), 
        source.nodeId, source.blockId, source.portId); 

    //verify that the message is of type kMsg (the most common message type; used for video messages)
    if (kObject_Is(message, kTypeOf(kMsg)))
    {
        const kStamp* stamp = kMsg_Stamp(message); 
        kObject content = kMsg_Data(message); 

        kLogf("  Time: %llu", stamp->time); 
        kLogf("  Encoder: %lld", stamp->encoder); 
        kLogf("  Content Type: %s", kType_Name(kObject_Type(content))); 

        if (kObject_Is(content, kTypeOf(kImage)))
        {
            kLogf("  Pixel Type: %s", kType_Name(kImage_PixelType(content))); 
            kLogf("  Width: %u", kImage_Width(content)); 
            kLogf("  Height: %u", kImage_Height(content)); 
        }
    }

    return kOK; 
}

@endcode

 */
//typedef kObject kMsgInfo;        --forward-declared in kFsDef.x.h

/** 
 * Constructs a kMsgInfo object. 
 *
 * @public              @memberof kMsgInfo
 * @param   info        Receives the constructed kMsgInfo object. 
 * @param   source      Message source. 
 * @param   tag         Message tag.  
 * @param   message     Message content. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
 kFsFx(kStatus) kMsgInfo_Construct(kMsgInfo* info, kMsgSource source, kSize tag, kMsgSet message, kAlloc allocator); 

/** 
 * Sets the message source. 
 *
 * @public              @memberof kMsgInfo
 * @param   info        kMsgInfo object. 
 * @param   source      Message source. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kMsgInfo_SetSource(kMsgInfo info, kMsgSource source)
{
    kObj(kMsgInfo, info); 

    obj->source = source; 

    return kOK; 
}

/** 
 * Gets the message source. 
 *
 * @public              @memberof kMsgInfo
 * @param   info        kMsgInfo object. 
 * @return              Message source. 
 */
kInlineFx(kMsgSource) kMsgInfo_Source(kMsgInfo info)
{
    kObj(kMsgInfo, info); 

    return obj->source; 
}

/** 
 * Sets the message tag. 
 * 
 * The message tag is used in the implementation of FireSync message processing 
 * machinery; this value should be ignored by client software receiving messages 
 * from a remote node. 
 * 
 * Note: when kMsgInfo objects are deserialized, only the lower 32-bits of the tag 
 * are preserved. 
 *
 * @public              @memberof kMsgInfo
 * @param   info        kMsgInfo object. 
 * @param   tag         Message tag. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kMsgInfo_SetTag(kMsgInfo info, kSize tag)
{
    kObj(kMsgInfo, info); 

    obj->tag = tag; 

    return kOK; 
}

/** 
 * Gets the message tag. 
 * 
 * The message tag is used in the implementation of FireSync message processing 
 * machinery; this value should be ignored by client software receiving messages 
 * from a remote node. 
 *
 * @public              @memberof kMsgInfo
 * @param   info        kMsgInfo object. 
 * @return              Message tag. 
 */
kInlineFx(kSize) kMsgInfo_Tag(kMsgInfo info)
{
    kObj(kMsgInfo, info); 

    return obj->tag; 
}

/** 
 * Sets the message object.  
 * 
 * @public              @memberof kMsgInfo
 * @param   info        kMsgInfo object. 
 * @param   message     Message object. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kMsgInfo_SetMessage(kMsgInfo info, kMsgSet message)
{
    kObj(kMsgInfo, info); 

    obj->message = message; 

    return kOK; 
}

/** 
 * Gets the message object.  
 * 
 * @public              @memberof kMsgInfo
 * @param   info        kMsgInfo object. 
 * @return              Message object. 
 */
kInlineFx(kMsgSet) kMsgInfo_Message(kMsgInfo info)
{
    kObj(kMsgInfo, info); 

    return obj->message; 
}

#endif
