/** 
 * @file    kCompressedPhase.h
 * @brief   Declares the kCompressedPhase type. 
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_COMPRESSED_PHASE_H
#define K_FIRESYNC_COMPRESSED_PHASE_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/Data/kCompressedPhase.x.h>

/**
 * @class   kCompressedPhaseStream
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Represents a stream within a kCompressedPhase object. 
 * 
 * kCompressedPhaseStream is not a kStream-derived type. Rather, it represents one 'stream' or 'channel'
 * of information within a compressed phase dataset. 
 */
//typedef kObject kCompressedPhaseStream;            --forward-declared in kFsDef.x.h  

/**
 * @class   kCompressedPhase
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Represents compressed phase (and/or intensity) data. 
 * 
 * The kCompressedPhase class is a container for compressed phase data. The purpose of this container is 
 * to support the efficient movement of compressed phase data from the FireSync camera driver to remote hosts 
 * via lightweight object serialization. In conjunction with its helper classes, such as kCompressedPhaseReader, 
 * kCompressedPhase supports packing (writing) or unpacking (reading) compressed phase data from the various 
 * bit-packed encodings used by FireSync PL. However, the kCompressedPhase class does not perform 
 * compression/decompression, support for which is provided in the kVision library. 
 * 
 * Support for packing/writing (or attaching) compressed phase data is intended for use within FireSync camera 
 * drivers and unit/integration tests; it is not intended for use in other contexts. Support for unpacking/reading 
 * compressed data is provided so that decompression logic can rely on a simple API for extracting data from 
 * the various bit-packed, packetized, PL stream encodings.   
 * 
 * The example below illustrates use of kCompressedPhase and kCompressedPhaseReader to extra a phase delta list 
 * from a compressed phase dataset. 
 * 
 * @code {.c}
 * 
 * kStatus ExtractPhaseDeltaList(kCompressedPhase dataset, kArrayList deltaList)
 * {
 *     kCompressedPhaseStream stream = kNULL; 
 *     kCompressedPhaseReader reader = kNULL; 
 *     k32s deltaItem; 
 * 
 *     kTry
 *     {
 *         //kCompressedPhaseReader supports forward read-only iteration over a compressed phase stream; 
 *         //normally the reader would be reused, but for this example we'll construct/destroy it here
 *         kTest(kCompressedPhaseReader_Construct(&reader, kNULL));
 * 
 *         //locate the desired stream
 *         kTest(kCompressedPhase_FindStream(dataset, kCOMPRESSED_PHASE_STREAM_ID_PHASE_DELTA, &stream));
 * 
 *         //log the bit depth, for informational purposes
 *         kCompressedPhaseEncoding encoding = kCompressedPhase_Encoding(dataset, stream); 
 *         kLogf("Delta bit depth: %u", kCompressedPhaseEncoding_ValueBitDepth(encoding));
 * 
 *         //get the number of entries in the stream        
 *         kSize itemCount = kCompressedPhase_ItemCount(dataset, stream);
 * 
 *         //set read iterator to beginning of desired stream
 *         kTest(kCompressedPhaseReader_Begin(reader, dataset, stream));
 * 
 *         //init output container
 *         kTest(kArrayList_Allocate(deltaList, kTypeOf(k32s), itemCount)); 
 * 
 *         //read out entries
 *         for (kSize i = 0; i < itemCount; ++i)
 *         {
 *             kTest(kCompressedPhaseReader_ReadDelta(reader, &deltaItem));
 *             kTest(kArrayList_AddT(deltaList, &deltaItem));
 *         }
 *     }
 *     kFinally
 *     {
 *         kObject_Destroy(reader);
 *         kEndFinally();
 *     }
 * 
 *     return kOK;
 * }
 *  
 * @endcode
 * 
 * kCompressedPhase supports the kObject_Clone and kObject_Size methods.
 *
 * kCompressedPhase supports the kdat6 serialization protocol.
 */
//typedef kObject kCompressedPhase;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a kCompressedPhase object.
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Destination for the constructed object handle. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_Construct(kCompressedPhase* data, kAlloc allocator); 

/** 
 * Returns the object to a default state (e.g., removes all streams).
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_Clear(kCompressedPhase data);

/** 
 * Sets the subframe index for this object. 
 * 
 * Compressed phase data is usually subdivided into multiple subframes. Each subframe 
 * has a unique index that can be used for identification purposes.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   index       Subframe index.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kCompressedPhase_SetSubframeIndex(kCompressedPhase data, kSize index)
{
    kObj(kCompressedPhase, data);

    obj->subframeIndex = index;
    
    return kOK; 
}

/** 
 * Reports the subframe index for this object. 
 * 
 * Compressed phase data is usually subdivided into multiple subframes. Each subframe 
 * has a unique index that can be used for identification purposes.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              Subframe index.
 */
kInlineFx(kSize) kCompressedPhase_SubframeIndex(kCompressedPhase data)
{
    kObj(kCompressedPhase, data);

    return obj->subframeIndex;
}

/** 
 * Sets the final subframe flag.
 * 
 * Compressed phase data is usually subdivided into multiple subframes. The final subframe
 * flag is used to denote the last subframe in the compressed phase data frame.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   isFinal     Is this subframe the final subframe in the sequence?
 * @return              Operation status. 
 */
kInlineFx(kStatus) kCompressedPhase_SetFinalSubframe(kCompressedPhase data, kBool isFinal)
{
    kObj(kCompressedPhase, data);

    obj->isFinalSubframe = isFinal;

    return kOK;
}

/** 
 * Reports whether the final subframe flag is set.
 * 
 * Compressed phase data is usually subdivided into multiple subframes. The final subframe
 * flag is used to denote the last subframe in the compressed phase data frame.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              kTRUE if the last subframe flag is set.
 */
kInlineFx(kBool) kCompressedPhase_IsFinalSubframe(kCompressedPhase data)
{
    kObj(kCompressedPhase, data);

    return obj->isFinalSubframe;
}

/** 
 * Sets phase prediction value.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   prediction  Phase prediction value.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_SetPhasePrediction(kCompressedPhase data, k32u prediction); 

/** 
 * Reports phase prediction value.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              Phase prediction value.
 */
kInlineFx(k32u) kCompressedPhase_PhasePrediction(kCompressedPhase data)
{
    kObj(kCompressedPhase, data); 

    return obj->phasePrediction; 
}

/** 
 * Sets the uncompressed length of this subframe.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   length      Uncompressed length of subframe, in pixels.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kCompressedPhase_SetSubframeLength(kCompressedPhase data, kSize length)
{
    kObj(kCompressedPhase, data); 

    obj->subframeLength = length; 

    return kOK;
}

/** 
 * Reports the uncompressed length of this subframe.
 * 
 * @public          @memberof kCompressedPhase
 * @param   data    Compressed phase object.
 * @return          Uncompressed length of subframe, in pixels.
 */
kInlineFx(kSize) kCompressedPhase_SubframeLength(kCompressedPhase data)
{
    kObj(kCompressedPhase, data); 

    return obj->subframeLength; 
}

/** 
 * Sets the length of the specified uncompressed 2D dataset dimension.
 * 
 * This information is not strictly required for decoding, but can be helpful to recreate 
 * a 2D array representation of the data. The lengths reported by this method refer to the 
 * total dimensions of the original uncompressed object (which may be represented by multiple 
 * compressed phase subframes).
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   dimension   Dimension index (0 or 1).
 * @param   length      Uncompressed dimension length.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_SetLength(kCompressedPhase data, kSize dimension, kSize length); 

/** 
 * Reports the length of the specified uncompressed 2D dataset dimension.
 * 
 * This information is not strictly required for decoding, but can be helpful to recreate 
 * a 2D array representation of the data. The lengths reported by this method refer to the 
 * total dimensions of the original uncompressed object (which may be represented by multiple 
 * compressed phase subframes).
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   dimension   Dimension index (0 or 1).
 * @return              Uncompressed dimension length.
 */
kFsFx(kSize) kCompressedPhase_Length(kCompressedPhase data, kSize dimension); 

/** 
 * Adds a stream to the compressed phase dataset.
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   streamId    Stream identifier.
 * @param   encoding    Stream encoding style.
 * @param   stream      Receives stream handle (optional; can be null).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_DefineStream(kCompressedPhase data, kCompressedPhaseStreamId streamId, kCompressedPhaseEncoding encoding, kCompressedPhaseStream* stream);

/** 
 * Prepares the compressed phase object to begin external packet attachment.
 * 
 * This method must be called before kCompressedPhase_AttachPacket. 
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_BeginAttach(kCompressedPhase data); 

/** 
 * Attaches one external packet to the compressed phase object.
 * 
 * kCompressedPhase_BeginAttach must be called before packets can be attached.
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream to which packet should be attached.
 * @param   packet      Pointer to packet memory.
 * @param   packetSize  Size (in bytes) of attached packet. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_AttachPacket(kCompressedPhase data, kCompressedPhaseStream stream, const void* packet, kSize packetSize); 

/** 
 * Sets the total number of compressed items in the stream.
 * 
 * This method must be called when external packets are attached, to inform kCompressedPhase about how 
 * many items are included in the stream. It is only necessary to call this method once for each stream. This 
 * method is typically called after attaching the last packet in the stream. 
 * 
 * This method should not be used when writing individual items to the stream (e.g., kCompressedPhase_BeginWrite, 
 * kCompressedPhase_WriteDelta. 
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream to which packet should be attached.
 * @param   itemCount   Total number of compressed items in the stream.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_SetItemCount(kCompressedPhase data, kCompressedPhaseStream stream, kSize itemCount); 

/** 
 * Prepares the compressed phase object to begin writing. 
 * 
 * This method must be called before stream formatting methods such as kCompressedPhase_WriteDelta. The 
 * kCompressedPhase_EndWrite method must be called when writing is complete.
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_BeginWrite(kCompressedPhase data); 

/** 
 * Writes a delta item to a stream. 
 * 
 * kCompressedPhase_BeginWrite must be called before writing individual values. kCompressedPhase_EndWrite must 
 * be called when writing is complete.
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @param   delta       Delta value. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_WriteDelta(kCompressedPhase data, kCompressedPhaseStream stream, k32s delta); 

/** 
 * Writes a "fail" item to a stream. 
 * 
 * This method can be used to write to either "fail 1st" or "fail 2nd" streams.
 * 
 * kCompressedPhase_BeginWrite must be called before writing individual values. kCompressedPhase_EndWrite must 
 * be called when writing is complete.
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @param   index       Index value.  
 * @param   value       Fail value. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_WriteFail(kCompressedPhase data, kCompressedPhaseStream stream, k32s index, k32s value); 

/** 
 * Writes a null item to a stream. 
 * 
 * kCompressedPhase_BeginWrite must be called before writing individual values. kCompressedPhase_EndWrite must 
 * be called when writing is complete.
 *
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @param   index       Index value.  
 * @param   count       Null count.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_WriteNull(kCompressedPhase data, kCompressedPhaseStream stream, k32s index, k32s count); 

/** 
 * Completes writing and flushes streams.
 * 
 * This method must be called after stream writing is complete.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhase_EndWrite(kCompressedPhase data); 

/** 
 * Find the stream with the specified stream identifier, if it is present in the compressed phase dataset. 
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   streamId    Stream identifier. 
 * @param   stream      Receives stream handle, if present. 
 * @return              Operation status (kERROR_NOT_FOUND if stream type not present).
 */
kFsFx(kStatus) kCompressedPhase_FindStream(kCompressedPhase data, kCompressedPhaseStreamId streamId, kCompressedPhaseStream* stream);

/** 
 * Reads out all phase streams into lists. 
 * 
 * This method provides an optimized alternative to using kCompressedPhaseReader. 
 * 
 * The context object generated by the this method can be passed back in to future invocations of this 
 * method (or kCompressedPhase_ReadIntensityStreams). Reusing this context object can reduce memory allocation
 * overhead. The context object must be disposed by the caller when no longer needed.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   delta       Receives delta items (k32s). 
 * @param   fail1       Receives fail1 items (kPair32s). 
 * @param   fail2       Receives fail2 items (kPair32s). 
 * @param   null        Receives null items (kPair32s).
 * @param   context     Optionally receives reusable processing context object (must be disposed when no longer needed).
 * @return              Operation status (kERROR_NOT_FOUND if stream type not present).
 */
kFsFx(kStatus) kCompressedPhase_ReadPhaseStreams(kCompressedPhase data, kArrayList delta, kArrayList fail1, kArrayList fail2, kArrayList null, kObject* context);

/** 
 * Reads out all intensity streams into lists. 
 * 
 * This method provides an optimized alternative to using kCompressedPhaseReader. 
 * 
 * The context object generated by the this method can be passed back in to future invocations of this 
 * method (or kCompressedPhase_ReadPhaseStreams). Reusing this context object can reduce memory allocation
 * overhead. The context object must be disposed by the caller when no longer needed.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   delta       Receives delta items (k32s). 
 * @param   fail1       Receives fail1 items (kPair32s). 
 * @param   fail2       Receives fail2 items (kPair32s). 
 * @param   null        Receives null items (kPair32s).
 * @param   context     Receives reusable processing context object (must be disposed when no longer needed).
 * @return              Operation status (kERROR_NOT_FOUND if stream type not present).
 */
kFsFx(kStatus) kCompressedPhase_ReadIntensityStreams(kCompressedPhase data, kArrayList delta, kArrayList fail1, kArrayList fail2, kArrayList null, kObject* context);

/** 
 * Reports whether the compressed phase dataset has the specified stream type.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   streamId    Stream identifier. 
 * @return              kTRUE if the compressed phase dataset has the specified stream type. 
 */
kInlineFx(kBool) kCompressedPhase_HasStream(kCompressedPhase data, kCompressedPhaseStreamId streamId)
{
    return kSuccess(kCompressedPhase_FindStream(data, streamId, kNULL)); 
}

/** 
 * Reports the number of stream present in the compressed phase dataset.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @return              Count of streams in the dataset.
 */
kInlineFx(kSize) kCompressedPhase_StreamCount(kCompressedPhase data)
{
    kObj(kCompressedPhase, data);

    return kArrayList_Count(obj->streams); 
}

/** 
 * Gets the stream at the specified index.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   index       Stream index.
 * @return              Stream object at the specified index.
 */
kInlineFx(kCompressedPhaseStream) kCompressedPhase_StreamAt(kCompressedPhase data, kSize index)
{
    kObj(kCompressedPhase, data);

    return kArrayList_AsT(obj->streams, index, kCompressedPhaseStream); 
}

/** 
 * Gets the total number of compressed items in the stream.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @return              Total count of compressed items in the stream.
 */
kInlineFx(kSize) kCompressedPhase_ItemCount(kCompressedPhase data, kCompressedPhaseStream stream)
{
    kObjN(kCompressedPhaseStream, streamObj, stream);

    return streamObj->itemCount;
}

/** 
 * Reports the number of packets in the underlying stream.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @return              Total count of packets comprising the stream.
 */
kInlineFx(kSize) kCompressedPhase_PacketCount(kCompressedPhase data, kCompressedPhaseStream stream)
{
    kObj(kCompressedPhaseStream, stream); 

    return kArrayList_Count(obj->packetList); 
}

/** 
 * Gets the size of the specified stream packet, in bytes.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @param   index       Packet index.
 * @return              Packet size (bytes).
 */
kInlineFx(kSize) kCompressedPhase_PacketSize(kCompressedPhase data, kCompressedPhaseStream stream, kSize index)
{
    kObjN(kCompressedPhaseStream, streamObj, stream);
    kCompressedPhasePacket* packet = kArrayList_AtT(streamObj->packetList, index, kCompressedPhasePacket); 

    return packet->size;
}

/** 
 * Gets a pointer to the specified stream packet.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @param   index       Packet index.
 * @return              Packet data pointer.
 */
kInlineFx(kPointer) kCompressedPhase_PacketData(kCompressedPhase data, kCompressedPhaseStream stream, kSize index)
{
    kObjN(kCompressedPhaseStream, streamObj, stream);
    kCompressedPhasePacket* packet = kArrayList_AtT(streamObj->packetList, index, kCompressedPhasePacket); 

    return packet->data;
}

/** 
 * Gets the compressed size of the specified stream, in bytes.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @return              Stream size (bytes).
 */
kInlineFx(kSize) kCompressedPhase_StreamSize(kCompressedPhase data, kCompressedPhaseStream stream)
{
    kObjN(kCompressedPhaseStream, streamObj, stream);
   
    return streamObj->streamSize;
}

/** 
 * Gets the stream identifier associated with the stream.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @return              Stream identifier.
 */
kInlineFx(kCompressedPhaseStreamId) kCompressedPhase_Id(kCompressedPhase data, kCompressedPhaseStream stream)
{
    kObjN(kCompressedPhaseStream, streamObj, stream);

    return streamObj->id;
}

/** 
 * Gets the encoding style associated with the stream.
 * 
 * @public              @memberof kCompressedPhase
 * @param   data        Compressed phase object.
 * @param   stream      Stream object.
 * @return              Encoding style.
 */
kInlineFx(kCompressedPhaseEncoding) kCompressedPhase_Encoding(kCompressedPhase data, kCompressedPhaseStream stream)
{
    kObjN(kCompressedPhaseStream, streamObj, stream);

    return streamObj->encoding;
}

/**
* Gets bit depths associated with intensity streams.
*
* @public                 @memberof kCompressedPhase
* @param   data           Compressed phase object.
* @param   deltaBitDepth  Receives delta depth.
* @param   fail1BitDepth  Receives fail-first bit depth.
* @param   fail2BitDepth  Receives fail-second bit depth.
* @param   nullBitDepth   Receives null bit depth as well as index bit depth.
* @return                 Operation status.
*/
kFsFx(kStatus) kCompressedPhase_IntensityBitDepths(kCompressedPhase data, k32u* deltaBitDepth, k32u* fail1BitDepth, k32u* fail2BitDepth, k32u* nullBitDepth);

/**
* Gets bit depths associated with phase streams.
*
* @public                 @memberof kCompressedPhase
* @param   data           Compressed phase object.
* @param   deltaBitDepth  Receives delta depth.
* @param   fail1BitDepth  Receives fail-first bit depth.
* @param   fail2BitDepth  Receives fail-second bit depth.
* @param   nullBitDepth   Receives null bit depth as well as index bit depth.
* @return                 Operation status.
*/
kFsFx(kStatus) kCompressedPhase_PhaseBitDepths(kCompressedPhase data, k32u* deltaBitDepth, k32u* fail1BitDepth, k32u* fail2BitDepth, k32u* nullBitDepth);


/**
 * @class   kCompressedPhaseReader
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Represents a kCompressedPhase stream-reading context.  
 */
//typedef kObject kCompressedPhaseReader;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a compressed phase reader object.
 * 
 * @public              @memberof kCompressedPhaseReader
 * @param   reader      Receives constructed phase reader object.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhaseReader_Construct(kCompressedPhaseReader* reader, kAlloc allocator); 

/** 
 * Prepares the compressed phase reader object to begin reading a stream. 
 * 
 * This method must be called before stream parsing methods such as kCompressedPhaseReader_ReadDelta.
 *
 * @public              @memberof kCompressedPhaseReader
 * @param   reader      Compressed phase reader.
 * @param   data        Compressed phase dataset to be read.
 * @param   stream      Compressed phase stream to be read.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCompressedPhaseReader_Begin(kCompressedPhaseReader reader, kCompressedPhase data, kCompressedPhaseStream stream); 

/** 
 * Reads a delta item. 
 * 
 * kCompressedPhaseReader_Begin must be called before reading individual values. 
 *
 * @public              @memberof kCompressedPhaseReader
 * @param   reader      Compressed phase reader.
 * @param   delta       Receives delta value. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kCompressedPhaseReader_ReadDelta(kCompressedPhaseReader reader, k32s* delta)
{
    kObj(kCompressedPhaseReader, reader); 

    return xkCompressedPhaseStream_ReadDelta(obj->stream, delta);
}

/** 
 * Reads a "fail" item from a stream. 
 * 
 * This method can be used to read from either "fail 1st" or "fail 2nd" streams.
 * 
 * kCompressedPhaseReader_Begin must be called before reading individual values. 
 *
 * @public              @memberof kCompressedPhaseReader
 * @param   reader      Compressed phase reader.
 * @param   index       Receives index value.  
 * @param   value       Receives fail value. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kCompressedPhaseReader_ReadFail(kCompressedPhaseReader reader, k32s* index, k32s* value)
{
    kObj(kCompressedPhaseReader, reader); 

    return xkCompressedPhaseStream_ReadFail(obj->stream, index, value);
}

/** 
 * Reads a null item from a stream. 
 * 
 * kCompressedPhaseReader_Begin must be called before reading individual values. 
 *
 * @public              @memberof kCompressedPhaseReader
 * @param   reader      Compressed phase reader.
 * @param   index       Receives index value.  
 * @param   count       Receives count value. 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kCompressedPhaseReader_ReadNull(kCompressedPhaseReader reader, k32s* index, k32s* count)
{
    kObj(kCompressedPhaseReader, reader); 

    return xkCompressedPhaseStream_ReadNull(obj->stream, index, count);
}

/** 
 * Gets the total number of compressed items in the stream to which this reader is attached.
 * 
 * @public              @memberof kCompressedPhaseReader
 * @param   reader      Compressed phase reader.
 * @return              Total count of compressed items in the stream to which this reader is attached.
 */
kInlineFx(kSize) kCompressedPhaseReader_ItemCount(kCompressedPhaseReader reader)
{
    kObj(kCompressedPhaseReader, reader); 

    return xkCompressedPhaseStream_ItemCount(obj->stream);
}

/**
* Check for intensity data.
*
* @public              @memberof kCompressedPhase
* @param   data        Compressed phase object.
* @return              kTRUE if intensity compression present.
*/
kInlineFx(kBool) kCompressedPhaseReader_IsIntensity(kCompressedPhase data)
{
    return kSuccess(kCompressedPhase_FindStream(data, kCOMPRESSED_PHASE_STREAM_ID_INTENSITY_DELTA, kNULL));
}

#endif
