/** 
 * @file    kColor.h
 * @brief   Declares the kColor type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_COLOR_H
#define K_FIRESYNC_COLOR_H

#include <kFireSync/kFsDef.h>

/** @relates kColor @{ */
#define kCOLOR_ARGB_MAX               (0xFF) ///<  Maximum value for an alpha, red, green, or blue color component.
#define kCOLOR_HSL_MAX                (0xF0) ///<  Maximum value for a hue, saturation, or lightness value.
#define kCOLOR_ALPHA_OPAQUE           (0xFF) ///<  Alpha value of an opaque color.
#define kCOLOR_ALPHA_CLEAR            (0x00) ///<  Alpha value of a transparent color.
/** @} */

/**
 * @class   kColor
 * @extends kValue
 * @ingroup kFireSync-Data
 * @brief   Color as an integer with 8-bit alpha, red, green, and blue components.
 */
typedef k32u kColor;

/** 
 * Returns a color value with the specified components.
 *
 * The value of each component can range from 0 to kCOLOR_ARGB_MAX.
 *
 * @public              @memberof kColor
 * @param   alpha       Alpha component.  
 * @param   red         Red component.  
 * @param   green       Green component.  
 * @param   blue        Blue component.  
 * @return              Returns a color value with the specified components.
 */
kInlineFx(kColor) kColor_FromArgb(k32s alpha, k32s red, k32s green, k32s blue)
{
    return (kColor)(((alpha & 0xFF) << 24) | 
                    ((red   & 0xFF) << 16) | 
                    ((green & 0xFF) << 8)  | 
                     (blue  & 0xFF));
}

/** 
 * Returns a color value with the specified components.
 *
 * The value of each component can range from 0 to kCOLOR_ARGB_MAX. 
 *
 * @public              @memberof kColor
 * @param   red         Red component.  
 * @param   green       Green component.  
 * @param   blue        Blue component.  
 * @return              Returns a color value with the specified components.
 */
kFsFx(kColor) kColor_FromRgb(k32s red, k32s green, k32s blue);

/** 
 * Returns a color value based on the specified hue, saturation, and lightness.
 *
 * The value of each component can range from 0 to kCOLOR_HSL_MAX. 
 *
 * @public              @memberof kColor
 * @param   hue         Hue value.  
 * @param   saturation  Saturation value.  
 * @param   lightness   Lightness value.  
 * @return              Returns a color value based on the specified hue, saturation, and lightness.
 */
kFsFx(kColor) kColor_FromHsl(k32s hue, k32s saturation, k32s lightness);

/** 
 * Gets the alpha, red, green, and blue components from a color value.
 *
 * @public              @memberof kColor
 * @param   color       Color value.  
 * @param   alpha       Destination for the alpha component.  
 * @param   red         Destination for the red component.  
 * @param   green       Destination for the green component.  
 * @param   blue        Destination for the blue component.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kColor_ToArgb(kColor color, k32s* alpha, k32s* red, k32s* green, k32s* blue);

/** 
 * Gets the red, green, and blue components from a color value.
 *
 * @public              @memberof kColor
 * @param   color       Color value.  
 * @param   red         Destination for the red component.  
 * @param   green       Destination for the green component.  
 * @param   blue        Destination for the blue component.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kColor_ToRgb(kColor color, k32s* red, k32s* green, k32s* blue);

/** 
 * Computes hue, saturation, and lightness values from a given color value.
 *
 * @public              @memberof kColor
 * @param   color       Color value.  
 * @param   hue         Destination for the hue value.  
 * @param   saturation  Destination for the saturation value.  
 * @param   lightness   Destination for the lightness value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kColor_ToHsl(kColor color, k32s* hue, k32s* saturation, k32s* lightness);

/** 
 * Returns the alpha component of a color value. 
 *
 * @public              @memberof kColor
 * @param   color       Color value.  
 * @return              Returns the alpha component of a color value. 
 */
kInlineFx(k32s) kColor_A(kColor color)
{
     return (color >> 24) & 0xFF; 
}

/** 
 * Returns the red component of a color value. 
 *
 * @public              @memberof kColor
 * @param   color       Color value.  
 * @return              Returns the red component of a color value. 
 */
kInlineFx(k32s) kColor_R(kColor color)
{
    return (color >> 16) & 0xFF; 
}

/** 
 * Returns the green component of a color value. 
 *
 * @public              @memberof kColor
 * @param   color       Color value.  
 * @return              Returns the green component of a color value. 
 */
kInlineFx(k32s) kColor_G(kColor color)
{
    return (color >> 8) & 0xFF; 
}

/** 
 * Returns the blue component of a color value. 
 *
 * @public              @memberof kColor
 * @param   color       Color value.  
 * @return              Returns the blue component of a color value. 
 */
kInlineFx(k32s) kColor_B(kColor color)
{
    return color & 0xFF; 
}

/** @relates kColor @{ */

#define kCOLOR_ALICE_BLUE               (0xFFF0F8FF) ///< Alice blue
#define kCOLOR_ANTIQUE_WHITE            (0xFFFAEBD7) ///< Antique white
#define kCOLOR_AQUA                     (0xFF00FFFF) ///< Aqua
#define kCOLOR_AQUAMARINE               (0xFF7FFFD4) ///< Aquamarine 
#define kCOLOR_AZURE                    (0xFFF0FFFF) ///< Azure
#define kCOLOR_BEIGE                    (0xFFF5F5DC) ///< Beige
#define kCOLOR_BISQUE                   (0xFFFFE4C4) ///< Bisque
#define kCOLOR_BLACK                    (0xFF000000) ///< Black
#define kCOLOR_BLANCHED_ALMOND          (0xFFFFEBCD) ///< Blanched almond
#define kCOLOR_BLUE                     (0xFF0000FF) ///< Blue
#define kCOLOR_BLUE_VIOLET              (0xFF8A2BE2) ///< Blue violet
#define kCOLOR_BROWN                    (0xFFA52A2A) ///< Brown
#define kCOLOR_BURLY_WOOD               (0xFFDEB887) ///< Burly wood
#define kCOLOR_CADET_BLUE               (0xFF5F9EA0) ///< Cadet blue
#define kCOLOR_CHARTREUSE               (0xFF7FFF00) ///< Chartreuse 
#define kCOLOR_CHOCOLATE                (0xFFD2691E) ///< Chocolate
#define kCOLOR_CORAL                    (0xFFFF7F50) ///< Coral
#define kCOLOR_CORNFLOWER_BLUE          (0xFF6495ED) ///< Cornflower blue
#define kCOLOR_CORNSILK                 (0xFFFFF8DC) ///< Cornsilk
#define kCOLOR_CRIMSON                  (0xFFDC143C) ///< Crimson 
#define kCOLOR_CYAN                     (0xFF00FFFF) ///< Cyan 
#define kCOLOR_DARK_BLUE                (0xFF00008B) ///< Dark blue
#define kCOLOR_DARK_CYAN                (0xFF008B8B) ///< Dark cyan
#define kCOLOR_DARK_GOLDENROD           (0xFFB8860B) ///< Dark goldenrod
#define kCOLOR_DARK_GRAY                (0xFFA9A9A9) ///< Dark gray
#define kCOLOR_DARK_GREEN               (0xFF006400) ///< Dark green
#define kCOLOR_DARK_KHAKI               (0xFFBDB76B) ///< Dark khaki
#define kCOLOR_DARK_MAGENTA             (0xFF8B008B) ///< Dark magenta
#define kCOLOR_DARK_OLIVE_GREEN         (0xFF556B2F) ///< Dark olive green 
#define kCOLOR_DARK_ORANGE              (0xFFFF8C00) ///< Dark orange 
#define kCOLOR_DARK_ORCHID              (0xFF9932CC) ///< Dark orchid
#define kCOLOR_DARK_RED                 (0xFF8B0000) ///< Dark red
#define kCOLOR_DARK_SALMON              (0xFFE9967A) ///< Dark salmon 
#define kCOLOR_DARK_SEA_GREEN           (0xFF8FBC8F) ///< Dark sea green
#define kCOLOR_DARK_SLATE_BLUE          (0xFF483D8B) ///< Dark slate blue 
#define kCOLOR_DARK_SLATE_GRAY          (0xFF2F4F4F) ///< Dark slate gray 
#define kCOLOR_DARK_TURQUOISE           (0xFF00CED1) ///< Dark turquoise
#define kCOLOR_DARK_VIOLET              (0xFF9400D3) ///< Dark violet
#define kCOLOR_DEEP_PINK                (0xFFFF1493) ///< Deep pink 
#define kCOLOR_DEEP_SKY_BLUE            (0xFF00BFFF) ///< Deep sky blue
#define kCOLOR_DIM_GRAY                 (0xFF696969) ///< Dim gray 
#define kCOLOR_DODGER_BLUE              (0xFF1E90FF) ///< Dodger blue  
#define kCOLOR_FIRE_BRICK               (0xFFB22222) ///< Fire brick
#define kCOLOR_FLORAL_WHITE             (0xFFFFFAF0) ///< Floral white
#define kCOLOR_FOREST_GREEN             (0xFF228B22) ///< Forest green
#define kCOLOR_FUCHSIA                  (0xFFFF00FF) ///< Fuchsia 
#define kCOLOR_GAINSBORO                (0xFFDCDCDC) ///< Gainsboro 
#define kCOLOR_GHOST_WHITE              (0xFFF8F8FF) ///< Ghost white 
#define kCOLOR_GOLD                     (0xFFFFD700) ///< Gold
#define kCOLOR_GOLDENROD                (0xFFDAA520) ///< Goldenrod
#define kCOLOR_GRAY                     (0xFF808080) ///< Gray
#define kCOLOR_GREEN                    (0xFF008000) ///< Green
#define kCOLOR_GREEN_YELLOW             (0xFFADFF2F) ///< Green yellow
#define kCOLOR_HONEYDEW                 (0xFFF0FFF0) ///< Honeydew
#define kCOLOR_HOT_PINK                 (0xFFFF69B4) ///< Hot pink
#define kCOLOR_INDIAN_RED               (0xFFCD5C5C) ///< Indian red
#define kCOLOR_INDIGO                   (0xFF4B0082) ///< Indigo
#define kCOLOR_IVORY                    (0xFFFFFFF0) ///< Ivory
#define kCOLOR_KHAKI                    (0xFFF0E68C) ///< Khaki
#define kCOLOR_LAVENDER                 (0xFFE6E6FA) ///< Lavender
#define kCOLOR_LAVENDER_BLUSH           (0xFFFFF0F5) ///< Lavender blush
#define kCOLOR_LAWN_GREEN               (0xFF7CFC00) ///< Lawn green 
#define kCOLOR_LEMON_CHIFFON            (0xFFFFFACD) ///< Lemon chiffon
#define kCOLOR_LIGHT_BLUE               (0xFFADD8E6) ///< Light blue 
#define kCOLOR_LIGHT_CORAL              (0xFFF08080) ///< Light coral
#define kCOLOR_LIGHT_CYAN               (0xFFE0FFFF) ///< Light cyan 
#define kCOLOR_LIGHT_GOLDENROD_YELLOW   (0xFFFAFAD2) ///< Light goldenrod yellow
#define kCOLOR_LIGHT_GRAY               (0xFFD3D3D3) ///< Light gray
#define kCOLOR_LIGHT_GREEN              (0xFF90EE90) ///< Light green 
#define kCOLOR_LIGHT_PINK               (0xFFFFB6C1) ///< Light pink
#define kCOLOR_LIGHT_SALMON             (0xFFFFA07A) ///< Light salmon 
#define kCOLOR_LIGHT_SEA_GREEN          (0xFF20B2AA) ///< Light sea green 
#define kCOLOR_LIGHT_SKY_BLUE           (0xFF87CEFA) ///< Light sky blue
#define kCOLOR_LIGHT_SLATE_GRAY         (0xFF778899) ///< Light slate gray
#define kCOLOR_LIGHT_STEEL_BLUE         (0xFFB0C4DE) ///< Light steel blue
#define kCOLOR_LIGHT_YELLOW             (0xFFFFFFE0) ///< Light yellow 
#define kCOLOR_LIME                     (0xFF00FF00) ///< Lime
#define kCOLOR_LIME_GREEN               (0xFF32CD32) ///< Lime green 
#define kCOLOR_LINEN                    (0xFFFAF0E6) ///< Linen
#define kCOLOR_MAGENTA                  (0xFFFF00FF) ///< Magenta
#define kCOLOR_MAROON                   (0xFF800000) ///< Maroon
#define kCOLOR_MEDIUM_AQUAMARINE        (0xFF66CDAA) ///< Medium aquamarine
#define kCOLOR_MEDIUM_BLUE              (0xFF0000CD) ///< Medium blue
#define kCOLOR_MEDIUM_ORCHID            (0xFFBA55D3) ///< Medium orchid 
#define kCOLOR_MEDIUM_PURPLE            (0xFF9370DB) ///< Medium purple
#define kCOLOR_MEDIUM_SEA_GREEN         (0xFF3CB371) ///< Medium sea green
#define kCOLOR_MEDIUM_SLATE_BLUE        (0xFF7B68EE) ///< Medium slate blue
#define kCOLOR_MEDIUM_SPRING_GREEN      (0xFF00FA9A) ///< Medium spring green
#define kCOLOR_MEDIUM_TURQUOISE         (0xFF48D1CC) ///< Medium turquoise 
#define kCOLOR_MEDIUM_VIOLET_RED        (0xFFC71585) ///< Medium violet red
#define kCOLOR_MIDNIGHT_BLUE            (0xFF191970) ///< Midnight blue
#define kCOLOR_MINT_CREAM               (0xFFF5FFFA) ///< Mint cream
#define kCOLOR_MISTY_ROSE               (0xFFFFE4E1) ///< Misty rose 
#define kCOLOR_MOCCASIN                 (0xFFFFE4B5) ///< Moccasin 
#define kCOLOR_NAVAJO_WHITE             (0xFFFFDEAD) ///< Navajo white
#define kCOLOR_NAVY                     (0xFF000080) ///< Navy
#define kCOLOR_OLD_LACE                 (0xFFFDF5E6) ///< Old lace
#define kCOLOR_OLIVE                    (0xFF808000) ///< Olive
#define kCOLOR_OLIVE_DRAB               (0xFF6B8E23) ///< Olive drab
#define kCOLOR_ORANGE                   (0xFFFFA500) ///< Orange 
#define kCOLOR_ORANGE_RED               (0xFFFF4500) ///< Orange red  
#define kCOLOR_ORCHID                   (0xFFDA70D6) ///< Orchid
#define kCOLOR_PALE_GOLDENROD           (0xFFEEE8AA) ///< Pale goldenrod 
#define kCOLOR_PALE_GREEN               (0xFF98FB98) ///< Pale green 
#define kCOLOR_PALE_TURQUOISE           (0xFFAFEEEE) ///< Pale turquoise
#define kCOLOR_PALE_VIOLET_RED          (0xFFDB7093) ///< Pale violet red
#define kCOLOR_PAPAYA_WHIP              (0xFFFFEFD5) ///< Papaya whip
#define kCOLOR_PEACH_PUFF               (0xFFFFDAB9) ///< Peach puff 
#define kCOLOR_PERU                     (0xFFCD853F) ///< Peru
#define kCOLOR_PINK                     (0xFFFFC0CB) ///< Pink
#define kCOLOR_PLUM                     (0xFFDDA0DD) ///< Plum 
#define kCOLOR_POWDER_BLUE              (0xFFB0E0E6) ///< Powder blue
#define kCOLOR_PURPLE                   (0xFF800080) ///< Purple
#define kCOLOR_RED                      (0xFFFF0000) ///< Red
#define kCOLOR_ROSY_BROWN               (0xFFBC8F8F) ///< Rosy brown
#define kCOLOR_ROYAL_BLUE               (0xFF4169E1) ///< Royal blue
#define kCOLOR_SADDLE_BROWN             (0xFF8B4513) ///< Saddle brown
#define kCOLOR_SALMON                   (0xFFFA8072) ///< Salmon
#define kCOLOR_SANDY_BROWN              (0xFFF4A460) ///< Sandy brown
#define kCOLOR_SEA_GREEN                (0xFF2E8B57) ///< Sea green
#define kCOLOR_SEA_SHELL                (0xFFFFF5EE) ///< Sea shell
#define kCOLOR_SIENNA                   (0xFFA0522D) ///< Sienna
#define kCOLOR_SILVER                   (0xFFC0C0C0) ///< Silver
#define kCOLOR_SKY_BLUE                 (0xFF87CEEB) ///< Sky blue
#define kCOLOR_SLATE_BLUE               (0xFF6A5ACD) ///< Slate blue
#define kCOLOR_SLATE_GRAY               (0xFF708090) ///< Slate gray
#define kCOLOR_SNOW                     (0xFFFFFAFA) ///< Snow
#define kCOLOR_SPRING_GREEN             (0xFF00FF7F) ///< Spring green
#define kCOLOR_STEEL_BLUE               (0xFF4682B4) ///< Steel blue
#define kCOLOR_TAN                      (0xFFD2B48C) ///< Tan 
#define kCOLOR_TEAL                     (0xFF008080) ///< Teal
#define kCOLOR_THISTLE                  (0xFFD8BFD8) ///< Thistle
#define kCOLOR_TOMATO                   (0xFFFF6347) ///< Tomato
#define kCOLOR_TRANSPARENT              (0x00000000) ///< Transparent 
#define kCOLOR_TURQUOISE                (0xFF40E0D0) ///< Turquoise 
#define kCOLOR_VIOLET                   (0xFFEE82EE) ///< Violet
#define kCOLOR_WHEAT                    (0xFFF5DEB3) ///< Wheat
#define kCOLOR_WHITE                    (0xFFFFFFFF) ///< White
#define kCOLOR_WHITE_SMOKE              (0xFFF5F5F5) ///< White smoke
#define kCOLOR_YELLOW                   (0xFFFFFF00) ///< Yellow
#define kCOLOR_YELLOW_GREEN             (0xFF9ACD32) ///< Yellow green

/** @} */

#include <kFireSync/Data/kColor.x.h>

#endif
