/** 
 * @file    kBinCSumMsg.h
 * @brief   Declares the kBinCSumMsg type. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_BINCSUM_MSG_H
#define K_FIRESYNC_BINCSUM_MSG_H

#include <kFireSync/Data/kAlgMsg.h>

/**
 * @class   kBinCSumMsg
 * @extends kAlgMsg
 * @ingroup kFireSync-Data
 * @brief   Represents output from a binarize-csum algorithm. 
 * 
 * Binarize output consists of 1-bit pixels packed into k8u values. For the sake of convenient
 * data representation, this message format requires that the source image width is a multiple 
 * of 8 columns, so that each row can be represented with a whole number of bytes. 
 * 
 * If results are generated by the binarize-csum camera algorithm (v4), the binarize 
 * height will be one less than the source image height (results are not provided for the last row 
 * of the source image).
 * 
 * kBinCSumMsg supports allocating data memory internally, or attaching to external memory 
 * (as required by the FireSync camera driver). Separate constructors are provided for these cases. 
 * 
 * kBinCSumMsg supports the kObject_Clone and kObject_Size methods. kObject_Dispose has the 
 * same effect as kObject_Destroy.
 *
 * kBinCSumMsg supports the kdat6 serialization protocol.
 */
//typedef kAlgMsg kBinCSumMsg;            --forward-declared in kFsDef.x.h

/** 
 * Constructs a kBinCSumMsg object that allocates its own memory for data content.
 *
 * @public                  @memberof kBinCSumMsg
 * @param   msg             Destination for the constructed object handle.
 * @param   binWidth        Number of 8-bit elements in one row of a binarize array (source width /8).
 * @param   binHeight       Number of rows in a single binarize array (typicaly source image height minus one).
 * @param   csumType        Type of the column sum data.
 * @param   csumLength      Number of csum results in each frame (typically equal to source image width). 
 * @param   frameCount      Number of frames in this message. 
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status. 
 */
kFsFx(kStatus) kBinCSumMsg_Construct(kBinCSumMsg* msg, kSize binWidth, kSize binHeight, kType csumType, kSize csumLength, kSize frameCount, kAlloc allocator); 

/** 
 * Constructs a kBinCSumMsg object that will be attached to external data content.
 * 
 * After calling this method, the kBinCSumMsg_Attach method must be used to  
 * attach memory pointers for each frame.
 *
 * @public                  @memberof kBinCSumMsg
 * @param   msg             Destination for the constructed object handle.
 * @param   binWidth        Number of 8-bit elements in one row of a binarize array (source width /8).
 * @param   binHeight       Number of rows in a single binarize array (typicaly source image height minus one).
 * @param   csumType        Type of the column sum data.
 * @param   csumLength      Number of csum results in each frame (typically equal to source image width). 
 * @param   frameCount      Number of frames in this message. 
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status. 
 */
kFsFx(kStatus) kBinCSumMsg_ConstructAttach(kBinCSumMsg* msg, kSize binWidth, kSize binHeight, kType csumType, kSize csumLength, kSize frameCount, kAlloc allocator); 

/** 
 * Attaches memory pointers for the specified frame.
 *
 * @public                  @memberof kBinCSumMsg
 * @param   msg             Message object. 
 * @param   frameIndex      Index of frame within this message.
 * @param   stamp           Stamp pointer.
 * @param   bin             Pointer to binarize data. 
 * @param   csum            Pointer to csum data. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kBinCSumMsg_Attach(kBinCSumMsg msg, kSize frameIndex, kStamp* stamp, void* bin, void* csum);

/** 
 * Attaches memory pointers for the specified frame.
 *
 * A debug assertion will be raised if the size of the specified bin and csum array types is not equal to the 
 * size of the message bin and csum data types, respectively.
 * 
 * @relates                     kBinCSumMsg
 * @param   kBinCSumMsg_msg     Message object. 
 * @param   kSize_frameIndex    Index of frame within this message.
 * @param   kStampPtr_stamp     Stamp pointer.
 * @param   BPtr_bin            Strongly-typed pointer to binarize data. 
 * @param   CPtr_csum           Strongly-typed pointer to csum data. 
 * @return                      Operation status. 
 */
#define kBinCSumMsg_AttachT(kBinCSumMsg_msg, kSize_frameIndex, kStampPtr_stamp, BPtr_bin, CPtr_csum) \
    xkBinCSumMsg_AttachT(kBinCSumMsg_msg, kSize_frameIndex, kStampPtr_stamp, BPtr_bin, CPtr_csum, sizeof(*BPtr_bin), sizeof(*CPtr_csum))

/** 
 * Reports the number of frames in the message. 
 *
 * @public          @memberof kBinCSumMsg
 * @param   msg     Message object.  
 * @return          Frame count.
 */
kFsFx(kSize) kBinCSumMsg_FrameCount(kBinCSumMsg msg);

/** 
 * Gets a pointer to the stamp at the specified frame index. 
 *
 * @public              @memberof kBinCSumMsg
 * @param   msg         Message object.  
 * @param   frameIndex  Index of frame within this message.
 * @return              Stamp pointer.
 */
kFsFx(kStamp*) kBinCSumMsg_StampAt(kBinCSumMsg msg, kSize frameIndex);

/** 
 * Reports the height (number of rows) of the binarize data in each frame. 
 * 
 * If these results were generated by the binarize-csum camera algorithm (v4), the binarize 
 * height is one less than the source image height. 
 *
 * @public          @memberof kBinCSumMsg
 * @param   msg     Message object.  
 * @return          Height of binarize array for each frame.  
 */
kFsFx(kSize) kBinCSumMsg_BinHeight(kBinCSumMsg msg);

/** 
 * Reports the number of 8-bit elements in one row of binarize data. 
 * 
 * The binarize width is <em>not</em> the number of columns in the source image. It is the 
 * number of 8-bit elements used to represent one row of packed, 1-bit results. In other words, 
 * it is the source image width divided by 8. 
 * 
 * @public          @memberof kBinCSumMsg
 * @param   msg     Message object.  
 * @return          Number of 8-bit pixels in one row of binarize data. 
 */
kFsFx(kSize) kBinCSumMsg_BinWidth(kBinCSumMsg msg);

/** 
 * Reports the type of the binarize data. 
 * 
 * @public          @memberof kBinCSumMsg
 * @param   msg     Message object.  
 * @return          Binarize data type.
 */
kFsFx(kType) kBinCSumMsg_BinType(kBinCSumMsg msg);

/** 
 * Gets a pointer to the binarize data at the specified frame and row index.
 *
 * @public              @memberof kBinCSumMsg
 * @param   msg         Message object.  
 * @param   frameIndex  Index of frame within this message.
 * @param   rowIndex    Index of binarize row within this frame.
 * @return              Binarize row data.
 */
kFsFx(void*) kBinCSumMsg_BinAt(kBinCSumMsg msg, kSize frameIndex, kSize rowIndex);

/** 
 * Gets a strongly-typed pointer to the binarize data at the specified frame and row index.
 *
 * A debug assertion will be raised if the size of the specified binarize data type is not equal to the 
 * size of the message binarize data type.
 * 
 * @relates                     kBinCSumMsg
 * @param   kBinCSumMsg_msg     Message object.  
 * @param   kSize_frameIndex    Index of frame within this message.
 * @param   kSize_rowIndex      Index of binarize row within this frame.
 * @param   T                   Binarize data type identifier (e.g., k8u).
 * @return                      Strongly-typed pointer to binarize data.
 */
#define kBinCSumMsg_BinAtT(kBinCSumMsg_msg, kSize_frameIndex, kSize_rowIndex, T) \
    kCast(T*, xkBinCSumMsg_BinAtT(kBinCSumMsg_msg, kSize_frameIndex, kSize_rowIndex, sizeof(T)))

/** 
 * Reports the number of csum results in each frame. 
 *
 * @public          @memberof kBinCSumMsg
 * @param   msg     Message object.  
 * @return          Length of CSum array for each frame. 
 */
kFsFx(kSize) kBinCSumMsg_CSumLength(kBinCSumMsg msg);

/** 
 * Reports the type of the column sum data. 
 * 
 * @public          @memberof kBinCSumMsg
 * @param   msg     Message object.  
 * @return          Column sum data type.
 */
kFsFx(kType) kBinCSumMsg_CSumType(kBinCSumMsg msg);

/** 
 * Gets a pointer to the csum data at the specified frame index.
 * 
 * @public              @memberof kBinCSumMsg
 * @param   msg         Message object.  
 * @param   frameIndex  Index of frame within this message.
 * @return              Csum data.
 */
kFsFx(void*) kBinCSumMsg_CSumAt(kBinCSumMsg msg, kSize frameIndex);

/** 
 * Gets a strongly-typed pointer to the csum data at the specified frame index.
 * 
 * A debug assertion will be raised if the size of the specified csum data type is not equal to the 
 * size of the message csum data type.
 * 
 * @relates                     kBinCSumMsg
 * @param   kBinCSumMsg_msg     Message object.  
 * @param   kSize_frameIndex    Index of frame within this message.
 * @param   T                   CSum data type identifier (e.g., kCSum4).
 * @return                      Strongly-typed pointer to csum data.
 */
#define kBinCSumMsg_CSumAtT(kBinCSumMsg_msg, kSize_frameIndex, T) \
    kCast(T*, xkBinCSumMsg_CSumAtT(kBinCSumMsg_msg, kSize_frameIndex, sizeof(T)))

#include <kFireSync/Data/kBinCSumMsg.x.h>

#endif
