/** 
 * @file    kProjectorState.h
 * @brief   Declares the kProjectorState type. 
 *
 * @internal
 * Copyright (C) 2012-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PROJECTOR_STATE_H
#define K_FIRESYNC_PROJECTOR_STATE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kProjectorState
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents a projector state configuration.
 */
//typedef kObject kProjectorState;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the state. 
 *
 * The id of a state is the same as the index of the state in the projector's state list.
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @return              Returns the unique id of this state. 
 */
kFsFx(kSize) kProjectorState_Id(kProjectorState state);

/** 
 * Sets the intensity for the projector state.  
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   intensity   Intensity value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_SetIntensity(kProjectorState state, k32u intensity, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the intensity for the projector state.  
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @return              Returns the intensity for the projector state.  
 */
kFsFx(k32u) kProjectorState_Intensity(kProjectorState state);

/** 
 * Reports constraint and validity information for the Intensity setting.
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_IntensityInfo(kProjectorState state, kInfo32u* info);

/** 
 * Sets the exposure for the projector state, in nanoseconds.
 *
 * Exposure values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   exposure    Exposure value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_SetExposureNs(kProjectorState state, k64u exposure, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the exposure for the projector state, in nanoseconds.  
 *
 * Exposure values are specified in FireSync nanoseconds (~ 1 us).
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @return              Returns the exposure for the projector state, in nanoseconds.  
 */
kFsFx(k64u) kProjectorState_ExposureNs(kProjectorState state);

/** 
 * Reports constraint and validity information for the ExposureNs setting. 
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_ExposureNsInfo(kProjectorState state, kInfo64u* info);

/** 
 * Sets the gap for the projector state, in nanoseconds.
 *
 * Gap values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   gap         Gap value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_SetGapNs(kProjectorState state, k64u gap, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the gap for the projector state, in nanoseconds.  
 *
 * Gap values are specified in FireSync nanoseconds (~ 1 us).
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @return              Returns the gap for the projector state, in nanoseconds.  
 */
kFsFx(k64u) kProjectorState_GapNs(kProjectorState state);

/** 
 * Reports constraint and validity information for the GapNs setting. 
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_GapNsInfo(kProjectorState state, kInfo64u* info);

/** 
 * Enables automatic gap calculation for the projector state.
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   enabled     Specifies whether automatic gap is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_EnableAutoGap(kProjectorState state, kBool enabled);

/** 
 * Reports whether automatic gap is enabled for the projector state.
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @return              Returns whether automatic gap is enabled for the projector state.
 */
kFsFx(kBool) kProjectorState_AutoGapEnabled(kProjectorState state);

/** 
 * Reports constraint and validity information for the AutoGapEnabled setting. 
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorState_AutoGapEnabledInfo(kProjectorState state, kInfoBool* info);

/** 
 * Reports the total time required to execute this projector state, in nanoseconds. 
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @return              State duration, FireSync in nanoseconds. 
 */
kFsFx(k64u) kProjectorState_DurationNs(kProjectorState state);

/** 
 * Reports the projector image time based on the state configuration, in nanoseconds. 
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @return              Image time, in FireSync nanoseconds. 
 */
kFsFx(k64u) kProjectorState_ImageTimeNs(kProjectorState state);

/** 
 * Based on a desired exposure, suggests a camera exposure (ns) that can be used to avoid projector artifacts. 
 *
 * The desired camera exposure is the projector illumination time that camera should capture. 
 * It can be a proportion of projector exposure. For example, if projector exposure is 50 milliseconds and
 * the camera should only capture 50% of projection time, set desiredExposure to 25 milliseconds. The 
 * the returned exposure value will be the suggested camera exposure to capture 25 milliseconds of 
 * projector illumination time.
 *
 * @public              @memberof kProjectorState
 * @param   state       State object.  
 * @param   exposure    Desired camera exposure, in FireSync nanoseconds.
 * @return              Suggested camera exposure, in FireSync nanoseconds. 
 */
kFsFx(k64u) kProjectorState_NominateCameraExposureNs(kProjectorState state, k64u exposure);

#include <kFireSync/Client/kProjectorState.x.h>

#endif
