/** 
 * @file    kOrientation.h
 * @brief   Declares the kOrientation type. 
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ORIENTATION_H
#define K_FIRESYNC_ORIENTATION_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kOrientation
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents configuration for a node's local orientation. 
 */
//typedef kObject kOrientation;        --forward-declared in kFsDef.x.h

/** 
 * Resets orientation device settings to defaults. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.  
 * @return                  Operation status. 
 */
kFsFx(kStatus) kOrientation_ClearAll(kOrientation orientation);

/** 
 * Enables or disables the orientation. 
 *
 * Disabled orientations are not checked during verification.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.  
 * @param   enable          Specifies whether to enable or disable the orientation.  
 * @return                  Operation status. 
 */
kFsFx(kStatus) kOrientation_Enable(kOrientation orientation, kBool enable);

/** 
 * Reports whether the orientation is currently enabled. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.  
 * @return                  Returns whether the orientation is currently enabled. 
 */
kFsFx(kBool) kOrientation_IsEnabled(kOrientation orientation);

/** 
 * Gets the model of the orientation device.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.  
 * @return                  Returns the model of the orientation device.
 */
kFsFx(kOrientationModel) kOrientation_Model(kOrientation orientation);

/** 
 * Sets the remapping axis.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   axis            Original axis.
 * @param   remappedAxis    Remapped axis.
 * @return                  Operation status.
 */
kFsFx(kStatus) kOrientation_SetAxisRemap(kOrientation orientation, kOrientationAxis axis, kOrientationAxis remappedAxis);

/** 
 * Gets the remapped axis.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   axis            Original axis.
 * @return                  Returns the remapped axis.
 */
kFsFx(kOrientationAxis) kOrientation_AxisRemap(kOrientation orientation, kOrientationAxis axis);

/** 
 * Reports constraint and validity information for the axis remapping setting. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   axis            Original axis.
 * @param   info            Receives the info structure.
 * @return                  Returns constraint and validity information for the axis remapping setting. 
 */
kFsFx(kStatus) kOrientation_AxisRemapInfo(kOrientation orientation, kOrientationAxis axis, kInfoBits* info);

/** 
 * Sets the sign of remapped axis.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   axis            Original axis.
 * @param   positive        Remapped axis sign.
 * @return                  Operation status.
 */
kFsFx(kStatus) kOrientation_SetAxisSign(kOrientation orientation, kOrientationAxis axis, kBool positive);

/** 
 * Gets the sign of remapped axis.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   axis            Axis.
 * @return                  Returns the sign of remapped axis.
 */
kFsFx(kBool) kOrientation_AxisSign(kOrientation orientation, kOrientationAxis axis);

/** 
 * Reports constraint and validity information for the remapped axis sign setting. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   axis            Axis.
 * @param   info            Receives the info structure.
 * @return                  Returns constraint and validity information for the remapped axis sign setting. 
 */
kFsFx(kStatus) kOrientation_AxisSignInfo(kOrientation orientation, kOrientationAxis axis, kInfoBool* info);

/** 
 * Sets the threshold of any motion detection.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   threshold       Threshold of any motion detection.
 * @return                  Operation status.
 */
kFsFx(kStatus) kOrientation_SetAnyMotionThreshold(kOrientation orientation, k64u threshold);

/** 
 * Gets the threshold of any motion detection.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @return                  Returns the threshold of any motion detection.
 */
kFsFx(k64u) kOrientation_AnyMotionThreshold(kOrientation orientation);

/** 
 * Reports constraint and validity information for the any motion threshold setting. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   info            Receives the info structure.
 * @return                  Returns constraint and validity information for the any motion threshold setting. 
 */
kFsFx(kStatus) kOrientation_AnyMotionThresholdInfo(kOrientation orientation, kInfo64u* info);

/** 
 * Sets the threshold of no motion detection.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   threshold       Threshold of no motion detection.
 * @return                  Operation status.
 */
kFsFx(kStatus) kOrientation_SetNoMotionThreshold(kOrientation orientation, k64u threshold);

/** 
 * Gets the threshold of no motion detection.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @return                  Returns the threshold of no motion detection.
 */
kFsFx(k64u) kOrientation_NoMotionThreshold(kOrientation orientation);

/** 
 * Reports constraint and validity information for the no motion threshold setting. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   info            Receives the info structure.
 * @return                  Returns constraint and validity information for the no motion threshold setting. 
 */
kFsFx(kStatus) kOrientation_NoMotionThresholdInfo(kOrientation orientation, kInfo64u* info);

/** 
 * Sets the delay of no motion detection.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   delay           Delay of no motion detection.
 * @return                  Operation status.
 */
kFsFx(kStatus) kOrientation_SetNoMotionDelay(kOrientation orientation, k64u delay);

/** 
 * Gets the delay of no motion detection.
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @return                  Returns the delay of no motion detection.
 */
kFsFx(k64u) kOrientation_NoMotionDelay(kOrientation orientation);

/** 
 * Reports constraint and validity information for the no motion delay setting. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @param   info            Receives the info structure.
 * @return                  Returns constraint and validity information for the no motion delay setting. 
 */
kFsFx(kStatus) kOrientation_NoMotionDelayInfo(kOrientation orientation, kInfo64u* info);

/** 
 * Start the calibration procedure. 
 *
 * @public                  @memberof kOrientation
 * @param   orientation     Orientation object.
 * @return                  Operation status.
 */
kFsFx(kStatus) kOrientation_StartCalibration(kOrientation orientation);

/**
* Retrieve statistics.
*
* @public              @memberof kOrientation
* @param   orientation Orientation object.  
* @param   stats       Receives the stats structure.
* @return              Operation status.
*/
kFsFx(kStatus) kOrientation_Stats(kOrientation orientation, kOrientationStats* stats);

#include <kFireSync/Client/kOrientation.x.h>

#endif
