/** 
 * @file    kNode.h
 * @brief   Declares the kNode type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NODE_H
#define K_FIRESYNC_NODE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kNode
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents a FireSync node (sensor). 
 * 
 * The kNode class supports sensor configuration and control. kNode objects are constructed and managed by 
 * the kSystem class, and can be accessed using functions such as kSystem_NodeCount and kSystem_NodeAt. 
 * 
 * Node objects can transition through a variety of states (kNodeState), and the current state can be determined 
 * using the thread-safe kNode_State function. Node state can change as a result of actions performed using the kNode API, 
 * or from asynchronous changes in the remote device or network. The kNode_State function can be called at any time to 
 * inquire about the current state. 
 * 
 * Immediately after discovery, a node will typically be in the Online state, indicating that the node is present in the 
 * network but not currently connected. In this state, only a handful of node functions can be used, including kNode_Id, 
 * kNode_Type, kNode_PlatformVersion, kNode_ProtocolVersion, kNode_DiscoveryIpConfig, kNode_SetDiscoveryIpConfig, and kNode_Connect.  
 * 
 * For most other actions, a control connection must first be established to the node using the kNode_Connect or kSystem_Connect 
 * functions. Upon successful connection, a node will most often be in the Ready state, indicating that it is 
 * ready to accept additional commands. However, if the remote device implements an incompatible version of FireSync 
 * network protocols, it may be in the Incompatible state. In this state, only a few services are available, notably 
 * kNode_Info, kNode_Stop, kNode_LoadFirmware (or kNode_LoadFirmwareFromStream), and kNode_Storage. 
 * 
 * In the event that a remote device experiences problems at boot time, it may load and run an alternate program image. 
 * This can be detected by examining the kNodeInfo.bootState field. The type of program image is not reflected in the node 
 * state; regardless of the source of the program, a node can exist any state (e.g., Ready, Incompatible). 
 * 
 * A large variety of methods and sub-objects are available for sensor configuration. In general, the following sequence 
 * is often used for sensor configuration: 
 * * kNode_Stop to ensure that the node is not running.
 * * kNode_Clear to reset settings to defaults.
 * * Module-specific configuration functions, via classes such as kCamera or kLight.
 * * kNode_VerifyAndApply to check for configuration erors and transmit configuration changes to the device. 
 * 
 * kNode methods are not thread-safe, except where noted. The same rule applies to all node sub-objects, such as cameras, 
 * event managers, or storage modules. However, separate nodes can safely be configured in parallel using separate threads. 
 * This approach can be advantageous in large systems, where sequential configuration is time consuming.
 * 
 * The following example illustrates a simple node configuration routine. 
@code {.c}
kStatus ConfigureNode(kSystem system, kNode node)
{
    //connect & stop
    kCheck(kNode_Connect(node));
    kCheck(kNode_Stop(node)); 

    //check some preconditions
    kCheckState(kNode_State(node) == kNODE_STATE_READY); 
    kCheckState(kNode_EventManagerCount(node) > 0); 
    kCheckState(kNode_CameraCount(node) > 0); 
    kCheckState(kNode_LightCount(node) > 0); 

    //start with default settings
    kCheck(kNode_Clear(node)); 

    //configure sub-modules
    {
        kEventManager eventManager = kNode_EventManagerAt(node, 0); 
        kCamera camera = kNode_CameraAt(node, 0); 
        kLight light = kNode_LightAt(node, 0); 
        kPipe pipe = kNode_Pipe(node); 

        //configure event manager 0 to use the minimum period; otherwise, accept default settings
        kCheck(kEventManager_Enable(eventManager, kTRUE));
        kCheck(kEventManager_EnableAutoPeriod(eventManager, kTRUE));

        //enable camera 0; otherwise, accept default settings
        kCheck(kCamera_Enable(camera, kTRUE));

        //enable light 0; otherwise, accept default settings
        kCheck(kLight_Enable(light, kTRUE));

        //route camera data to sensor network output
        kCheck(kPipe_AddRoute(pipe, kBlockId_Camera(0), 0, kBlockId_NetOut(), 0, kNULL));
    }

    //check for errors
    kCheck(kNode_Verify(node)); 
    kCheckState(kNode_ErrorCount(node) == 0); 

    //trasmit settings to node
    kCheck(kNode_Apply(node)); 
  
    return kOK; 
}

@endcode
 */ 
//typedef kObject kNode;        --forward-declared in kFsDef.x.h

/** 
 * Gets the current device identifier for this node, which typically corresponds to a manufacturing serial number. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Returns the device identifier for this node. 
 */
 kFsFx(k32u) kNode_Id(kNode node); 

 /** 
 * Gets the type of this node. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Returns the node type.
 */
 kFsFx(kNodeType) kNode_Type(kNode node); 

/** 
 * Sets the current IP configuration via a sensor discovery broadcast. 
 * 
 * This function uses discovery broadcast messages (UDP) to alter IP address configuration. This approach 
 * can be used to change the IP address of a node discovered on a different subnet. After receiving a Set 
 * Address message, the node will automatically reset. 
 * 
 * This function should only be used on nodes in the online state (kNODE_STATE_ONLINE). For connected sensors, 
 * the IP configuration functions in the kNet module should be used instead. kNet functions provide a more 
 * consistent configuration experience for connected sensors (i.e., the node doesn't automatically reset as a 
 * result of configuration changes). 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   ipConfig    IP configuration.
 * @return              Operation status. 
 * @see                 kNode_DiscoveryIpConfig, kNet_SetNextIpConfig
 */
kFsFx(kStatus) kNode_SetDiscoveryIpConfig(kNode node, const kIpConfig* ipConfig); 

/** 
 * Gets the current IP configuration, as received via sensor discovery. 
 * 
 * This function provides access to IP configuration information received via sensor discovery. This function 
 * can be used in conjunction with the kNode_SetDiscoveryIpConfig function to perform IP configuration on 
 * disconnected sensors.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   ipConfig    Receives IP configuration.
 * @return              Operation status.
 * @see                 kNode_SetDiscoveryIpConfig, kNet_IpConfig
 */
kFsFx(kStatus) kNode_DiscoveryIpConfig(kNode node, kIpConfig* ipConfig); 

/** 
 * Gets the local server node handle associated with the specified client node handle.
 * 
 * This method will return kNULL if the client node does not represent a direct connection 
 * to a local server node. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Server node (or kNULL if none). 
 */
kFsFx(kSvNode) kNode_Server(kNode node); 

/** 
 * Reports whether the node handle is associated with a local server node. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              kTRUE if the node represents a local server. 
 */
kInlineFx(kBool) kNode_IsServer(kNode node)
{
    return !kIsNull(kNode_Server(node));
}

/** 
 * Creates a control connection to the node.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Connect(kNode node); 

/** 
 * Disconnects the node's control connection.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status.
 */
kFsFx(kStatus) kNode_Disconnect(kNode node); 

/** 
 * Refreshes node state. 
 * 
 * Unresponsive or cancelled nodes will be disconnected. Nodes in most other states will reload 
 * configuration and state. 
 * 
 * This function can be used to resolve the <em>inconsistent</em> state (kNODE_STATE_INCONSISTENT), 
 * which can arise when modifications have been performed by another client. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Refresh(kNode node); 

/** 
 * Sends current settings to this node. 
 *
 * This function can be called when the node is in the Ready state or the Running state. If called while
 * running, the sensor will momentarily stop and then restart with the new settings. Note, this has no 
 * effect on data streaming; connections will remain open, and messages from before the apply operation 
 * may be received after the apply operation completes. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Apply(kNode node);

/** 
 * Checks for errors and applies settings if no errors are detected. 
 *
 * This is a convenience method that combines kNode_Verify and kNode_Apply. Refer to the documentation for those
 * methods for more information.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_VerifyAndApply(kNode node);

/** 
 * Commits current settings to non-volatile storage, to be used as defaults after boot. 
 *
 * This function applies current configuration settings to the node before committing. 
 * 
 * This function can be called only when the node is in the Ready state.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Commit(kNode node);

/** 
 * Commits current device settings to non-volatile storage, to be used as defaults after boot. 
 * <em>This function is for advanced users only. Use kNode_Commit if unsure.</em>
 *
 * This function applies current device configuration settings to the node before committing. Mode
 * settings such as event manager(s), camera(s), light(s), analog output(s), digital output(s),
 * the pipe and the encoder will not be committed.
 * 
 * This function can be called only when the node is in the Ready state.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Operation status. 
 * @see                 kNode_Commit. 
 */
kFsFx(kStatus) kNode_CommitDeviceConfig(kNode node);

/** 
 * Commits current mode settings to non-volatile storage, to be used as defaults after boot. 
 * <em>This function is for advanced users only. Use kNode_Commit if unsure.</em>
 *
 * This function applies current mode configuration settings to the node before committing. Only mode
 * settings will be committed, which includes the event manager(s), camera(s), light(s), analog output(s)
 * digital output(s), the pipe and the encoder.
 * 
 * This function can be called only when the node is in the Ready state.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Operation status. 
 * @see                 kNode_Commit. 
 */
kFsFx(kStatus) kNode_CommitModeConfig(kNode node);

/** 
 * Loads mode configuration from a file in node storage.
 * 
 * Most settings are stored in mode configuration files. However, some settings (e.g. node ID) are stored 
 * separately and not affected by this function.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   filePath    Mode configuration file path.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_LoadModeConfig(kNode node, const kChar* filePath);

/** 
 * Saves current mode configuration to the specified file. 
 *
 * This function applies configuration settings to the node before saving the settings to file. 
 * 
 * Most settings are stored in mode configuration files. However, some settings (e.g. node ID) are stored 
 * separately. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   filePath    Mode configuration file path.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SaveModeConfig(kNode node, const kChar* filePath);

/** 
 * Loads mode configuration from a local file.
 * 
 * Most settings are stored in mode configuration files. However, some settings (e.g. node ID) are stored 
 * separately and not affected by this function.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   filePath    Mode configuration file path.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_ImportModeConfig(kNode node, const kChar* filePath);

/** 
 * Saves current mode configuration to the specified local file. 
 *
 * This function applies configuration settings to the node before saving the settings to file. 
 * 
 * Most settings are stored in mode configuration files. However, some settings (e.g. node ID) are stored 
 * separately. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   filePath    Mode configuration file path.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_ExportModeConfig(kNode node, const kChar* filePath);

/** 
 * Restores mode settings to default values. 
 *
 * This functions resets <em>mode</em> settings, which include most of the settings exposed by a node. 
 * Mode settings are transient configuration values that are often modified in the course of using a node
 * (e.g. camera exposure).  
 * 
 * This function does not automatically apply or commit mode/device changes (though storage and variable 
 * changes are immediate/irreversible). Use kNode_Apply or kNode_Commit to complete changes. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Clear(kNode node);

/** 
 * Restores the specified aspects of node state to default values.
 *
 * This function accepts an 'options' bitset that precisely specifies the aspects of node state 
 * to reset. Use kNODE_CLEAR_OPTION_ALL to restore the device to a default state, including all
 * settings, storage and variables. 
 * 
 * This function does not automatically apply or commit mode/device changes (though storage and variable 
 * changes are immediate/irreversible). Use kNode_Apply or kNode_Commit to complete changes. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.  
 * @param   options         Bitset of specific settings that should be cleared.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_ClearEx(kNode node, kNodeClearOption options);

/** 
 * Resets device settings and storage contents to defaults. 
 *
 * This function is similar to kNode_ClearEx with kNODE_CLEAR_OPTION_ALL. However, mode and device 
 * changes are automatically committed.  
 *
 * Reset the node to complete this operation. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.  
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_Wipe(kNode node);

/** 
 * Checks the validity of all node settings. 
 *
 * Use kNode_ErrorCount/kNode_ErrorAt to access the list of errors. If kNode_ErrorCount reports
 * zero errors, then all validation checks passed. 
 * Use kNode_WarningCount/kNode_WarningAt to access the list of warnings. Warning may cause run-time
 * error under certain circumstance but not always.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Verify(kNode node);

/** 
 * Returns the number of errors that were detected when the node was last verified. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Number of errors detected when the node was last verified. 
 */
kFsFx(kSize) kNode_ErrorCount(kNode node);

/** 
 * Gets the verification warning string at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   index       Index of the error.   
 * @param   warning     Buffer to accept the null-terminated warning string.   
 * @param   capacity    Size, in characters, of the string buffer.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_WarningAt(kNode node, kSize index, kChar* warning, kSize capacity);

/** 
 * Returns the number of warnings that were detected when the node was last verified. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Number of warnings detected when the node was last verified. 
 */
kFsFx(kSize) kNode_WarningCount(kNode node);

/** 
 * Gets the verification error string at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   index       Index of the error.   
 * @param   error       Buffer to accept the null-terminated error string.   
 * @param   capacity    Size, in characters, of the string buffer.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_ErrorAt(kNode node, kSize index, kChar* error, kSize capacity);

/** 
 * Gets general information about this node. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives node information.
 * @return          Operation status. 
 */
 kFsFx(kStatus) kNode_Info(kNode node, kNodeInfo* info); 

/** 
 * Gets the model of the daughterboard.
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Returns the model of the daughterboard.
 */
kFsFx(kDaughterboardModel) kNode_DaughterboardModel(kNode node); 

/** 
 * Gets the platform version running on this node. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Platform version. 
 */
 kFsFx(kVersion) kNode_PlatformVersion(kNode node); 

 /** 
 * Gets the protocol version support by this node. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Protocol version. 
 */
 kFsFx(kVersion) kNode_ProtocolVersion(kNode node); 

 /** 
 * Gets the application version running on this node. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Application version. 
 */
 kFsFx(kVersion) kNode_AppVersion(kNode node); 

/** 
 * Gets the name of the application running on this node. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Application name.  
 */
 kFsFx(const kChar*) kNode_AppName(kNode node); 

/** 
 * Gets the operating system version running on this node. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          OS version. 
 */
 kFsFx(kVersion) kNode_OsVersion(kNode node);

/** 
 * Sets the device identifier for this node, which typically corresponds to a manufacturing serial number. 
 *
 * The node identifier is a device configuration setting - it is stored in the node when settings are committed 
 * (e.g. kNode_Commit), but is stored separately from the mode configuration file.
 * 
 * After changing the id and committing changes, the new id will take effect the next time that the node is reset.
 * After changing a node's id and resetting, use kSystem_HasChanges/kSystem_Refresh to re-detect the node.
 * 
 * On some architectures, the node id is used to configure both the node's MAC address and the node's default 
 * IP address. 
 * 
 * Node ids are limited to 24-bits. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   id          The identifier for this node.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SetNextId(kNode node, k32u id);

/** 
 * Gets the device identifier that will take effect after reset. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Returns the device identifier for this node. 
 */
kFsFx(k32u) kNode_NextId(kNode node); 

/** 
 * Reports constraint and validity information for the NextId setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_NextIdInfo(kNode node, kInfo32u* info); 

/** 
 * Sets the method that should be used to boot this device. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   mode        Boot mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SetBootMode(kNode node, kBootMode mode);

/** 
 * Gets the method that will be used to boot this device. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Boot mode.
 */
kFsFx(kBootMode) kNode_BootMode(kNode node); 

/** 
 * Reports constraint and validity information for the BootMode setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_BootModeInfo(kNode node, kInfoBits* info); 

/**
* Sets the power saver timeout for all lights.
*
* @public              @memberof kNode
* @param   node        Node object.  
* @param   timeout     Power saver timeout.
* @return              Operation status.
*/
kFsFx(kStatus) kNode_SetPowerSaverTimeout(kNode node, k64u timeout);

/**
* Gets the power saver timeout for all lights.
*
* @public              @memberof kNode
* @return              Power saver timeout.
*/
kFsFx(k64u) kNode_PowerSaverTimeout(kNode node);

/** 
 * Reports constraint and validity information for the power saver timeout setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_PowerSaverTimeoutInfo(kNode node, kInfo64u* info);

/**
* Sets the power saver threshold for all lights.
*
* If deactivated the light remains deactivated until encoder moves past this threshold.
*
* @public              @memberof kNode
* @param   node        Node object.  
* @param   threshold   Power saver threshold.
* @return              Operation status.
*/
kFsFx(kStatus) kNode_SetPowerSaverThreshold(kNode node, k64u threshold);

/**
* Gets the power saver threshold for all lights.
*
* @public              @memberof kNode
* @return              Power saver threshold.
*/
kFsFx(k64u) kNode_PowerSaverThreshold(kNode node);

/** 
 * Reports constraint and validity information for the power saver threshold setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_PowerSaverThresholdInfo(kNode node, kInfo64u* info);

/**
 * Enables light driver high power setting.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   enable      kTRUE to enable the feature.
 * @return              Operation status.
 */
kFsFx(kStatus) kNode_EnableLightDriverHighPower(kNode node, kBool enable);

/**
 * Reports whether light driver high power is enabled.
 *
 * @public              @memberof kNode
 * @return              kTRUE if light driver high power is enabled.
 */
kFsFx(kBool) kNode_LightDriverHighPowerEnabled(kNode node);

/** 
 * Reports constraint and validity information for the light driver high power setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_LightDriverHighPowerInfo(kNode node, kInfoBool* info);

/**
 * Sets the health logging period. 
 * 
 * By default, this setting is zero. This allows health logging to run at the nominal period for 
 * a given device. By changing the health logging period to another value, the default behavior
 * can be overridden. 
 * 
 * WARNING: Changing the logging period from the default value can significantly increase
 * wear on non-volatile storage components (typically flash). This should never be done in the 
 * field, but may sometimes be helpful for internal testing purposes. However, given the potential 
 * for storage degradation, it is recommended to consult with the FireSync Software team before 
 * making use of this setting. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   period      Health logging period, microseconds (or 0 for default).
 * @return              Operation status.
 */
kFsFx(kStatus) kNode_SetHealthLogPeriod(kNode node, k64u period);

/**
 * Reports the current health logging period.
 *
 * @public          @memberof kNode
 * @return          Health logging period, in microseconds (or 0 for default).    
 */
kFsFx(k64u) kNode_HealthLogPeriod(kNode node);

/** 
 * Reports constraint and validity information for the health logging period setting.
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_HealthLogPeriodInfo(kNode node, kInfo64u* info);

/** 
 * Enables the feature to wait for a rescue message at boot time. 
 * 
 * This feature is more appropriate for use during development than in the field, because it extends boot time 
 * by a certain timeout, optionally controlled via kNode_EnableRescueWait. The decision to use this in production 
 * can be made on a product-by-product basis. Apart from extending the boot time, there's little downside to 
 * enabling this feature.
 * 
 * Refer to the kSystem_StartRescue documentation for information on how to rescue nodes that support 
 * this feature.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   enable      kTRUE to enable the feature.
 * @return              Operation status. 
 * @see                 kSystem_StartRescue, kSystem_StopRescue
 */
kFsFx(kStatus) kNode_EnableRescueWait(kNode node, kBool enable);

/** 
 * Reports whether the feature to wait for a rescue message at boot time is enabled.
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          kTRUE if rescue wait is enabled. 
 */
kFsFx(kBool) kNode_RescueWaitEnabled(kNode node); 

/** 
 * Reports constraint and validity information for the RescueWait setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_RescueWaitEnabledInfo(kNode node, kInfoBool* info); 

/**
 * Sets the RescueWait timeout (us). 
 * 
 * WARNING: larger timeouts will result in increased boot times.
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   waitTimeout Health logging period, microseconds (or 0 for default).
 * @return              Operation status.
 */
kFsFx(kStatus) kNode_SetRescueWaitTimeout(kNode node, k64u waitTimeout);

/**
 * Reports the current value for the RescueWait timeout setting.
 *
 * @public          @memberof kNode
 * @return          Health logging period, in microseconds (or 0 for default).    
 */
kFsFx(k64u) kNode_RescueWaitTimeout(kNode node);

/** 
 * Reports constraint and validity information for the RescueWait timeout setting.
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_RescueWaitTimeoutInfo(kNode node, kInfo64u* info);

/** 
 * Enables automatic execution of firmware at boot time.
 * 
 * Normally, firmware is executed automatically at boot time.  However, to debug some controller types (e.g., M1), 
 * firmware execution must be deferred so that a debugger can be connected. 
 * 
 * This feature should only be disabled during product development. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   enable      kTRUE to enable the feature.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_EnableAutoExecute(kNode node, kBool enable);

/** 
 * Reports whether firmware will execute automatically at boot time.
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          kTRUE if automatic execution is enabled. 
 */
kFsFx(kBool) kNode_AutoExecuteEnabled(kNode node); 

/** 
 * Reports constraint and validity information for the AutoExcecute setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_AutoExecuteEnabledInfo(kNode node, kInfoBool* info); 

/** 
 * Specifies whether the watchdog can be enabled/disabled.
 * 
 * This function is provided to for debugging. Enabling the watchdog may hinder 
 * debugging (i.e. it may cause spontaneous reboots). This function can be used 
 * to forcefully disable the watchdog regardless of application settings and 
 * behaviors.
 * 
 * Use kNode_Commit to complete changes. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   allow       kFALSE to disable the watchdog, kTRUE for default behavior.
 * @return              Operation status.
 * @see                 kNode_Commit. 
 */
kFsFx(kStatus) kNode_AllowWatchdog(kNode node, kBool allow);

/**
 * Reports whether watchdog feature is allowed to be enabled/disabled.
 *
 * @public          @memberof kNode
 * @param   node    Node object.
 * @return          kTRUE if watchdog is allowed to be enabled/disabled.
 */
kFsFx(kBool) kNode_WatchdogAllowed(kNode node);

/**
 * Reports constraint and validity information for the AllowWatchdog setting.
 *
 * @public          @memberof kNode
 * @param   node    Node object.
 * @param   info    Receives the info structure.
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_WatchdogAllowedInfo(kNode node, kInfoBool* info);

/** 
 * Enables Secure Shell Daemon (SSHD).
 *
 * SSHD is useful for debugging, but is best disabled in production environments (default setting).
 * 
 * Use kNode_Commit to complete changes. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   enable      kTRUE to enable the feature.
 * @return              Operation status.
 * @see                 kNode_Commit. 
 */
kFsFx(kStatus) kNode_EnableSshd(kNode node, kBool enable);

/**
 * Reports whether firmware will enable SSHD.
 *
 * @public          @memberof kNode
 * @param   node    Node object.
 * @return          kTRUE if SSHD is enabled.
 */
kFsFx(kBool) kNode_SshdEnabled(kNode node);

/**
 * Reports constraint and validity information for the EnableSshd setting.
 *
 * @public          @memberof kNode
 * @param   node    Node object.
 * @param   info    Receives the info structure.
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_SshdEnabledInfo(kNode node, kInfoBool* info);

/** 
 * Enables crash log facility. 

 * This function is provided to for debugging. Crash logging may hinder debugging and
 * may cause spontaneous reboots. This function can be used to disable the crash 
 * logging service.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   enable      kTRUE to enable the feature.
 * @return              Operation status.
 * @see                 kNode_Commit. 
 */
kFsFx(kStatus) kNode_EnableCrashLog(kNode node, kBool enable);

/**
 * Reports whether firmware will enable crash logging facility.
 *
 * @public          @memberof kNode
 * @param   node    Node object.
 * @return          kTRUE if crash log is enabled.
 */
kFsFx(kBool) kNode_CrashLogEnabled(kNode node);

/**
 * Reports constraint and validity information for the EnableCrashLog setting.
 *
 * @public          @memberof kNode
 * @param   node    Node object.
 * @param   info    Receives the info structure.
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_CrashLogEnabledInfo(kNode node, kInfoBool* info);

/** 
 * Sets the IP address of the FTP server that will provide firmware during a network-based boot. 
 * 
 * This setting only applies when the boot mode is set to kBOOT_MODE_NETWORK. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   address     Server address. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SetNetBootServer(kNode node, kIpAddress address);

/** 
 * Reports the IP address of the FTP server that will provide firmware during a network-based boot. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Server address. 
 */
kFsFx(kIpAddress) kNode_NetBootServer(kNode node);

/** 
 * Sets the FTP resource path used to fetch firmware during a network-based boot. 
 * 
 * This setting only applies when the boot mode is set to kBOOT_MODE_NETWORK. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   path        Path used when fetching firmware from FTP server. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SetNetBootPath(kNode node, const kChar* path);

/** 
 * Reports the FTP resource path used to fetch firmware during a network-based boot. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Firmware path. 
 */
kFsFx(const kChar*) kNode_NetBootPath(kNode node);

/** 
 * Enables or disables the node. 
 *
 * Disabled nodes do not respond to Start/Stop commands. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   enable      Specifies whether to enable or disable the node.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Enable(kNode node, kBool enable); 

/** 
 * Reports whether the node is currently enabled. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @return              Returns whether the node is currently enabled. 
 */
kFsFx(kBool) kNode_IsEnabled(kNode node); 

/** 
 * Specifies the primary source for temperature information used by this node. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   probeId     Temperature probe id. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SetMainTempProbe(kNode node, kTempProbeId probeId);

/** 
 * Gets the primary source for temperature information used by this node.
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          Temperature probe id. 
 */
kFsFx(kTempProbeId) kNode_MainTempProbe(kNode node);

/** 
 * Reports constraint and validity information for the MainTempProbe setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_MainTempProbeInfo(kNode node, kInfoBits* info);

/** 
 * Enables automatic recovery from hardware/software errors (e.g. imager desync). 
 * 
 * This option is enabled by default, but can disabled for trouble-shooting purposes. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.  
 * @param   enable      kTRUE to enable automatic error recovery; kFALSE otherwise. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_EnableErrorRecovery(kNode node, kBool enable);

/** 
 * Reports whether automatic error recovery is enabled. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @return          kTRUE if automatic error recovery is enabled; kFALSE otherwise. 
 */
kFsFx(kBool) kNode_ErrorRecoveryEnabled(kNode node);

/** 
 * Reports constraint and validity information for the ErrorRecovery setting. 
 *
 * @public          @memberof kNode
 * @param   node    Node object.  
 * @param   info    Receives the info structure.  
 * @return          Operation status.
 */
kFsFx(kStatus) kNode_ErrorRecoveryEnabledInfo(kNode node, kInfoBool* info); 

/** 
 * Clears a firmware program from non-volatile storage.
 * 
 * Firmware programs required for successful boot cannot be cleared using this method (i.e., bootloader, rescue program). 
 * 
 * This method was introduced in FireSync 6.1.35.  If this method is used with older sensors, it will return 
 * kERROR_COMMAND. If the file does not exist prior to calling this method, the method will return kERROR_NOT_FOUND.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   type        Type of firmware to clear.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_ClearFirmware(kNode node, kNodeFirmwareType type); 

/** 
 * Determines the types of firmware that can be loaded to the device.
 * 
 * The absoluteOptions field of the received info structure can be used to determine which types can be loaded to the device.
 * It sould be noted that these options are raised to the power of two.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   info        Receives firmware type info.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_FirmwareInfo(kNode node, kInfoBits* info);

/** 
 * Loads firmware from a local file to non-volatile storage.
 * 
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 *
 * When PL programs are changed, some existing node settings may be invalidated as a result. These violations
 * are not automatically addressed. Accordingly, it is recommended to clear node settings and reconfigure
 * from scratch after switching configurations.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   type        Type of firmware to write 
 * @param   path        Path to local firmware file. 
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_LoadFirmware(kNode node, kNodeFirmwareType type, const kChar* path, kCallbackFx progress, kPointer context); 

/** 
 * Loads firmware from a local stream to non-volatile storage.
 * 
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * When PL programs are changed, some existing node settings may be invalidated as a result. These violations
 * are not automatically addressed. Accordingly, it is recommended to clear node settings and reconfigure
 * from scratch after switching configurations.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   type        Type of firmware to write 
 * @param   stream      Source of firmware data. 
 * @param   size        Size of firmware data
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_LoadFirmwareFromStream(kNode node, kNodeFirmwareType type, kStream stream, kSize size, kCallbackFx progress, kPointer context); 

/**
 * Replaces currently running firmware without affecting non-volatile storage.
 *
 * This method can be used to dynamically replace a firmware component without affecting the boot image
 * and without the need to reset the device. This methodology is only supported on some devices and only
 * with some types of firmware. (E.g., at the time of this writing, it can only dynamically replace the
 * primary PL program on Zynq7k-based controllers).
 *
 * Upon completion, this method will automatically refresh the node. As such, it is possible that
 * high-level modules such as cameras, projectors, or outputs may be created or destroyed as a result
 * of calling this method. It is recommended that clients should discard any kNode child handles (e.g,
 * kCamera, kLight) before calling this method and acquire new handles after calling this method.
 * 
 * When PL programs are changed, some existing node settings may be invalidated as a result. These violations 
 * are not automatically addressed. Accordingly, it is recommended to clear node settings and reconfigure 
 * from scratch after switching configurations.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   type        Type of firmware to write
 * @param   path        Path to local firmware file (or kNULL to unload firmware without loading a new program).
 * @return              Operation status.
 */
kFsFx(kStatus) kNode_ReloadFirmware(kNode node, kNodeFirmwareType type, const kChar* path);

/**
 * Replaces currently running firmware without affecting non-volatile storage.
 *
 * This method can be used to dynamically replace a firmware component without affecting the boot image
 * and without the need to reset the device. This methodology is only supported on some devices and only
 * with some types of firmware. (E.g., at the time of this writing, it can only dynamically replace the
 * primary PL program on Zynq7k-based controllers).
 *
 * Upon completion, this method will automatically refresh the node. As such, it is possible that
 * high-level modules such as cameras, projectors, or outputs may be created or destroyed as a result
 * of calling this method. It is recommended that clients should discard any kNode child handles (e.g,
 * kCamera, kLight) before calling this method and acquire new handles after calling this method.
 *
 * When PL programs are changed, some existing node settings may be invalidated as a result. These violations
 * are not automatically addressed. Accordingly, it is recommended to clear node settings and reconfigure
 * from scratch after switching configurations.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   type        Type of firmware to write
 * @param   stream      Source of firmware data (or kNULL, to unload firmware without loading a new program).
 * @param   size        Size of firmware data (or 0, if stream is null).
 * @return              Operation status.
 */
kFsFx(kStatus) kNode_ReloadFirmwareFromStream(kNode node, kNodeFirmwareType type, kStream stream, kSize size);

/** 
 * Creates the necessary content in the local file system to support simulating this node.
 * 
 * This function will create a new child directory in the specified parent directory. The name of the child 
 * directory will match the id of the node. If a directory with that name already exists, it will be 
 * deleted. 
 * 
 * The contents of the new directory will contain the necessary storage and configuration files 
 * to support simulating this node as a virtual node.
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   path        Parent directory for virtual node directory. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SaveAsVirtual(kNode node, const kChar* path); 

/** 
 * Begins rebooting the node. 
 *
 * By default, the reset process will begin immediately and the node will be disconnected. If the defer option is 
 * specified, the reset operation will be enqueued and will take effect when the node is later disconnected. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   defer       Should restart be deferred until the node is later disconnected?
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Reset(kNode node, kBool defer = kFALSE); 

/** 
 * Begins restarting node application software. 
 *
 * This operation shuts down node application software and restarts it, but without resetting hardware or 
 * restarting the node operating system. 
 * 
 * This feature enables information from error-checking that normally occurs during shutdown (e.g., memory leaks) 
 * to be preserved and accessed after restart. It can also be used (with some risk) as a faster form of reset 
 * when debugging. 
 * 
 * This feature is not supported on every platform. If this feature is not supported, restarting will 
 * have the same effect as resetting.
 * 
 * By default, the restart process will begin immediately and the node will be disconnected. If the defer option is 
 * specified, the restart operation will be enqueued and will take effect when the node is later disconnected. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   defer       Should restart be deferred until the node is later disconnected?
 * @return              Operation status. 
 * @see                 kNode_Reset
 */
kFsFx(kStatus) kNode_Restart(kNode node, kBool defer = kFALSE);

/** 
 * Opens a health connection to the remote node. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_OpenHealth(kNode node); 

/** 
 * Closes the health connection to the remote node. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_CloseHealth(kNode node); 

/** 
 * Sets the network data output ports to which this client should connect. 
 * 
 * When the kNode_OpenData function (or kSystem_Start function) is called, 
 * this client will connect to one or more of the remote node's data output ports. By default, 
 * the client will connect to data output port 0; the kNode_SubscribeData function can be used 
 * to override the default behaviour. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   ports       Ports that should be connected. 
 * @param   portCount   Count of ports to be connected. 
 * @return              Operation status. 
 * @see                 kNode_OpenData, kNet_SetDataOutPortCount
 */
kFsFx(kStatus) kNode_SubscribeData(kNode node, const k16u* ports, kSize portCount); 

/** 
 * Opens a data connection to the remote node. 
 * 
 * By default, the client will connect to network data output port 0. The kNode_SubscribeData 
 * function can be used to override the default behaviour.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 * @see                 kNode_SubscribeData, kNet_SetDataOutPortCount
 */
kFsFx(kStatus) kNode_OpenData(kNode node); 

/**
* Ends data collection until the next time that this node is started/resumed.
*
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status.
 * @see                 kNode_OpenData, kNode_CloseData
*/
kFsFx(kStatus) kNode_EndDataSession(kNode node);

/** 
 * Closes the data connection to the remote node. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 * @see                 kNode_OpenData
 */
kFsFx(kStatus) kNode_CloseData(kNode node); 

/** 
 * Starts node acquisition and data processing activities. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 * @see                 kNode_Stop
 */
kFsFx(kStatus) kNode_Start(kNode node); 

/** 
 * Stops node acquisition and data processing activities. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 * @see                 kNode_Start
 */
kFsFx(kStatus) kNode_Stop(kNode node); 

/** 
 * (Deprecated) Function currently calls kNode_Stop.
 *
 * Temporarily pauses node acquisition and data processing activities. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 * @see                 kNode_Resume
 */
kFsFx(kStatus) kNode_Pause(kNode node); 

/** 
 * (Deprecated) Function currently calls kNode_Start.
 * 
 * Resumes node acquisition and data processing activities. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 * @see                 kNode_Pause
 */
kFsFx(kStatus) kNode_Resume(kNode node); 

/** 
 * Begins preparing node for acquisition and data processing activities. 
 * 
 * This command initiates an asynchronous operation on the node. Call the kNode_EndStart
 * method to wait for the end of the operation before using any other kNode methods. 
 * 
 * After BeginStart/EndStart, use the BeginEngage/EndEngage operations to start acquisition.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_BeginStart(kNode node); 

/** 
 * Completes an asynchronous start operation that was initiated using the kNode_BeginStart method. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_EndStart(kNode node); 

/** 
 * Begins stopping node acquisition and data processing activities. 
 * 
 * This command initiates an asynchronous operation on the node. Call the kNode_EndStop
 * method to wait for the end of the operation before using any other kNode methods. 
 * 
 * Note: if the 'synchronizeData' option is used, the expectation is that all nodes in the system
 * that have node-node data connections will be stopped (with 'synchronizeData' == kTRUE) before 
 * any nodes are restarted. If only a subset are stopped, node-node data connections may stall the 
 * next time that acquisition is restarted.
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   synchronizeData Ensure that data from current session is not accepted in next session.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_BeginStop(kNode node, kBool synchronizeData); 

/** 
 * Completes an asynchronous stop operation that was initiated using the kNode_BeginStop method. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_EndStop(kNode node); 

/** 
 * Initiates acquisition after BeginStart/EndStart.
 * 
 * This command initiates an asynchronous operation on the node. Call the kNode_EndEngage
 * method to wait for the end of the operation before using any other kNode methods. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   startTime       Node start time (FS us), or k64U_NULL for auto-synchronized start.
 * @param   startEncoder    Node start encoder (ticks), or k64S_NULL for auto-synchronized start. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_BeginEngage(kNode node, k64u startTime, k64s startEncoder); 

/** 
 * Completes an asynchronous engage operation that was initiated using the kNode_BeginEngage method. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_EndEngage(kNode node); 

/** 
 * Prepares node to begin receiving and processing replay data. 
 * 
 * Use kNode_Stop to end replay. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 * @see                 kNode_Stop
 */
kFsFx(kStatus) kNode_StartReplay(kNode node); 

/** 
 * Replays an input message and waits to receive output.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   input       Input message to be replayed. 
 * @param   output      List to receive output messages. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Replay(kNode node, kMsgInfo input, kArrayList output); 

/** 
 * Gets the list of custom variables available from this node. 
 * 
 * Use kArrayList_Purge to free the names returned by this function.
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   variableNames   Receives names of custom variables (kArrayList<kString>). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_ListVariables(kNode node, kArrayList variableNames); 

/** 
 * Gets information about a custom variable. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   name        Variable name. 
 * @param   info        Receives variable information. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_VariableInfo(kNode node, const kChar* name, kNodeVariableInfo* info); 

/** 
 * Gets the value of the specified remote variable. 
 *
 * The variable object returned by this function may be a shared object. Accordingly, if the variable 
 * object does not provide thread-safe access, then it should typically be treated as read-only. Use kObject_Dispose 
 * to free the variable object when no longer needed.
 * 
 * In order for a variable object to be accessible across the network, the variable class must 
 * implement kDat-6 serialization. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   name        Variable name. 
 * @param   variable    Receives variable object. 
 * @param   alloc       Optional allocator for remote variable, if applicable (can be null). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_Variable(kNode node, const kChar* name, kObject* variable, kAlloc alloc); 

/** 
 * Sets the value of a remote variable. 
 *
 * The variable object passed to this function becomes a shared object. Accordingly, if the variable 
 * object does not provide thread-safe access, then it should typically be treated as read-only thereafter. 
 * The caller retains ownership of the original variable reference and should use kObject_Dispose 
 * to free the variable object when it is no longer needed.
 * 
 * If a variable with the specified name already exists, the existing variable object will be disposed
 * and replaced with the new variable object. 
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   name        Variable name. 
 * @param   variable    Variable object (cannot be null).
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_SetVariable(kNode node, const kChar* name, kObject variable); 

/** 
 * Disposes a remote variable with the specified name. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   name        Variable name. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_DeleteVariable(kNode node, const kChar* name); 

/** 
 * Disposes all remote variables. 
 *
 * @public              @memberof kNode
 * @param   node        Node object.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_DeleteVariables(kNode node); 

/** 
 * Reads from the specified memory space. 
 *
 * This method is intended for low-level debug access by FireSync platform developers. 
 * No input validation is performed; the use of invalid memory addresses or sizes may crash 
 * the device. Available memory spaces may vary from node to node. Use of this method 
 * beyond internal development/debug scenarios is prohibited.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   space       Memory space in which address offset is interpreted. 
 * @param   offset      Address offset, in bytes, from the beginning of the memory space.
 * @param   length      Length of memory to read.
 * @param   data        Receives data read from node.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_ReadMemory(kNode node, kNodeMemorySpace space, k64u offset, kSize length, void* data);

/** 
 * Writes to the specified memory space. 
 *
 * This method is intended for low-level debug access by FireSync platform developers. 
 * No input validation is performed; the use of invalid memory addresses or sizes may crash 
 * the device. Available memory spaces may vary from node to node. Use of this method 
 * beyond internal development/debug scenarios is prohibited.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   space       Memory space in which address offset is interpreted. 
 * @param   offset      Address offset, in bytes, from the beginning of the memory space.
 * @param   length      Length of memory to write.
 * @param   data        Data to write.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode_WriteMemory(kNode node, kNodeMemorySpace space, k64u offset, kSize length, const void* data);

/** 
 * Gets a list of registers for a given module type and module index. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   moduleType      Module type.
 * @param   moduleIndex     Module index.
 * @param   registers       Receives register info (kArrayList<kRegisterInfo32u>). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_ReadRegisters(kNode node, kRegisterModule moduleType, kSize moduleIndex, kArrayList registers);

/** 
 * Reads a list of register overrides for a given module type. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   moduleType      Module type.
 * @param   registers       Receives register info (kArrayList<kRegisterOverride32u>). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_ReadRegisterOverrides(kNode node, kRegisterModule moduleType, kArrayList registers);

/** 
 * Writes a list of registers overrides for a given module type. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   moduleType      Module type.
 * @param   registers       Register info (kArrayList<kRegisterOverride32u>). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_WriteRegisterOverrides(kNode node, kRegisterModule moduleType, kArrayList registers);

/** 
 * Reads I2C register value. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   deviceId        Slave address.
 * @param   address         Location where data is to be read from.
 * @param   data            Data read from I2C.
 * @param   size            Number of bytes to read.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_I2cRead(kNode node, k32u deviceId, k32u address, kByte* data, kSize size);

/** 
 * Writes to I2C register. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   deviceId        Slave address.
 * @param   address         Location where data is to be written to.
 * @param   data            Data to write.
 * @param   size            Number of bytes to write. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_I2cWrite(kNode node, k32u deviceId, k32u address, const kByte* data, kSize size);

/** 
 * Creates a report containing a description of current node state. 
 * 
 * The generated report will include at least general node information, log data, crash logs (if applicable), 
 * health statistics, health logs, mode configuration, and device configuration. 
 * 
 * The report object created by this method can be serialized with kDat6Serializer, cloned, and/or destroyed. 
 * For future compatibility, it is recommended to avoid additional programmatic interactions with the report 
 * object.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   report      Receives node report object.
 * @param   allocator   Allocator for report (optional; can be kNULL).
 * @return              Operation status. 
 * @see                 kNode_SaveReport
 */
kFsFx(kStatus) kNode_Report(kNode node, kObject* report, kAlloc allocator);

/** 
 * Saves a report containing a description of current node state to the specified file.
 * 
 * This method is equivalent to calling kNode_Report followed by kSave6. 
 * 
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   filePath    Local path of the file to be saved.
 * @return              Operation status. 
 * @see                 kNode_Report
 */
kFsFx(kStatus) kNode_SaveReport(kNode node, const kChar* filePath);

/** 
 * Gets a list of log messages. 
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   logItems        Receives log items (kArrayList<kLogItem>). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_ReadLogHistory(kNode node, kArrayList logItems);

/** 
 * Retrieves a crash log summary from the node. 
 * 
 * When a software error occurs, some devices can collect critical information and save this to 
 * non-volatile storage for post-crash diagnostics. This function harvests the collected information 
 * and generates a summary in text form (as a kString object). Amongst other details, the crash log 
 * may contain a stack traces, CPU registers, disassembled code, as well as a global, monotonically 
 * increasing crash counter.
 * 
 * @public              @memberof kNode
 * @param   node        Node object.
 * @param   crashLog    Receives crash log as kString. 
 * @param   allocator   Optional allocator to receive crash log string, if applicable (can be null). 
 * @return              Operation status. 
 * @see                 kNode_ClearCrashLog.
 */
kFsFx(kStatus) kNode_ReadCrashLog(kNode node, kString* crashLog, kAlloc allocator);

/** 
 * Clears the crash log data stored on the device. 
 *
 * When a software error occurs, some devices can collect critical information and save this to 
 * non-volatile storage for post-crash diagnostics. This function clears all crash log data 
 * stored in the device and resets the crash counter.
 * 
 * @public              @memberof kNode
 * @param   node        Node object. 
 * @return              Operation status. 
 * @see                 kNode_ReadCrashLog.
 */
kFsFx(kStatus) kNode_ClearCrashLog(kNode node);

/** 
 * Reports the current state of the node.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Node state. 
 */
kFsFx(kNodeState) kNode_State(kNode node);

/** 
 * Gets the current synchronization time value from the node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Timestamp (fs us). 
 */
kFsFx(k64u) kNode_TimeCounter(kNode node);

/** 
 * Gets the current encoder value from the node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Encoder count (ticks). 
 */
kFsFx(k64s) kNode_EncoderCounter(kNode node);

/** 
 * Reports a selection of top-level run-time statistics for the node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   stats       Receives node run-time statistics. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kNode_Stats(kNode node, kNodeStats* stats);

/** 
 * Retrieves latest health statistics from the node. 
 * 
 * The retrieved health statistics are stored internally by this object. Statistics can then be 
 * accessed using kNode_StatCount and kNode_StatAt, or with kNode_FindStat/kNode_FindStatByName. 
 * 
 * kNode_RefreshStats should be avoided in performance-sensitive contexts. While sometimes convenient 
 * (e.g., integration testing), kNode_RefreshStats is <em>considerably</em> less efficient than standard 
 * health support methods (i.e., kSystem_SetHealthHandler, kSystem_ReceiveHealth). 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Operation status.  
 * @see                 kNode_StatCount, kNode_StatAt, kNode_FindStat, kNode_FindStatByName, kSystem_SetHealthHandler, kSystem_ReceiveHealth
 */
kFsFx(kStatus) kNode_RefreshStats(kNode node);

/** 
 * Gets the number of available health statistics.
 * 
 * The count reported by this method corresponds to the number of health statistics retrieved during 
 * the most recent call to kNode_RefreshStats.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Count of available statistics.
 * @see                 kNode_RefreshStats, kNode_StatAt
 */
kFsFx(kSize) kNode_StatCount(kNode node);

/** 
 * Gets the health statistic at the specified index.
 * 
 * The statistics reported by this method correspond to the health statistics retrieved during 
 * the most recent call to kNode_RefreshStats.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Health statistic index.
 * @return              Pointer to specified health statistic.
 * @see                 kNode_RefreshStats, kNode_StatCount
 */
kFsFx(const kHealthStat*) kNode_StatAt(kNode node, kSize index);

/** 
 * Gets the health statistic with the specified id and instance.
 * 
 * The statistics reported by this method correspond to the health statistics retrieved during 
 * the most recent call to kNode_RefreshStats.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   id          Health indicator ID.
 * @param   instance    Health indicator instance (or k32U_NULL for any).
 * @param   value       Receives health indicator value.
 * @return              Operation status (kERROR_NOT_FOUND if match not found).
 * @see                 kNode_RefreshStats, kNode_FindStatByName
 */
kFsFx(kStatus) kNode_FindStat(kNode node, kHealthId id, k32u instance, k64s* value);

/** 
 * Gets the health statistic with the specified name.
 * 
 * The statistics reported by this method correspond to the health statistics retrieved during 
 * the most recent call to kNode_RefreshStats.
 * 
 * Health indicator names can occasionally change over time. Accordingly, if the desired indicator 
 * has a publicly-defined indicator ID, kNode_FindStat should usually be preferred to kNode_FindStatByName. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   name        Health indicator name.
 * @param   value       Receives health indicator value.
 * @return              Operation status (kERROR_NOT_FOUND if match not found).
 * @see                 kNode_RefreshStats, kNode_FindStat
 */
kFsFx(kStatus) kNode_FindStatByName(kNode node, const kChar* name, k64s* value);

/** 
 * Reads and summarizes the node health log.  
 * 
 * If a node supports health logging, it will periodically record any persistent health indicators 
 * to non-volatile storage. The ReadHealthLog method can be used to read and summarize health log 
 * snapshots to generate a health statistics report. 
 * 
 * If the node does not support health logging, the received summary object will be empty.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   summary     Receives a summary of the node health log. 
 * @param   alloc       Memory allocator for health log summary. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kNode_ReadHealthLog(kNode node, kHealthSummary* summary, kAlloc alloc);

/** 
 * Clears the node health log.  
 *
 * Warning: this method will permanently erase all persistent health log history. Use with caution. 
 * 
 * If the node does not support health logging, this method will return successfully but will 
 * have no effect.
 *
 * Changes will take effect upon reboot/restart.
 * 
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Operation status.  
 */
kFsFx(kStatus) kNode_ClearHealthLog(kNode node);

/** 
 * Adjusts the date-time value that will be reported by the node. 
 * 
 * At present, nodes do not support persistent calendar date-time storage. However, calendar date-time 
 * awareness can be beneficial in some circumstances, e.g., for persistent health logging. This method 
 * enables time-aware clients to explicitly set the current node date-time. 
 * 
 * After setting the node date-time with this method, the node will adjust its internal date-time 
 * representation such that future date-time values reported by the node are relative to the new baseline 
 * established with this method. 
 * 
 * This method has no effect on node thread-scheduling (e.g., will not affect any active timeouts).  
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   dateTime    Current calendar date-time.
 * @return              Operation status.
 */
kFsFx(kStatus) kNode_SetDateTime(kNode node, kDateTime dateTime); 

/** 
 * Reports the number of event managers supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Event manager count. 
 */
kFsFx(kSize) kNode_EventManagerCount(kNode node); 

/** 
 * Gets the event manager at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Event manager index. 
 * @return              Event manager. 
 */
kFsFx(kEventManager) kNode_EventManagerAt(kNode node, kSize index); 

/** 
 * Reports the number of camera modules supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Camera module count. 
 */
kFsFx(kSize) kNode_CameraCount(kNode node); 

/** 
 * Gets the camera module at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Camera module index. 
 * @return              Camera module. 
 */
kFsFx(kCamera) kNode_CameraAt(kNode node, kSize index); 

/** 
 * Reports the number of light modules supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Light module count. 
 */
kFsFx(kSize) kNode_LightCount(kNode node); 

/** 
 * Gets the light module at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Light module index. 
 * @return              Light module. 
 */
kFsFx(kLight) kNode_LightAt(kNode node, kSize index); 

/** 
 * Reports the number of projector modules supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Projector module count. 
 */
kFsFx(kSize) kNode_ProjectorCount(kNode node); 

/** 
 * Gets the projector module at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Projector module index. 
 * @return              Projector module. 
 */
kFsFx(kProjector) kNode_ProjectorAt(kNode node, kSize index); 

/** 
 * Reports the number of analog outputs supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Analog output count. 
 */
kFsFx(kSize) kNode_AnalogOutCount(kNode node); 

/** 
 * Gets the analog output at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Analog output index. 
 * @return              Analog output. 
 */
kFsFx(kAnalogOut) kNode_AnalogOutAt(kNode node, kSize index); 

/** 
 * Reports the number of digital outputs supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Digital output count. 
 */
kFsFx(kSize) kNode_DigitalOutCount(kNode node); 

/** 
 * Gets the digital output at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Digital output index. 
 * @return              Digital output. 
 */
kFsFx(kDigitalOut) kNode_DigitalOutAt(kNode node, kSize index); 

/** 
 * Reports the number of serial outputs supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Serial output count. 
 */
kFsFx(kSize) kNode_SerialOutCount(kNode node); 

/** 
 * Gets the serial output at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Serial output index. 
 * @return              Serial output. 
 */
kFsFx(kSerialOut) kNode_SerialOutAt(kNode node, kSize index); 

/** 
 * Reports the number of I/O tests supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Serial output count. 
 */
kFsFx(kSize) kNode_IoTestCount(kNode node); 

/** 
 * Gets the I/O test at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       Serial output index. 
 * @return              Serial output. 
 */
kFsFx(kIoTest) kNode_IoTestAt(kNode node, kSize index); 

/** 
 * Reports the number of GPIO banks supported by this node. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              GPIO bank count. 
 */
kFsFx(kSize) kNode_GpioBankCount(kNode node);

/** 
 * Gets the GPIO bank at the specified index. 
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @param   index       GPIO bank index. 
 * @return              GPIO bank. 
 */
kFsFx(kGpioBank) kNode_GpioBankAt(kNode node, kSize index);

/**
* Gets the orientation configuration module, if present.
*
* @public              @memberof kNode
* @param   node        kNode object.
* @return              Orientation module, or kNULL if not present.
*/
kFsFx(kOrientation) kNode_Orientation(kNode node);

/** 
 * Gets the encoder configuration module, if present.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Encoder module, or kNULL if not present. 
 */
kFsFx(kEncoder) kNode_Encoder(kNode node); 

/**
* Gets the test jig configuration module, if present.
*
* @public              @memberof kNode
* @param   node        kNode object.
* @return              Test jig module, or kNULL if not present.
*/
kFsFx(kTestJig) kNode_TestJig(kNode node);

/** 
 * Gets the temperature control configuration module, if present.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              TempControl module, or kNULL if not present. 
 */
kFsFx(kTempControl) kNode_TempControl(kNode node);

/** 
 * Gets the network configuration module.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Net module.
 */
kFsFx(kNet) kNode_Net(kNode node); 

/** 
 * Gets the pipe configuration module.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Pipe module.
 */
kFsFx(kPipe) kNode_Pipe(kNode node); 

/** 
 * Gets the actions configuration module.
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Actions module.
 */
kFsFx(kActions) kNode_Actions(kNode node); 

/** 
 * Gets the storage module.
 * 
 * The storage handle returned by this method is guaranteed to be valid. However, storage 
 * functionality is only available when the node object is connected (e.g., kNode_Connect).  
 *
 * @public              @memberof kNode
 * @param   node        kNode object.
 * @return              Storage module.
 */
kFsFx(kStorage) kNode_Storage(kNode node); 

/** 
* Sets the given LED to the desired operational mode.
* 
* LED operational mode is write-only.
* 
* @public               @memberof kNode
* @param   node         Node object. 
* @param   instance     LED instance.
* @param   mode         LED operational mode.
* @return               Operation status. 
*/
kFsFx(kStatus) kNode_SetLedMode(kNode node, kLed instance, kLedMode mode);

/** 
 * Increments the node software input counter.
 * 
 * Cameras have the ability to track and report input events that have occurred since the previous 
 * camera frame. The type of event to be tracked is specified individually for each camera via 
 * the kCamera_SetInputCounterSource method. Cameras with an input counter source of type 
 * kCAMERA_INPUT_COUNTER_SOURCE_SOFTWARE will track changes to the node's software input counter. 
 * The software input counter can be incremented by calling this method. 
 * 
 * Input counter changes are reported in camera message stamps. The InputCounter bits within the 
 * kStamp.status field reflect the number of events recorded since the previous camera frame (or since 
 * acquisition start, for the first frame). The reported event count is limited by the number of available 
 * stamp status bits (as of this writing, 2). If the number of events exceeds the available bits, the 
 * reported count will saturate.  
 * 
 * Software input counters are sampled when camera messages are processed by camera drivers 
 * rather than at frame acquisition time. As such, software input counter sample timing is somewhat 
 * less deterministic than hardware input counter sample timing.
 * 
 * @public              @memberof kNode
 * @param   node        Node object. 
 * @return              Operation status. 
 * @see                 kCamera_SetInputCounterSource, kCameraInputCounterSource
 */
kFsFx(kStatus) kNode_IncrementInputCounter(kNode node);

#include <kFireSync/Client/kNode.x.h>

#endif
