/** 
 * @file    kNet.h
 * @brief   Declares the kNet class. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NET_H
#define K_FIRESYNC_NET_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kNet
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents configuration for various node network services (e.g. control, data). 
 */
//typedef kObject kNet;        --forward-declared in kFsDef.x.h

/** 
 * Resets net mode settings to defaults.
 *
 * @public          @memberof kNet
 * @param   net     Net object.  
 * @return          Operation status. 
 */
kFsFx(kStatus) kNet_Clear(kNet net);

/** 
 * Gets the number of available network interfaces. 
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @return              Count of network interfaces. 
 */
kFsFx(kSize) kNet_InterfaceCount(kNet net);

/** 
 * Gets the network interface at the specified index.
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @param   index       Index of the network interface.  
 * @return              Network interface at the specified index.
 */
kFsFx(kNetInterface) kNet_InterfaceAt(kNet net, kSize index);

/** 
 * Sets the source of synchronization signals for the node. 
 *
 * A sensor that receives synchronization from a master (kSYNC_SOURCE_MASTER) will be globally synchronized - 
 * module timing on the sensor will be synchronized to the module timing of other sensors that are connected to 
 * the same master. A sensor that generates synchronization internally (kSYNC_SOURCE_SENSOR) will be locally 
 * synchronized - module timing on the sensor will be internally synchronized, but will not be synchronized
 * to the module timing of other sensors on the network. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @param   source      Spcecifies the synchronization source.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_SetSyncSource(kNet net, kSyncSource source);

/** 
 * Gets the synchronization source for the node. 
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @return              Node synchronization source. 
 */
kFsFx(kSyncSource) kNet_SyncSource(kNet net);

/** 
 * Reports constraint and validity information for the SyncSource setting. 
 *
 * @public              @memberof kNet
 * @param   net         Net object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_SyncSourceInfo(kNet net, kInfoBits* info);

/** 
 * Resets net mode and device settings to defaults.
 *
 * @public                  @memberof kNet
 * @param   net             Net object.  
 * @param   clearIpConfig   Should the node's IP configuration be reset?
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNet_ClearAll(kNet net, kBool clearIpConfig);

/** 
 * Sets the number of network data output ports exposed by this node. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   portCount   Count of output ports. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_SetDataOutPortCount(kNet net, kSize portCount);

/** 
 * Reports the number of network data output ports exposed by this node. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @return              Count of output ports.
 */
kFsFx(kSize) kNet_DataOutPortCount(kNet net);

/** 
 * Sets the maximum total queue size (in bytes) associated with a single network data output port. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   size        Maximum output queue size (bytes).
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_SetDataOutMaxQueueSize(kNet net, kSize size);

/** 
 * Reports the maximum total queue size (in bytes) associated with a single network data output port. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @return              Maximum output queue capacity (bytes). 
 */
kFsFx(kSize) kNet_DataOutMaxQueueSize(kNet net);

/** 
 * Sets the maximum queue item count associated with a single network data output port. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   count       Maximum queue item count.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_SetDataOutMaxQueueCount(kNet net, kSize count);

/** 
 * Reports the maximum queue item count associated with a single network data output port. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @return              Maximum queue item count. 
 */
kFsFx(kSize) kNet_DataOutMaxQueueCount(kNet net);

/** 
 * Sets the thread priority class for any network data output threads.
 *
 * Indiscriminate thread priority adjustments can result in deadlock/starvation under 
 * high processing load. Accordingly, thread priority adjustment should usually be avoided, 
 * except where indicated by the FireSync Software Team to support special cases. 
 *
 * Refer to kThread_SetPriority for more information on thread priorities. 
 * 
 * @public                  @memberof kNet
 * @param   net             Net object.  
 * @param   priorityClass   Thread priority class.  
 * @return                  Operation status.
 * @see                     kThread_SetPriority
 */
kFsFx(kStatus) kNet_SetDataOutThreadPriorityClass(kNet net, kThreadPriorityClass priorityClass);

/** 
 * Gets the thread priority class for any network data output threads.
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @return              Thread priority class.
 */
kFsFx(kThreadPriorityClass) kNet_DataOutThreadPriorityClass(kNet net);

/**
* Reports constraint and validity information for the DataOutThreadPriorityClass setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kNet_DataOutThreadPriorityClassInfo(kNet net, kInfo32s* info);

/** 
 * Sets the thread priority offset for any network data output threads.
 *
 * @public                  @memberof kNet
 * @param   net             Net object.  
 * @param   priorityOffset  Thread priority offset.
 * @return                  Operation status.
 * @see                     kThread_SetPriority
 */
kFsFx(k32s) kNet_SetDataOutThreadPriorityOffset(kNet net, k32s priorityOffset);

/** 
 * Gets the thread priority offset for any network data output threads.
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @return              Thread priority offset.
 */
kFsFx(k32s) kNet_DataOutThreadPriorityOffset(kNet net);

/**
* Reports constraint and validity information for the DataOutThreadPriorityOffset setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kNet_DataOutThreadPriorityOffsetInfo(kNet net, kInfo32s* info);

/** 
 * Sets the thread affinity for any network data output threads.
 *
 * Thread affinity adjustments can result in suboptimal performance for some workloads.
 * Accordingly, thread affinity adjustment should usually be avoided, except where indicated 
 * by the FireSync Software Team to support special cases. 
 *
 * Refer to kThread_SetAffinity for more information on thread priorities.  
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @param   affinity    Thread affinity.  
 * @return              Operation status.
 * @see                 kThread_SetAffinity
 */
kFsFx(kStatus) kNet_SetDataOutThreadAffinity(kNet net, kBitArray affinity);

/** 
 * Gets the thread affinity for any network data output threads.
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @param   affinity    Thread affinity.  
 * @return              Operation status.
 */
kFsFx(kStatus) kNet_DataOutThreadAffinity(kNet net, kBitArray affinity);

/**
* Reports constraint and validity information for the kNet_DataOutThreadAffinity setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kNet_DataOutThreadAffinityInfo(kNet net, kInfo* info);

/**
* Sets the socket buffer size associated with each data output port.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   size        Socket buffer size, in bytes. 
* @return              Operation status.
*/
kFsFx(kStatus) kNet_SetDataOutSocketBufferSize(kNet net, kSize size);

/**
* Reports the socket buffer size associated with each data output port. 
*
* @public              @memberof kNet
* @param   net         Net object.
* @return              Socket buffer size, in bytes. 
*/
kFsFx(kSize) kNet_DataOutSocketBufferSize(kNet net);

/**
* Reports constraint and validity information for the DataOutSocketBufferSize setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Info structure.
* @return              Operation status.
*/
kFsFx(kSize) kNet_DataOutSocketBufferSizeInfo(kNet net, kInfoSize* info);

/**
* Sets the socket buffer size associated with each data input port.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   size        Socket buffer size, in bytes.
* @return              Operation status.
*/
kFsFx(kStatus) kNet_SetDataInSocketBufferSize(kNet net, kSize size);

/**
* Reports the socket buffer size associated with each data input port.
*
* @public              @memberof kNet
* @param   net         Net object.
* @return              Socket buffer size, in bytes.
*/
kFsFx(kSize) kNet_DataInSocketBufferSize(kNet net);

/**
* Reports constraint and validity information for the DataInSocketBufferSize setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Info structure.
* @return              Operation status.
*/
kFsFx(kSize) kNet_DataInSocketBufferSizeInfo(kNet net, kInfoSize* info);

/** 
 * Sets the number of data input ports defined for this node. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   count       Count of data input ports. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_SetDataInPortCount(kNet net, kSize count);

/** 
 * Defines a network data input port for this node. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   sourceNode  Source node that will be connected to this port. 
 * @param   sourcePort  Source port that will be connected to this port. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_AddDataInPort(kNet net, k32u sourceNode, k32u sourcePort);

/** 
 * Removes the network data input port at the specified index. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   index       Index of input port to be removed. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNet_RemoveDataInPort(kNet net, kSize index);

/** 
 * Reports the number of network data input ports defined for this node. 
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @return              Count of input ports.
 */
kFsFx(kSize) kNet_DataInPortCount(kNet net);

/** 
 * Sets information describing the network data input port at the specified index.
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   index       Index of input port. 
 * @param   sourceNode  Source node that will be connected to this port. 
 * @param   sourcePort  Source port that will be connected to this port. 
 * @return              Structure describing the data input port.
 */
kFsFx(kStatus) kNet_SetDataInPortAt(kNet net, kSize index, k32u sourceNode, k32u sourcePort);

/** 
 * Accesses information describing the network data input port at the specified index.
 * 
 * @public              @memberof kNet
 * @param   net         Net object.   
 * @param   index       Index of input port. 
 * @return              Structure describing the data input port.
 */
kFsFx(const kDataInSource*) kNet_DataInPortAt(kNet net, kSize index);

/** 
 * Sets the thread priority class for any network data input threads.
 *
 * Indiscriminate thread priority adjustments can result in deadlock/starvation under 
 * high processing load. Accordingly, thread priority adjustment should usually be avoided, 
 * except where indicated by the FireSync Software Team to support special cases. 
 *
 * Refer to kThread_SetPriority for more information on thread priorities. 
 * 
 * @public                  @memberof kNet
 * @param   net             Net object.  
 * @param   priorityClass   Thread priority class.  
 * @return                  Operation status.
 * @see                     kThread_SetPriority
 */
kFsFx(kStatus) kNet_SetDataInThreadPriorityClass(kNet net, kThreadPriorityClass priorityClass);

/** 
 * Gets the thread priority class for any network data input threads.
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @return              Thread priority class.
 */
kFsFx(kThreadPriorityClass) kNet_DataInThreadPriorityClass(kNet net);

/**
* Reports constraint and validity information for the DataInThreadPriorityClass setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kNet_DataInThreadPriorityClassInfo(kNet net, kInfo32s* info);

/** 
 * Sets the thread priority offset for any network data input threads.
 *
 * @public                  @memberof kNet
 * @param   net             Net object.  
 * @param   priorityOffset  Thread priority offset.
 * @return                  Operation status.
 * @see                     kThread_SetPriority
 */
kFsFx(k32s) kNet_SetDataInThreadPriorityOffset(kNet net, k32s priorityOffset);

/** 
 * Gets the thread priority offset for any network data input threads.
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @return              Thread priority offset.
 */
kFsFx(k32s) kNet_DataInThreadPriorityOffset(kNet net);

/**
* Reports constraint and validity information for the DataInThreadPriorityOffset setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kNet_DataInThreadPriorityOffsetInfo(kNet net, kInfo32s* info);

/** 
 * Sets the thread affinity for any network data input threads.
 *
 * Thread affinity adjustments can result in suboptimal performance for some workloads.
 * Accordingly, thread affinity adjustment should usually be avoided, except where indicated 
 * by the FireSync Software Team to support special cases. 
 *
 * Refer to kThread_SetAffinity for more information on thread priorities.  
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @param   affinity    Thread affinity.  
 * @return              Operation status.
 * @see                 kThread_SetAffinity
 */
kFsFx(kStatus) kNet_SetDataInThreadAffinity(kNet net, kBitArray affinity);

/** 
 * Gets the thread affinity for any network data input threads.
 *
 * @public              @memberof kNet
 * @param   net         Net object.  
 * @param   affinity    Thread affinity.  
 * @return              Operation status.
 */
kFsFx(kStatus) kNet_DataInThreadAffinity(kNet net, kBitArray affinity);

/**
* Reports constraint and validity information for the kNet_DataInThreadAffinity setting.
*
* @public              @memberof kNet
* @param   net         Net object.
* @param   info        Receives the info structure.
* @return              Operation status. 
*/
kFsFx(kStatus) kNet_DataInThreadAffinityInfo(kNet net, kInfo* info);

#include <kFireSync/Client/kNet.x.h>

#endif
