/** 
 * @file    kLightState.h
 * @brief   Declares the kLightState type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_LIGHT_STATE_H
#define K_FIRESYNC_LIGHT_STATE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kLightState
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents a light state configuration.
 */
//typedef kObject kLightState;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the state. 
 *
 * The id of a state is the same as the index of the state in the light's state list.
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @return              Returns the unique id of this state. 
 */
kFsFx(kSize) kLightState_Id(kLightState state);

/** 
 * Sets the delay for the light state, in nanoseconds.  
 *
 * Delay values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @param   delay       Delay value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightState_SetDelayNs(kLightState state, k64u delay, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the delay for the light state, in nanoseconds.  
 *
 * Delay values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @return              Returns the delay for the light state, in nanoseconds.  
 */
kFsFx(k64u) kLightState_DelayNs(kLightState state);

/** 
 * Reports constraint and validity information for the DelayNs setting, in nanoseconds.
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightState_DelayNsInfo(kLightState state, kInfo64u* info);

/** 
 * Sets the exposure for the light state, in nanoseconds.
 *
 * Exposure values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @param   exposure    Exposure value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightState_SetExposureNs(kLightState state, k64u exposure, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the exposure for the light state, in nanoseconds.  
 *
 * Exposure values are specified in FireSync nanoseconds (~ 1 us).
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @return              Returns the exposure for the light state, in nanoseconds.  
 */
kFsFx(k64u) kLightState_ExposureNs(kLightState state);

/** 
 * Reports constraint and validity information for the ExposureNs setting. 
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightState_ExposureNsInfo(kLightState state, kInfo64u* info);

/** 
 * Sets the gap for the light state, in nanoseconds.
 *
 * Gap values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @param   gap         Gap value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightState_SetGapNs(kLightState state, k64u gap, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the gap for the light state, in nanoseconds.  
 *
 * Gap values are specified in FireSync nanoseconds (~ 1 us).
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @return              Returns the gap for the light state, in nanoseconds.  
 */
kFsFx(k64u) kLightState_GapNs(kLightState state);

/** 
 * Reports constraint and validity information for the GapNs setting. 
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightState_GapNsInfo(kLightState state, kInfo64u* info);

/** 
 * Reports the total time required to execute this light state, in nanoseconds. 
 *
 * Duration values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kLightState
 * @param   state       State object.  
 * @return              State duration, in nanoseconds. 
 */
kFsFx(k64u) kLightState_DurationNs(kLightState state);

/**
 * Reports the lead-in time for the light state.
 *
 * Lead-in is the time offset between the assertion of the physical light exposure signal 
 * and the start of nominal light exposure (defined by DEG parameters).
 * 
 * @public              @memberof kLightState
 * @param   state       State object.
 * @return              Lead-in time, in FireSync nanoseconds.
 */
kFsFx(k64u) kLightState_LeadInNs(kLightState state);

/**
 * Reports the total lead-in time for the light state.
 *
 * Total lead-in includes the local light state lead-in as well as the lead-ins of 
 * any module to which the light is coupled (e.g., camera).
 * 
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Total lead-in time, in FireSync nanoseconds.
 */
kFsFx(k64u) kLightState_TotalLeadInNs(kLightState state);

/**
 * Reports the lead-out time for the light state.
 *
 * Lead-out is the time offset between the deassertion of the physical light exposure signal 
 * and the end of nominal light exposure (defined by DEG parameters).
 * 
 * @public              @memberof kLightState
 * @param   state       State object.
 * @return              Lead-out time, in FireSync nanoseconds.
 */
kFsFx(k64u) kLightState_LeadOutNs(kLightState state);

/**
 * Reports the total lead-out time for the light state.
 *
 * Total lead-out includes the local light state lead-out as well as the lead-outs of 
 * any module to which the light is coupled (e.g., camera).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Total lead-out time, in FireSync nanoseconds.
 */
kFsFx(k64u) kLightState_TotalLeadOutNs(kLightState state);

/**
 * Estimates the time for which the light may be emissive prior to nominal exposure.
 * 
 * This value should be used when determining multiplexed camera/light time slots. This value will 
 * usually be zero unless light priming is set to Always. 
 * 
 * @public              @memberof kLightState
 * @param   state       State object.
 * @return              Pre-exposure time, in FireSync nanoseconds.
 */
kFsFx(k64u) kLightState_PreExposureNs(kLightState state);

/**
 * Estimates the time for which the light may be emissive after nominal exposure.
 * 
 * This value should be used when determining multiplexed camera/light time slots. In practice, 
 * this value will typically be zero. 
 *
 * @public              @memberof kLightState
 * @param   state       State object.
 * @return              Post-exposure time, in FireSync nanoseconds.
 */
kFsFx(k64u) kLightState_PostExposureNs(kLightState state);

#include <kFireSync/Client/kLightState.x.h>

#endif
