/** 
 * @file    kCamera.x.h
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_X_H
#define K_FIRESYNC_CAMERA_X_H

#define kCAMERA_DEFAULT_VIDEO_BUFFER_SIZE           (1 << 24)
#define kCAMERA_DEFAULT_ALGORITHM_BUFFER_SIZE       (1 << 22)
#define kCAMERA_DEFAULT_FRAME_SIZE_GRANULARITY      (1 << 7)
#define kCAMERA_FRAME_ALIGN_SIZE                    (4096)
#define kCAMERA_BUFFER_ALIGN_SIZE                   (1048576)
#define kCAMERA_BUFFER_PADDING_SIZE                 (1048576)
#define kCAMERA_PHASE_DECODER_BUFFER_SIZE_FACTOR    (2)

 /* PL EPS constants, see https://docs.google.com/spreadsheets/d/1KXKxMhDP6wkCGR8gGFJC6lVcWwNM9uGW_UKGmWj3rp0/edit#gid=1078318383 */
#define kCAMERA_EPS_CAM_DMA_UNI_BW                      (3420)
#define kCAMERA_EPS_MAX_BW_PER_DIR                      (4053.4)
#define kCAMERA_EPS_DDR_MAX_BW_WRITE                    (3411.2)
#define kCAMERA_EPS_DDR_MAX_BW_READ                     (3624.4)
#define kCAMERA_EPS_MAX_PL_PS_WRITE_BW                  (650.) 
#define kCAMERA_EPS_ROW_MULTIPLIER_COUNT                (2)
#define kCAMERA_EPS_FILTER_DIV_COUNT                    (2)
#define kCAMERA_EPS_PL_DMA_WIDTH_BITS                   (128)
#define kCAMERA_EPS_PHASE_COMPRESSION_RATIO             (135. / 100.)
#define kCAMERA_EPS_INT_PHASE_COMPRESSION_RATIO         (200. / 100.)
#define kCAMERA_EPS_COMPRESSION_FIFO_SIZE               (8.)
#define kCAMERA_EPS_DDR_INTERFACE_EFFICIENCY            (95. / 100.)

typedef struct kCameraClass
{
    kObjectClass base; 

    kNode node;
    kPipe pipe;
    kBlock block;

    kCameraModel model; 
    kCameraCapability capabilities; 
    kCameraInfo modelInfo;
    k64f pixelClockFrequency;                       //PL Pixel clock frequency in real Hz.
    k32u plPipeCount;                               //Number of PL camera pipes.
    k32u imagerChannelCount;                        //Number of imager channels.

    kCameraInputCounterSource inputCounterSource;
    k32u inputCounterSourceId;

    kSize videoBufferSize;
    kSize nextVideoBufferSize;
    kSize algorithmBufferSize;
    kSize nextAlgorithmBufferSize;

    kSize index; 
    kCameraAlgorithm algorithm;
    kBool enabled;     
    kBool isConnected; 
    
    kCameraControl control;    
    k32u controlId; 

    kArrayList states;                              //kArrayList<kCameraState>
    kArrayList extensions;                          //kArrayList<kCameraExt>

    kCameraReadout readoutMode;

    kCameraOutputMode outputCapabilities;
    kCameraOutputMode outputMode;
    kCameraPortMode portMode;

    kBool ramImageEnabled; 
    kBool prnuEnabled; 
    kBool fpnEnabled; 
    kBool fpnCalibrationEnabled;
    kBool calibrationEnabled;
    kBool testPatternEnabled;
    kCameraAccelerationTestMode accelerationTestMode;

    kBool isTrailingRequired;
    kSize frameSizeGranularity;

    k32u reductionFactor;
    k32u reductionWindow;

    kBool leadOutEnabled;                           //Should lead-out be included exposure?

} kCameraClass; 

kDeclareClassEx(kFs, kCamera, kObject)
        
kFsFx(kStatus) kCamera_Construct(kCamera* camera, kNode node, kSize index, kAlloc allocator); 

kFsFx(kStatus) kCamera_Init(kCamera camera, kType type, kNode node, kSize index, kAlloc alloc); 
kFsFx(kStatus) kCamera_VRelease(kCamera camera); 

kFsFx(kStatus) kCamera_ParseDevice(kCamera camera, kXml xml, kXmlItem item); 
kFsFx(kStatus) kCamera_FormatDevice(kCamera camera, kXml xml, kXmlItem item); 
kFsFx(kStatus) kCamera_FormatVirtual(kCamera camera, kXml xml, kXmlItem item);

kFsFx(kStatus) kCamera_Parse(kCamera camera, kXml xml, kXmlItem item); 
kFsFx(kStatus) kCamera_ParseStates(kCamera camera, kXml xml, kXmlItem item); 
kFsFx(kStatus) kCamera_ParseExtensions(kCamera camera, kXml xml, kXmlItem item); 

kFsFx(kStatus) kCamera_Format(kCamera camera, kXml xml, kXmlItem item); 
kFsFx(kStatus) kCamera_FormatStates(kCamera camera, kXml xml, kXmlItem item); 
kFsFx(kStatus) kCamera_FormatExtensions(kCamera camera, kXml xml, kXmlItem item); 

kFsFx(kStatus) kCamera_ClearEx(kCamera camera, kNodeClearOption options, kSize stateCount);

kFsFx(kStatus) kCamera_VerificationLabel(kCamera camera, kChar* label, kSize capacity); 
kFsFx(kStatus) kCamera_Verify(kCamera camera); 

kFsFx(kBool) kCamera_IsExtensionEnabled(kCamera camera, kCameraExtension id);

kFsFx(kStatus) kCamera_RegisterBlock(kCamera camera);
kFsFx(kStatus) kCamera_UnregisterBlock(kCamera camera);
kFsFx(kStatus) kCamera_UpdateBlock(kCamera camera);

kFsFx(kStatus) kCamera_FindSequenceExtension(kCamera camera, kCameraExt* extension); 
kFsFx(kBool) kCamera_InterruptsEnabled(kCamera camera); 

kFsFx(kCameraInfo) kCamera_ModelInfo(kCamera camera); 
kFsFx(kCameraCapability) kCamera_Capabilities(kCamera camera);

kFsFx(kCameraOutputMode) kCamera_OutputCapabilities(kCamera camera);

kFsFx(kStatus) kCamera_EnableCalibration(kCamera camera, kBool enabled);
kFsFx(kBool) kCamera_CalibrationEnabled(kCamera camera);

//real frequency (Hz)
kFsFx(k64f) kCamera_PixelClockFrequency(kCamera camera); 

//pixel clock period in real ns.
kFsFx(k64f) kCamera_PixelPeriod(kCamera camera); 

kFsFx(k32u) kCamera_ChannelCount(kCamera camera);
kFsFx(k32u) kCamera_PlPipeCount(kCamera camera);

kFsFx(k64u) kCamera_MinimumDuration(kCamera camera);
kFsFx(k64u) kCamera_MinimumEventPeriod(kCamera camera, kSize eventId);
kFsFx(k64u) kCamera_MinPatternDuration(kCamera camera);
kFsFx(k32u) kCamera_MaximumColumns(kCamera camera);

kFsFx(kNode) kCamera_Node(kCamera camera); 

kFsFx(k64u) kCamera_Throughput(kCamera camera);
kFsFx(k64u) kCamera_AlgorithmThroughput(kCamera camera);
kFsFx(k64u) kCamera_VideoThroughput(kCamera camera);

kFsFx(kSize) kCamera_NormalizeBufferSize(kCamera camera, kSize frameSize, kSize frameCount);

kFsFx(kBool) kCamera_PlSequencingEnabled(kCamera camera);

kInlineFx(kBool) kCamera_VideoEnabled(kCamera camera)
{
    kCameraOutputMode mode = kCamera_OutputMode(camera); 

    return (mode == kCAMERA_OUTPUT_MODE_VIDEO) || (mode == kCAMERA_OUTPUT_MODE_VIDEO_AND_ALGORITHM);
}

kInlineFx(kBool) kCamera_AlgorithmEnabled(kCamera camera)
{
    kCameraOutputMode mode = kCamera_OutputMode(camera); 

    return (mode == kCAMERA_OUTPUT_MODE_ALGORITHM) || (mode == kCAMERA_OUTPUT_MODE_VIDEO_AND_ALGORITHM);
}

kFsFx(kStatus) xkCamera_PlAccelerationTime(kCamera camera, kSize cameraStateId, k64u cameraPeriod, k64u* processingTime);

kFsFx(kSize) xkCamera_EnabledCameraCount(kCamera camera);

kFsFx(kStatus) xkCamera_EpsProcessingTime(kCamera camera, kSize cameraStateId, k64u cameraPeriod, k64u* processingTime);
kFsFx(k64f) xkCamera_EpsMaxDmaChannelReadBandwidth(kCamera camera);
kFsFx(k64f) xkCamera_EpsMaxDmaChannelWriteBandwidth(kCamera camera);

#endif
