/** 
 * @file    kAnalogOut.h
 * @brief   Declares the kAnalogOut type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ANALOG_OUT_H
#define K_FIRESYNC_ANALOG_OUT_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kAnalogOut
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents analog configuration.
 */
//typedef kObject kAnalogOut;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the output. 
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @return              Returns the unique id of this output. 
 */
kFsFx(kSize) kAnalogOut_Id(kAnalogOut analog);

/** 
 * Re-initializes settings.
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOut_Clear(kAnalogOut analog);

/** 
 * Enables or disables the analog output. 
 *
 * Disabled analog outputs are not checked during verification, and do not respond to 
 * Start/Pause/Resume/Stop commands. 
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @param   enable      Specifies whether to enable or disable the analog output.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOut_Enable(kAnalogOut analog, kBool enable);

/** 
 * Reports whether the analog output is currently enabled. 
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @return              Returns whether the analog output is currently enabled. 
 */
kFsFx(kBool) kAnalogOut_IsEnabled(kAnalogOut analog);

/** 
 * Sets the control mode of the analog output.
 *
 * For control modes that suppport multiple sources (e.g. kANALOG_OUT_CONTROL_TRIGGERED_BY_EVENT), 
 * use the kAnalogOut_SetControlId function to specify a source.
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @param   type        Analog output control mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOut_SetControl(kAnalogOut analog, kAnalogOutControl type);

/** 
 * Reports the control mode of the analog output. 
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @return              Returns the control mode of the analog output. 
 */
kFsFx(kAnalogOutControl) kAnalogOut_Control(kAnalogOut analog);

/** 
 * Reports constraint and validity information for the Control setting. 
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOut_ControlInfo(kAnalogOut analog, kInfoBits* info);

/** 
 * Sets the control source id for an analog output. 
 *
 * For control modes that suppport multiple sources (e.g. kANALOG_OUT_CONTROL_TRIGGERED_BY_EVENT), 
 * this function cam be used to specify a source id.
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @param   id          Specifies the control source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOut_SetControlId(kAnalogOut analog, k32u id);

/** 
 * Gets the control source id for an analog output.
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @return              Control source id.
 */
kFsFx(k32u) kAnalogOut_ControlId(kAnalogOut analog);

/** 
 * Reports constraint and validity information for the ControlId setting. 
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOut_ControlIdInfo(kAnalogOut analog, kInfo* info);

/**
* Sets (or clears) calibration information.
*
* The calibration table provides a mapping between nominal current and calibrated current.
* The table should have the following properties:
* * kArray2<k64f>
* * First column represents nominal current.
* * Second column represents calibrated current.
* * At least two rows with unique nominal/calibrated current values.
*
* Calibration is optional and can be cleared by calling this function with a null calibration object.
* If calibration is not provided, output will governed by the nominal device calibration model
* (e.g., 0 - 22 mA, mapped directly to DAC codes).
*
* Calibration is transient (will not be saved to storage). Reapply calibration as needed
* after reset.
* 
* This method will automatically apply any outstanding configuration changes before sending 
* calibration data to the node.  
*
* @public              @memberof kAnalogOut
* @param   analog      Analog output object.
* @param   calibration Calibration table (kArray2<k64f>), or null to clear calibration.
* @return              Operation status.
*/
kFsFx(kStatus) kAnalogOut_SetCalibration(kAnalogOut analog, kArray2 calibration);

/**
* Reports constraints for output current.
*
* Reports constraints for nominal (uncalibrated) currrent output when uncalibrated. Reports constraints
* for calibrated current output when calibrated. 
*
* @public              @memberof kAnalogOut
* @param   analog      Analog output object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kAnalogOut_OutputInfo(kAnalogOut analog, kInfo64f* info);

/**
* Reports constraints for nominal output current.
*
* Nominal current refers to uncalibrated current output. The purpose of this method
* is provide information about nominal hardware capabilities, even when calibrated.
*
* @public              @memberof kAnalogOut
* @param   analog      Analog output object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kAnalogOut_NominalOutputInfo(kAnalogOut analog, kInfo64f* info);

/**
* Enqueues an analog output value, in milliamps.
*
* If the analog output is paired with an event manager, the output current will be enqueued for
* later release. If the analog output is under software control, the output current will become
* active immediately.
*
* @public              @memberof kAnalogOut
* @param   analog      Analog output object.
* @param   current     Output current, in milliamps.
* @return              Operation status.
*/
kFsFx(kStatus) kAnalogOut_EnqueueOutput(kAnalogOut analog, k64f current);

/** 
 * Retrieve statistics for this analog output module.
 *
 * @public              @memberof kAnalogOut
 * @param   analog      Analog output object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAnalogOut_Stats(kAnalogOut analog, kAnalogOutStats* stats);

#include <kFireSync/Client/kAnalogOut.x.h>

#endif
