/** 
 * @file    kCameraInfo.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_INFO_X_H
#define K_FIRESYNC_CAMERA_INFO_X_H

#define kCAMERA_INFO_SUB_SKIP_OPTIONS               (0x7)                       //Supported in PL; same for all cameras.

#define kCAMERA_INFO_DEFAULT_EXPOSURE               (kDEFAULT_EXPOSURE_NS)      //Default exposure setting. 
#define kCAMERA_INFO_DEFAULT_GAP                    (1024000)                   //Default gap setting. 
#define kCAMERA_INFO_MIN_NORMALIZED_INTER_KNEE_GAP  (0.000001)                  //Minimum inter knee gap.

#define kCAMERA_INFO_MIN_TRAILING                   (8)                         //Minimum number of trailing bytes.   
#define kCAMERA_INFO_FRAME_DATA_SIZE_GRANULARITY    (8)                         //Granularity for frame data size.

#define kCAMERA_INFO_COLOR_GAIN_GRANULARITY         (0.0078125)                 ///< Color gain step granularity.
#define kCAMERA_INFO_COLOR_GAIN_MAX_STEP            (896)                       ///< Maximum color gain step.
#define kCAMERA_INFO_COLOR_GAIN_1X_FACTOR           (128)                       ///< The value of color gain 1.0x.

#define kCAMERA_INFO_POST_PROCESSING_PADDING        (1000)                      //Post processing padding.
#define kCAMERA_INFO_PL_INTERNAL_PROCESSING         (275)                       //Pixel clocks to take into account PL internal processing. (75 for inband, 200 for internal delays)
#define kCAMERA_INFO_PL_INBAND_POST_PROCESSING      (150)                       //Pixel clocks to take into account inband insertion and post processing.
#define kCAMERA_INFO_X_RESAMPLING_INTERPOLATION_NUM (2.0)                       //Minimum number of interpolation per pair point.

//Shared between LC200x and LC500x
#define kLC2K5K_BLACK_ROW_COUNT                     (4)                         // Number of black rows.
#define kLC2K5K_BLACK_ROWS_GATED                    (2)                         // Number of black rows gated.
#define kLC2K5K_DUMMY_ROW_COUNT                     (1)                         // Number of dummy rows.
#define kLC2K5K_DUMMY_ROWS_GATED                    (0)                         // Number of dummy rows gated.

kFsFx(k64u) kCameraInfo_ConvertToFsClockCeil(k64f realTimeNs);
kFsFx(k64u) kCameraInfo_ConvertToFsClockFloor(k64f realTimeNs);

typedef struct kCameraInfoClass
{
    kObjectClass base; 
    kCameraModel model;           
    kCameraCapability capabilities; 
    k32u minWidth;                                          // Imager minimum width (pixels).
    k32u maxWidth;                                          // Imager maximum width (pixels).
    k32u widthGranularity;                                  // Imager width granularity based on kernel size (pixels).
    k32u minHeight;                                         // Imager minimum height (pixels).
    k32u maxHeight;                                         // Imager maximum height (pixels).
    k32u heightGranularity;                                 // Imager height granularity (pixels).
    k32u leftGranularity;                                   // Window x granularity (pixels). 
    k32u topGranularity;                                    // Window y granularity (pixels). 
    kCameraSubsampling xSubModes;                           // Supported x-subsampling modes (imager and pl). 
    k32u xSubOptions;                                       // Supported x-subsampling options (image and pl).
    kCameraSubsampling ySubModes;                           // Supported y-subsampling modes (imager and pl). 
    k32u ySubOptions;                                       // Supported y-subsampling options (image and pl).
    kBool subModeConstrainedEqual;                          // Are x and y subsampling modes constrained to be equal?
    kBool imagerSubConstrainedEqual;                        // Are x and y imager subsampling amounts constrained to be equal?
    k32u maxImagerXSub;                                     // Maximum x-subsampling amount supported by imager. 
    k32u maxImagerYSub;                                     // Maximum y-subsampling amount supported by imager. 
    kBool requiresPixelMap;                                 // Is PL pixel re-ordering required for this imager?
    kBool canInvertX;                                       // Is x-mirroring supported by this imager?
    kBool canInvertY;                                       // Is y-mirroring supported by this imager?
    k32s minBlackLevel;                                     // Minimum supported black level value. 
    k32s maxBlackLevel;                                     // Maximum supported black level value. 
    kSize kneeCountMin;                                     // Multi-Slope minimum knee count
    kSize kneeCountMax;                                     // Multi-Slope max knee count
    kSize kneeCountGranularity;                             // Multi-Slope knee count granularity
    kCameraExposure exposureModes;                          // Supported exposure modes.
    kRangeLutCaps lutCaps;
    kBool isTrailingRequired;                               // Is trailing required padded to end of frame?
    kSize frameSizeGranularity;                             // Frame size granularity shift bits.
    kBool hasTempSensor;                                    // Does this imager has temperature sensor on board?
    kBool isColor;                                          // Is this a color imager?
    kCfa cfa;                                               // Color filter array.
    kPixelFormat pixelFormat;                               // Pixel format.
    kCameraBufferSpace videoBufferSpace;                    // Camera video stream buffer space.
    kBool requiresCalibration;                              // Is camera calibration required for this imager?
    kBool hasTestPattern;                                   // Does camera support test pattern?
    k32u minLvdsPower;                                      // Minimum LVDS power setting. k32U_NULL if not supported.
    k32u maxLvdsPower;                                      // Maximum LVDS power setting. k32U_NULL if not supported.
    k32u releaseLvdsPower;                                  // LVDS power setting for release sensor. k32U_NULL if not supported.
    k32u factoryLvdsPower;                                  // LVDS power setting for factory sensor. k32U_NULL if not supported.
    kVersion plVersion;                                     // PL version. -- TODO: Remove once PLs are 6.2.7.2 or higher.
    k64f minIsrGap;                                         // Minimum ISR gap (ns); TODO: refine the model. 
} kCameraInfoClass; 

typedef struct kCameraInfoVTable
{
    kObjectVTable base; 

    kBool (kCall* VHasGainChannel)(kCameraInfo info, kCameraGain channel);
    k32u (kCall* VMinGainStep)(kCameraInfo info, kCameraGain channel);
    k32u (kCall* VMaxGainStep)(kCameraInfo info, kCameraGain channel);
    k32u (kCall* VGainStep)(kCameraInfo info, kCameraGain channel, k64f gain);
    k64f (kCall* VGainFactor)(kCameraInfo info, kCameraGain channel, k32u index);
    k64f (kCall* VDefaultGainFactor)(kCameraInfo info, kCameraGain channel);

    k64f (kCall* VReadoutTime)(kCameraInfo info, const kCameraInfoParams* params); 
    k64f (kCall* VMinInterExposureGap)(kCameraInfo info, const kCameraInfoParams* params); 
    k64f (kCall* VMinInterReadoutGap)(kCameraInfo info, const kCameraInfoParams* params);
    k64f (kCall* VMinExposure)(kCameraInfo info, const kCameraInfoParams* params); 
    k64f (kCall* VMaxExposure)(kCameraInfo info, const kCameraInfoParams* params);
    k64f (kCall* VPreExposure)(kCameraInfo info, const kCameraInfoParams* params); 
    k64f (kCall* VPostExposure)(kCameraInfo info, const kCameraInfoParams* params);
    k64f (kCall* VPlInternalProcessing)(kCameraInfo info, const kCameraInfoParams* params);
    k64f (kCall* VSortProcessing)(kCameraInfo info, const kCameraInfoParams* params);
    k64f (kCall* VPostProcessing)(kCameraInfo info, const kCameraInfoParams* params);
    k64f (kCall* VRowReadoutEfficiency)(kCameraInfo info, const kCameraInfoParams* params);

    kBool (kCall* VIsWidthGrouped)(kCameraInfo info);
    kBool (kCall* VIsHeightGrouped)(kCameraInfo info);
    kBool (kCall* VIsTopGrouped)(kCameraInfo info);
    kBool (kCall* VIsLeftGrouped)(kCameraInfo info);

    kCameraExtension (kCall* VExtensionTypes)(kCameraInfo info);

    kStatus (kCall* VPrimePeriod)(kCameraInfo info, k64u* delay, k64u* exposure, k64u* gap);

} kCameraInfoVTable; 

kDeclareVirtualClassEx(kFs, kCameraInfo, kObject)
        
kFsFx(kStatus) kCameraInfo_Init(kCameraInfo info, kType type, kCameraModel model, kCameraCapability capabilities, kBool isTrailingRequired, kSize frameSizeGranularity, kVersion plVersion, kAlloc alloc);
kFsFx(kStatus) kCameraInfo_VRelease(kCameraInfo info);

kFsFx(kBool) kCameraInfo_VHasGainChannel(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kCameraInfo_VMinGainStep(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kCameraInfo_VMaxGainStep(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kCameraInfo_VGainStep(kCameraInfo info, kCameraGain channel, k64f gain);
kFsFx(k64f) kCameraInfo_VGainFactor(kCameraInfo info, kCameraGain channel, k32u index);
kFsFx(k64f) kCameraInfo_VDefaultGainFactor(kCameraInfo info, kCameraGain channel);

kFsFx(k64f) kCameraInfo_MinPlGammaExposure(kCameraInfo info, const kPoint64f* kneePoints, kSize kneeCount);
kFsFx(k32u) kCameraInfo_FindNearestGain(kCameraInfo info, k64f* gainData, kSize gainCount, k64f gain);

kFsFx(k64u) kCameraInfo_PlInternalProcessing(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64u) kCameraInfo_SortProcessing(kCameraInfo info, const kCameraInfoParams* params);

kFsFx(k64f) kCameraInfo_VReadoutTime(kCameraInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kCameraInfo_VMinInterExposureGap(kCameraInfo info, const kCameraInfoParams* params); 
kFsFx(k64f) kCameraInfo_VMinInterReadoutGap(kCameraInfo info, const kCameraInfoParams* params);  
kFsFx(k64f) kCameraInfo_VMinExposure(kCameraInfo info, const kCameraInfoParams* params);  
kFsFx(k64f) kCameraInfo_VMaxExposure(kCameraInfo info, const kCameraInfoParams* params);  
kFsFx(k64f) kCameraInfo_VPreExposure(kCameraInfo info, const kCameraInfoParams* params);  
kFsFx(k64f) kCameraInfo_VPostExposure(kCameraInfo info, const kCameraInfoParams* params);  
kFsFx(k64f) kCameraInfo_VRowReadoutEfficiency(kCameraInfo info, const kCameraInfoParams* params);

kFsFx(kBool) kCameraInfo_VIsWidthGrouped(kCameraInfo info);
kFsFx(kBool) kCameraInfo_VIsHeightGrouped(kCameraInfo info);
kFsFx(kBool) kCameraInfo_VIsTopGrouped(kCameraInfo info);
kFsFx(kBool) kCameraInfo_VIsLeftGrouped(kCameraInfo info);

kFsFx(kCameraExtension) kCameraInfo_VExtensionTypes(kCameraInfo info);

kFsFx(k32u) kCameraInfo_MaxImagerXSub(kCameraInfo info);
kFsFx(k32u) kCameraInfo_MaxImagerYSub(kCameraInfo info);
kFsFx(kBool) kCameraInfo_ImagerSubConstrainedEqual(kCameraInfo info);

kFsFx(kStatus) kCameraInfo_VPrimePeriod(kCameraInfo info, k64u* delay, k64u* exposure, k64u* gap);

#endif
